/*
  Logging.java / Frost
  Copyright (C) 2003  Jan-Thomas Czornack <jantho@users.sourceforge.net>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
package frost;

import java.beans.*;
import java.io.IOException;
import java.util.logging.*;

/**
 * 
 */
public class Logging {
	
	private static Logger logger = Logger.getLogger(Logging.class.getName());
	
	/**
	 * 
	 */
	private class ShutdownHook extends Thread {

		/**
		 * 
		 */
		public ShutdownHook() {
			super();
		}
		/* (non-Javadoc)
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			frostSettings.removePropertyChangeListener(SettingsClass.LOG_TO_FILE, listener);
			frostSettings.removePropertyChangeListener(SettingsClass.LOG_FILE_SIZE_LIMIT, listener);
			frostSettings.removePropertyChangeListener(SettingsClass.LOG_LEVEL, listener);
			
			if (fileHandler != null) {
				rootLogger.removeHandler(fileHandler);
				fileHandler.close();	
			}
		}
	}
	
	/**
	 * 
	 */
	private class Listener implements PropertyChangeListener {
		/**
		 * 
		 */
		public Listener() {
			super();
		}
		/* (non-Javadoc)
		 * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
		 */
		public void propertyChange(PropertyChangeEvent evt) {
			if (evt.getPropertyName().equals(SettingsClass.LOG_TO_FILE)) {
				logToFileSettingChanged();
			}
			if (evt.getPropertyName().equals(SettingsClass.LOG_FILE_SIZE_LIMIT)) {
				logFileSizeSettingChanged();
			}
			if (evt.getPropertyName().equals(SettingsClass.LOG_LEVEL)) {
				logLevelSettingChanged();
			}
		}
	}	
	
	public static final String VERY_LOW = "Very low";	//Severe
	public static final String LOW = "Low";				//Warning
	public static final String MEDIUM = "Medium";		//Info
	public static final String HIGH = "High";			//Finer
	public static final String VERY_HIGH = "Very high";	//Finest
	public static final String DEFAULT = "Low";
	
	private static final String LOG_FILE_NAME = "frost%g.log";

	private SettingsClass frostSettings = null;
	private Listener listener = new Listener();
	private Logger rootLogger = null;
	private FileHandler fileHandler = null;
	private SimpleFormatter simpleFormatter = new SimpleFormatter();

	/**
	 * 
	 */
	public Logging(SettingsClass frostSettings) {
		super();
		this.frostSettings = frostSettings;
		initialize();
	}
	
	/**
	 * 
	 */
	private void initialize() {
		LogManager logManager = LogManager.getLogManager();
		rootLogger = logManager.getLogger("");
		//We remove the console handler that is used by default
		Handler[] handlers = rootLogger.getHandlers();
		for (int i = 0; i < handlers.length; i++) {
			rootLogger.removeHandler(handlers[i]);	
		}
		
		Runtime.getRuntime().addShutdownHook(new ShutdownHook());

		logToFileSettingChanged();
		
		frostSettings.addPropertyChangeListener(SettingsClass.LOG_TO_FILE, listener);
		frostSettings.addPropertyChangeListener(SettingsClass.LOG_FILE_SIZE_LIMIT, listener);
		frostSettings.addPropertyChangeListener(SettingsClass.LOG_LEVEL, listener);
	}
	
	/**
	 * 
	 */
	private void logLevelSettingChanged() {
		boolean valueFound = setLevel(frostSettings.getValue(SettingsClass.LOG_LEVEL));
		if (!valueFound) {
			//If the value in the settings was not a valid one, we set the default one
			setLevel(frostSettings.getDefaultValue(SettingsClass.LOG_LEVEL));
		}
	}
		
	/**
	 * @param string
	 * @return
	 */
	private boolean setLevel(String level) {
		if (level.equals(VERY_LOW)) {
			rootLogger.setLevel(Level.SEVERE);
			return true;
		}
		if (level.equals(LOW)) {
			rootLogger.setLevel(Level.WARNING);
			return true;
		}
		if (level.equals(MEDIUM)) {
			rootLogger.setLevel(Level.INFO);
			return true;
		}
		if (level.equals(HIGH)) {
			rootLogger.setLevel(Level.FINER);
			return true;
		}
		if (level.equals(VERY_HIGH)) {
			rootLogger.setLevel(Level.FINEST);
			return true;
		}
		return false;
	}

	/**
	 * 
	 */
	private void logFileSizeSettingChanged() {
		// We only change the file size if logging is not disabled
		if (!Level.OFF.equals(rootLogger.getLevel())) {
			try {
				int fileSize = frostSettings.getIntValue(SettingsClass.LOG_FILE_SIZE_LIMIT);
				if (fileHandler != null) {
					rootLogger.removeHandler(fileHandler);
					fileHandler.close();
					fileHandler = null;
				}
				fileHandler = new FileHandler(LOG_FILE_NAME, fileSize * 1024, 2, true);
				fileHandler.setEncoding("UTF-8");
				fileHandler.setFormatter(simpleFormatter);
				rootLogger.addHandler(fileHandler);
			} catch (IOException exception) {
				logger.log(Level.SEVERE, "There was an error while initializing the logging system.", exception);
			}
		}
	}
	
	/**
	 * 
	 */
	private void logToFileSettingChanged() {
		// We only change the level if logging is not disabled
		if (!Level.OFF.equals(rootLogger.getLevel())) {
			if (frostSettings.getBoolValue(SettingsClass.LOG_TO_FILE)) {
				rootLogger.setLevel(null);
				logLevelSettingChanged();
				logFileSizeSettingChanged();
			} else {
				rootLogger.setLevel(Level.OFF);
			}
		}
	}
}
