package fiw.fcp;

import fiw.core.*;
import java.io.*;


public class FCPSubsystemTest {

    public static final String METADATA="Version\nRevision=1\nEnd";   
    public static final String DATA_STR= "FIW's FCP tests!";
    public static final int DATA_LENGTH=DATA_STR.length();
    
    public static final InputStreamBuilder DATA= new InputStreamBuilder() {
	    public InputStream buildStream() throws IOException {
		return new ByteArrayInputStream
		    (DATA_STR.getBytes("ISO-8859-1"));
	    }
	};
    
    private FCPConnection conn;
    private int htl;
    
    public FCPSubsystemTest(FCPConnection conn, int htl) {
	this.conn=conn;
	this.htl=htl;
    }

    public static void main(String[] args) {
	System.out.println("FCPSubsystemTest running:");
	FCPSubsystemTest fst =
	    new FCPSubsystemTest(FCPConn.fcpFactory("127.0.0.1",8481), 0);
	try {
	    if (!fst.runTests()) {
		System.out.println("TEST FAILED!");
	    }
	} catch (Exception ex) {
	    ex.printStackTrace();
	}
    }

    public boolean runTests() throws IOException {
	if (!testHandshake()) return false;
	if (!testInserts()) return false;
	if (!testCompatibility()) return false;
	if (!testSSK()) return false;
	if (!testFEC()) return false;
//	if (!testErrors()) return false;
	System.out.println("\nTests successful!");
	return true;
    }
    
    /// tests follow:

    /**
     * Handshake tests.
     * This includes ClientHello and ClientInfo messages.
     */
    public boolean testHandshake() throws IOException {
	String hs = conn.handshake();
	if (hs==null) {
	    System.out.println("\nHandshake is null!");
	    return false;
	}
	System.out.print(".");
	String ni = conn.getNodeInfo();
	if (ni==null) {
	    System.out.println("\nNodeInfo is null!");
	    return false;
	}
	System.out.print(".");
	int nl = conn.getNetworkLoad();
	if (nl<0 || nl > 100) {
	    System.out.println("\nInvalid network load!");
	    return false;
	}	
	System.out.print("_");
	return true;
    }
    
    /**
     * General insert and retrieve tests.
     * First create random data, then check that that data is not
     * there (both fetchFingerprint and fetchMetadata return
     * DataNotFound) then insert it twice (second one should generate
     * a key collision) and request it twice again (both the metadata
     * and the fingerprint have to be correct). Last but not least it
     * tries to delete the key from the local datastore and drops a
     * message if it is still there (however, that is not seen as an
     * error).
     */
    public boolean testInserts() throws IOException {
	final String randomData = "FIW's FCP test rnd"+Math.random();
	InputStreamBuilder isb= new InputStreamBuilder() {
		public InputStream buildStream() throws IOException {
		    return new ByteArrayInputStream
			(randomData.getBytes("ISO-8859-1"));
		}
	    };
	String chk = conn.calcCHK(METADATA, isb, randomData.length(),"1");
	if (chk == null) {
	    System.out.println("\ncalcCHK is null");
	}
	System.out.print(".");
	FCPGetResult gres = conn.fetchFingerprint(chk, htl, true);
	if (!(gres instanceof DataNotFoundResult)) {
	    System.out.println("\nNo DataNotFoundResult: "+
			       gres.getClass().getName());
	    return false;
	}
	System.out.print(".");
	gres = conn.fetchMetadata(chk, htl);
	if (!(gres instanceof DataNotFoundResult)) {
	    System.out.println("\nNo DataNotFoundResult: "+
			       gres.getClass().getName());
	    return false;
	}
	System.out.print(".");
	FCPInsertResult ires = conn.insertStream
	    ("CHK@",METADATA, isb, htl, randomData.length(),"2");
	if(ires instanceof KeyCollisionResult) {
	    System.out.println("\nKey collision!");
	    return false;
	} else if (!(ires instanceof InsertSuccessResult)) {
	    System.out.println("\nNo InsertSuccessResult: "+
			       ires.getClass().getName());
	    return false;
	}
	String chk2=((InsertSuccessResult)ires).getKey();
	if (!chk.equals(chk2)) {
	    System.out.println("\nDifferent keys: "+chk+"!="+chk2);
	    return false;
	}
	System.out.print(".");
	ires = conn.insertStream
	    ("CHK@",METADATA, isb, htl, randomData.length(),"3");
	if (!(ires instanceof KeyCollisionResult)) {
	    System.out.println("\nNo KeyCollisionResult: "+
			       ires.getClass().getName());
	    return false;
	}
	chk2=((InsertSuccessResult)ires).getKey();
	if (!chk.equals(chk2)) {
	    System.out.println("\nDifferent keys: "+chk+"!="+chk2);
	    return false;
	}
	System.out.print(".");
	gres = conn.fetchFingerprint(chk, htl, true);
	if (!(gres instanceof FingerprintResult)) {
	    System.out.println("\nNo FingerprintResult: "+
			       gres.getClass().getName());
	    return false;
	}
	String fp = HashUtil.makeFingerprint(METADATA, isb,
					     randomData.length());
	String fp2 = gres.getPayload();
	if (!fp.equals(fp2)) {
	    System.out.println("\nDifferent fingerprints: "+fp+"!="+fp2);
	    return false;
	}
	System.out.print(".");
	gres = conn.fetchMetadata(chk, htl);
	if (!(gres instanceof MetadataResult)) {
	    System.out.println("\nNo FingerprintResult: "+
			       gres.getClass().getName());
	    return false;
	}
	String metadata2 = gres.getPayload();
	if (!METADATA.equals(metadata2)) {
	    System.out.println("\nDifferent metadata: "+
			       METADATA+"!="+metadata2);
	    return false;
	}
	System.out.print(".");
	String dk = conn.deleteKey(chk);
	if (dk==null) {
	    System.out.println("\nDeleteKey response is null!");
	    return false;
	}
	System.out.print(".");
	gres = conn.fetchFingerprint(chk, htl, true);
	if (!(gres instanceof DataNotFoundResult)) {
	    System.out.println("\nNote: Key deleting does not work!");
	}
	System.out.print("_");
	return true;
    }

    /**
     * Tests if the current network is "compatible" to freenet 0.5+.
     * This means that given data results in a pre-calculated CHK
     * value and that a given public key matches a given private key.
     */
    public boolean testCompatibility() throws IOException {
	final String resultCHK=
	    "CHK@IxKHvm2kny37YfignlkueZe0jKoKAwI,zDgbfXVABUMalUsw~Z8DxA",
	    modifiedCHK=            //modification is here |
	    "CHK@IxKHvm2kny37YfignlkueZe0jKoKAwI,zDgbfXVABUNalUsw~Z8DxA",
	    publicSSK="zqAWHHcbu2KtCRbOrnZMJ7loajo", // from Freenet Explained
	    privateSSK="AK3AmGr3TtoJYK8pH8TapcHgL7p9",
	    entropy="QjfR87GSJipA4JW0X4ly0g";
	FCPInsertResult ires=conn.insertStream
	    ("CHK@", METADATA, DATA, htl, DATA_LENGTH, "10");
	String chk=((InsertSuccessResult)ires).getKey();
	if (!resultCHK.equals(chk)) {
	    System.out.println("\nWARNING: Different CHK: "+chk);
	} else {
	    System.out.print("/");
	    FCPGetResult gres = conn.fetchFingerprint(modifiedCHK, htl, true);
	    if (!(gres instanceof URIErrorResult)) {
		System.out.println("\nNo URIErrorResult: "+
				   gres.getClass().getName());
		return false;
	    }
	    System.out.print("/");
	    gres = conn.fetchMetadata(modifiedCHK, htl);
	    if (!(gres instanceof URIErrorResult)) {
		System.out.println("\nNo URIErrorResult: "+
				   gres.getClass().getName());
		return false;
	    }	
	}
	System.out.print(".");
	ires=conn.insertStream
	    ("SSK@"+privateSSK+","+entropy+"/fiwFCPtest", METADATA, DATA,
	     htl, DATA_LENGTH, "10");
	chk=((InsertSuccessResult)ires).getKey();
	if (!chk.equals("SSK@"+publicSSK+"PAgM,"+entropy+"/fiwFCPtest")) {
	    System.out.println("\nWARNING: Different Pubkey: "+chk);
	}
	System.out.print("_");
	return true;
    }

    public boolean testSSK() throws IOException {
	SSKPair ssk = conn.getNewSSK();
	// a keypair to reproduce the bug
// 	ssk=new SSKPair("AMeWlICseejBDc4TODsImwt5wqq6",
// 			"ADJkkb75LCm666vBBvA4GPSzGLU",
// 			"bHbkr54OvwOztp2y38M9kg");
	if (ssk.getError() != null) {
	    System.out.println("\nError while generating SSK pair: "+
			       ssk.getError());
	    return false;
	}
	System.out.print(".");
	FCPInsertResult ires=conn.insertStream
	    ("SSK@"+ssk.getPrivateKey()+"/fiwFCPtest", METADATA, DATA,
	     htl, DATA_LENGTH, "30");
	String key=((InsertSuccessResult)ires).getKey();
	String key2 = "SSK@"+ssk.getPublicKey()+"PAgM/fiwFCPtest";
	if (!key.equals(key2)) {
	    System.out.println("\nDifferent Pubkey: "+key+ "!=" + key2);
	    return false;
	}
	System.out.print(".");
	FCPGetResult gres = conn.fetchMetadata(key,htl);
	String mdata=((MetadataResult)gres).getPayload();
	if (!METADATA.equals(mdata)) {
	    System.out.println("\nDifferent Metadata: "+mdata);
	}
	if (ssk.getEntropy().equals("")) {
	    System.out.println("\nWARNING: Your node does not "+
			       "support SSK entropy");
	} else {
	    System.out.print(".");
	    ires=conn.insertStream
		("SSK@"+ssk.getPrivateKey()+","+ssk.getEntropy()+
		 "/fiwFCPtestWE", METADATA, DATA, htl, DATA_LENGTH, "31");
	    key=((InsertSuccessResult)ires).getKey();
	    key2 = "SSK@"+ssk.getPublicKey()+"PAgM,"+ssk.getEntropy()+
		"/fiwFCPtestWE";
	    if (!key.equals(key2)) {
		System.out.println("\nDifferent Pubkey: "+key+ "!=" + key2);
		return false;
	    }
	    System.out.print(".");
	    gres = conn.fetchMetadata(key,htl);
	    mdata=((MetadataResult)gres).getPayload();
	    if (!METADATA.equals(mdata)) {
		System.out.println("\nDifferent Metadata: "+mdata);
	    }
	    key = "SSK@"+ssk.getPublicKey()+"PAgM,"+ssk.getEntropy()+
		"/fiwFCPtest";
	    System.out.print(".");
	    gres = conn.fetchMetadata(key,htl);
	    if (!(gres instanceof URIErrorResult)) {
		System.out.println("\nNo URIErrorResult: "+
				   gres.getClass().getName());
		if (gres instanceof GeneralErrorResult) {
		    System.out.println("This is a freenet bug. Ignoring.");
		} else {
		    return false;
		}
	    }	
	    key = "SSK@"+ssk.getPublicKey()+"PAgM/fiwFCPtestWE";
	    System.out.print(".");
	    gres = conn.fetchMetadata(key,htl);
	    if (!(gres instanceof URIErrorResult)) {
		System.out.println("\nNo URIErrorResult: "+
				   gres.getClass().getName());
		if (gres instanceof GeneralErrorResult) {
		    System.out.println("This is a freenet bug. Ignoring.");
		} else {
		    return false;
		}
	    }	
	}
	//FIXME: Add SVK tests or are these keys obsolete anyway?
	System.out.print("_");
	return true;
    }
    

    public boolean testFEC() throws IOException {
	// FIXME: add FEC tests
	return true;
    }

    /**
     * Test error conditions. This includes FormatErrors and
     * URIErrors, where possible. As this test pollutes the logfiles,
     * it is usually commented out.
     */
    public boolean testErrors() throws IOException {
	// cannot think of any calcCHK errors...
	System.out.println("\n\nTesting errors; ignore error messages:");
	FCPInsertResult ires = conn.insertStream("CHK@",METADATA, DATA,
					    -3, DATA_LENGTH, "21");
	if (ires!=null) {
	    System.out.println("Incorrect request has been accepted");
	    return false;
	}
	ires = conn.insertStream("KSK@10%more",METADATA, DATA,
				 htl, DATA_LENGTH, "22");
	if (ires!=null) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	ires = conn.insertStream("KSK@I_like\nBlah\nFasel",METADATA, DATA,
					    htl, DATA_LENGTH, "23");
	if (ires!=null) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	FCPGetResult gres = conn.fetchMetadata("KSK@foo",-3);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	gres = conn.fetchMetadata("KSK@10%more",0);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	gres = conn.fetchMetadata("KSK@I_like\nBlah\nFasel",0);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	gres = conn.fetchFingerprint("KSK@foo",-3, true);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	gres = conn.fetchFingerprint("KSK@10%more",0, true);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	gres = conn.fetchFingerprint("KSK@I_like\nBlah\nFasel",0, true);
	if (!(gres instanceof GeneralErrorResult)) {
	    System.out.println("Incorrect request has been accepted.");
	    return false;
	}
	String res = conn.deleteKey("KSK@10%more");
	System.out.println(res);
	res = conn.deleteKey("KSK@I_like\nBlah\nFasel");
	System.out.println(res);
	return true;
    }

}
