package fiw.fcp;
import fiw.core.*;

import java.io.*;
import java.net.*;

/**
 * Implements FEC check chunk generation. As only one FEC process
 * should be enabled the same time, this class uses a kind of
 * synchronized singleton.
 * @author mihi
 */
public class FECUtil implements FECConnection {
    
    private static boolean isFree=true;
    private static FECUtil instance;
    private FCPConnection sc;
    
    private FECUtil () {};

    /**
     * Get the singleton instance.
     * @param fc the FCPConnection to use
     * @return the instance, when it is free, or <code>null</code> if
     * someone else uses the instance at the moment.
     */
    protected static synchronized FECUtil getInstance(FCPConnection fc) {
	if (instance==null) {
	    instance=new FECUtil();
	}
	if (!isFree) {
	    return null;
	}
	isFree=false;
	instance.sc=fc;
	return instance;
    }

    /**
     * Get the singleton instance - blocks until it is free.
     * @param fc the FCPConnection to use
     * @return the singleton instance
     */
    protected static synchronized FECUtil waitForInstance(FCPConnection fc) {
	if (instance==null) {
	    instance=new FECUtil();
	}
	while (!isFree) {
	    try {
		FECUtil.class.wait();
	    } catch (InterruptedException e) {
		e.printStackTrace();
	    }
	}
	isFree=false;
	instance.sc=fc;
	return instance;
    }

    /**
     * Releases this singleton instance. After calling this, this
     * instance may no longer be used.
     */
    public void releaseInstance() {
	synchronized(FECUtil.class) {
	    isFree=true;	
	    FECUtil.class.notifyAll();
	}
    }

    /**
     * Produces FEC segment headers for a given file size.
     * @return the segment headers
     * @throws IOException when an IO error occurs.
     */
    private synchronized FECSegmentHeader[] getFECSegmentHeaders
	(long filelength) throws IOException {
	FECSegmentHeader[] segments=null;
	FCPMessage out = new FCPMessage("FECSegmentFile");
	out.put("AlgoName", "OnionFEC_a_1_2");
	out.put("FileLength", filelength);
	Socket s = sc.makeSocket();
	BufferedOutputStream bos=new BufferedOutputStream
	    (s.getOutputStream());
	BufferedInputStream bi=new BufferedInputStream
	    (s.getInputStream());
	out.writeMessage(bos);
	int segcount = 1;
	int segnum = -1;
	while (segnum < segcount-1) {
	    FCPMessage rep = FCPMessage.readMessage(bi);
	    if (rep.getCmd().equals("SegmentHeader")) {
		segcount=(int) rep.getInt("Segments");
		if (segments==null) {
		    segments=new FECSegmentHeader[segcount];
		}
		segnum = (int) rep.getInt("SegmentNum");
		int blockcount = (int) rep.getInt("BlockCount");
		int blocksize = (int) rep.getInt("BlockSize"); 
		int cblockcount = (int) rep.getInt("CheckBlockCount");
		int cblocksize = (int) rep.getInt("CheckBlockSize");
		segments[segnum] = new FECSegmentHeader
		    (rep.headToString()+"EndMessage\n",
		     blockcount, blocksize,
		     cblockcount, cblocksize);
	    } else if (rep.getCmd().equals("Failed")) {
		System.out.println(rep.headToString());
		FIWSystem.log().println(rep.headToString());
		return null;
	    }
	}
	s.close();
	return segments;
    }


    /**
     * Calculates the sizes of check and data chunks for a given file size.
     * @param filesize the size of the file to encode.
     * @return the FECMetrics object for this file size
     */
    public FECMetrics getFECMetrics(long filesize) {
	try {
	    FECSegmentHeader[] segs = getFECSegmentHeaders(filesize);
	    return new FECMetrics(segs);
	} catch (IOException e) {
	    e.printStackTrace();
	    return null;
	}
    }

    
    /**
     * Does the actual FEC encoding.
     * @param src the file to be encoded
     * @param dst the file to hold the check chunks
     * @return the FECMetrics, like {@link #getFECMetrics}, or
     * <code>null</code> if an error occurred
     */
    public FECMetrics fecEncodeFile(File src, File dst) {
	FECMetrics fms = getFECMetrics(src.length());
	return fecEncodeFile(src, dst, fms) ? fms : null;
    }


    /**
     * Does the actual FEC encoding.
     * @param src the file to be encoded
     * @param dst the file to hold the check chunks
     * @param fms the fecMetrics to be used
     * @return true iff the encoding was successful
     */
    public boolean fecEncodeFile(File src, File dst, FECMetrics fms) {
	final File srcF=src;
	try {
	    BufferedOutputStream dst_= new BufferedOutputStream
		(new FileOutputStream(dst));
	    InputStreamBuilder src_ = new InputStreamBuilder() {
		    public InputStream buildStream() throws IOException {
			return new BufferedInputStream
			    (new FileInputStream(srcF));
		    }
		};
	    return fecEncodeStream(src_, dst_,fms);
	} catch (IOException ex) {
	    ex.printStackTrace();
	    ex.printStackTrace(FIWSystem.log());
	    return false;
	}

    }
    
    public FECMetrics fecEncodeStream(InputStreamBuilder src, OutputStream dst,
				      long datasize) {
	FECMetrics fms = getFECMetrics(datasize);
	return fecEncodeStream(src, dst, fms) ? fms : null;
    }
    
    public boolean fecEncodeStream(InputStreamBuilder src, OutputStream dst,
				   FECMetrics fms) {
    
	try {
	    FECSegmentHeader[] segments = fms.getHeads();
	    if (segments.length==0) {
		return false;
	    }
	    for (int iii=0;iii<segments.length;iii++) {
		InputStream fin = src.buildStream();
		FCPMessage out = new FCPMessage("FECEncodeSegment");
		out.put("MetadataLength", segments[iii].getHeadLength());
		out.put("DataLength", segments[iii].getRawDataLength());
		out.addTrail(segments[iii].getHead());
		for (int i=0;i<iii;i++) {
		    int x = segments[i].getDataLength();
		    while (x>0) { x-= (int) fin.skip(x); }
		}
 		out.setIS(new PaddingInputStream
 			  (fin, 0,segments[iii].getDataLength()));
		Socket s=sc.makeSocket();
		BufferedOutputStream bos=new BufferedOutputStream
		    (s.getOutputStream());
		BufferedInputStream bi=new BufferedInputStream
		    (s.getInputStream());
		out.writeMessage(bos);
		FCPMessage rep = FCPMessage.readMessage(bi);
		if (rep.getCmd().equals("Failed")) {
		    FIWSystem.log().println(rep.headToString());
		    return false;
		} else if (rep.getCmd().equals("BlocksEncoded")) {
		    int count = (int) rep.getInt("BlockCount");
		    int size = (int) rep.getInt("BlockSize");
		    int read = 0;
		    while (read < count * size) {
			rep = FCPMessage.readMessage(bi);
			if (rep.getCmd().equals("DataChunk")) {
			    // another encoding bug.
			    // thanks thelema for 5 hours of clueless
			    // bug hunting.       -- mihi, 2003-12-22
			    dst.write(rep.getTrailing().toString()
				     .getBytes("ISO-8859-1"));
			    read += rep.getTrailing().length();
			} else {
			    FIWSystem.log().println(rep.headToString());
			    return false;
			}
		    }
		}
		bi.close();
		bos.close();
		s.close();
	    }
	    return true;
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return false;
	}
    }
    
    /**
     * Creates the splitfile metadata for a FEC splitfile.
     * @param filesize the size of the file
     * @param blocks the keys for the blocks
     * @param ctype the content type to use
     * @param csum an SHA1 hash of the file
     */
    public synchronized String makeSplitfile(long filesize,
					     String[] blocks,
					     String ctype,
					     String csum) {
	try {
	    for (int i=0;i<blocks.length;i++) {
		if (blocks[i] == null || "".equals(blocks[i]))
		    return null;
	    }
	    // FIXME: hand these headers through?
	    FECSegmentHeader[] segments = getFECSegmentHeaders(filesize);

	    if (segments.length==0) {
		return null;
	    }
	    int blocktotal=0, cblocktotal=0, blockdone = 0, cblockdone = 0;
	    for (int i=0;i<segments.length;i++) {
		blocktotal+=segments[i].getBlockCount();
		cblocktotal+=segments[i].getCheckBlockCount();
	    }
	    if (blocktotal+cblocktotal != blocks.length) {
		FIWSystem.log().println("FEC_SPLITFILE_ERROR: "+blocktotal +
				     " + "+ cblocktotal+
				     " != "+ blocks.length);
		return null;
	    }
	    cblockdone=blocktotal;
	    StringBuffer buf = new StringBuffer();
	    for (int i=0;i<segments.length;i++) {
		buf.append(segments[i].getHead());
		buf.append("BlockMap\n");
		for (int j=0;j<segments[i].getBlockCount();j++) {
		    buf.append("Block."+Integer.toHexString(j)+"=freenet:"+
			       blocks[blockdone+j]+"\n");
		}
		for (int j=0;j<segments[i].getCheckBlockCount();j++) {
		    buf.append("Check."+Integer.toHexString(j)+"=freenet:"+
			       blocks[cblockdone+j]+"\n");
		}
		blockdone+=segments[i].getBlockCount();
		cblockdone+=segments[i].getCheckBlockCount();
		buf.append("EndMessage\n");
	    }
	    if (blockdone != blocktotal || cblockdone != blocks.length) {
		FIWSystem.log().println("FEC_SPLITFILE_ERROR: Fatal 3819");
		return null;
	    }
	    Socket s = sc.makeSocket();
	    BufferedOutputStream bos=new BufferedOutputStream
		(s.getOutputStream());
	    BufferedInputStream bi=new BufferedInputStream
		(s.getInputStream());
	    FCPMessage out = new FCPMessage("FECMakeMetadata");
	    out.put("Description", "file");
	    out.put("MimeType", ctype);
	    if (csum != null) 
		out.put("Checksum", csum);
	    out.put("DataLength", buf.length());
	    out.addTrail(buf.toString());
	    out.writeMessage(bos);
	    boolean failed=false, data = false;
	    String line;
	    StringBuffer buf2=new StringBuffer();
	    while((line=FCPMessage.readLine(bi))!= null) {
		buf2.append(line).append('\n');
		if(line.equals("Failed")) 
		    failed=true;
		else if (line.equals("Data")) {
		    data=true;
		    buf2.setLength(0);
		}
	    }
	    if (failed || !data) {
		FIWSystem.log().println("FEC_SPLITFILE: "+buf2.toString());
		return null;
	    }
	    return buf2.toString();
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return null;
	}
    }
}
