package fiw.fcp;
import java.io.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * Represents one message sent via the FCP connection.
 * @author thelema
 */
public class FCPMessage {
    private String commandName;
    private Hashtable fields;
    private StringBuffer trailing = new StringBuffer();
    private InputStream is;

    /**
     * Creates a new FCP message.
     * @param cmd the FCP command
     */
    public FCPMessage(String cmd) {
	commandName = cmd;
	fields = new Hashtable();
    }

    /**
     * Reads a FCP message from an input stream (usually from a socket).
     * @param in the InputStream to read the data from.
     */
    public static FCPMessage readMessage (InputStream in) throws IOException {
	String line;
	FCPMessage message;
	    
	line = readLine(in);
	if (line == null) throw new IOException("Premature end of stream");
	message = new FCPMessage(line);
	    
	//Get and parse text message
	while((line=readLine(in))!= null) {
	    if (line.equals("Data")) break;
	    if (line.equals("EndMessage")) break;
	    int n = line.indexOf('=');
	    String var = line.substring(0,n);
	    String val = line.substring(n + 1);
	    message.put (var, val);
	}
	    
	//get trailing field if it exists
	if (line.equals("Data")) {
	    long len = message.getInt("Length");
	    byte[] buf = new byte[16384];
	    while (len > 0) {
		int toread = Math.min(16384, (int)len);
		int read = in.read(buf, 0, toread);
		// please do *not* decode arbitrary binary data as
		// UTF-8 - UTF-8 has lots of "undefined" sequences
		// and that will break binary data. The only bijective
		// encoding in the default SDK is latin-1
		//                                 -- mihi, 2003-12-16
		message.addTrail(new String(buf, 0, read, "ISO-8859-1"));
		len -= read;
	    }
	}
	    
	return message;
    }


    /**
     * Writes the message appropriately to the OutputStream.
     * Note: when both <code>trailing</code> and <code>is</code> are
     * set, <code>trailing</code> will be output first, then
     * <code>is</code>.  (<code>trailing</code> being the metadata and
     * <code>is</code> being the data.)
     * @param out the stream to write the data to
     */	
    public void writeMessage (OutputStream out) throws IOException{
	if (trailing.length() == 0 ) trailing = null; // FIXME
	// do not use the default charset; on some (greek?) machines
	// this it UTF-16, which will break everything...
	//      -- mihi, 2003-12-16
	out.write("\000\000\000\002".getBytes("ISO-8859-1"));
	out.write(headToString().getBytes("ISO-8859-1"));
	    
	if (trailing != null || is != null)
	    out.write("Data\n".getBytes("ISO-8859-1"));
	else
	    out.write("EndMessage\n".getBytes("ISO-8859-1"));

	if (trailing != null)
	    out.write(trailing.toString().getBytes("ISO-8859-1"));

	if (is != null) {
	    byte[] b=new byte[262144];
	    int len;	    
	    while((len=is.read(b))!=-1) {
		out.write(b,0,len);
	    }
	    is.close();
	}
	out.flush();
    }

    /**
     * Return the command name.
     */
    public String getCmd() { return commandName; }

    /**
     * Return the value of a field.
     * @param var the field name
     */
    public String getVal(String var) { return (String) fields.get(var); }

    /**
     * Return the int value of a field, or 0 if the files is no
     * (hexadecimal) integer.
     * @param var the field name
     */
    public long getInt(String var) {
	try {
	    return Long.parseLong((String) fields.get(var), 16);
	} catch (NumberFormatException ex) {
	    ex.printStackTrace();
	    return 0;
	}
    }

    /**
     * Return the int value of a field.
     * @param var the field name
     * @param defaul a default value if the field is not set
     */
    public long getInt(String var, long defaul) {
	if (fields.get(var)==null) {
	    return defaul;
	} else {
	    return getInt(var);
	}
    }
    public void put (String var, String val) { fields.put(var,val); }
    public void put (String var, int val) { fields.put(var, Integer.toHexString(val)); }
    public void put (String var, long val) { fields.put(var, Long.toHexString(val)); }

    /**
     * Return the message's head.
     */
    public String headToString() {
	StringBuffer sb = new StringBuffer(commandName);
	sb.append("\n");
	for (Enumeration e = fields.keys() ; e.hasMoreElements() ;) {
	    String var = (String) e.nextElement();
	    sb.append(var);
	    sb.append("=");
	    sb.append(getVal(var));
	    sb.append("\n");
	}
	return sb.toString();
    }

    public String toString() {
	// FIXME: implement proper toString()
	return headToString();
    }

    public void addTrail (String trail) { trailing.append(trail); }
    public StringBuffer getTrailing () { return trailing; }
    public void setIS (InputStream isIn) { is = isIn; }

    /**
     * kinda dirty hack to read lines (only terminated by LF) from
     * inputstreams w/o the bugs in DataOutputStream and allowing to
     * continue reading binary data
     */
    public static String readLine(InputStream in) throws IOException {
	StringBuffer sb= new StringBuffer();
	int chr;
	boolean read=false;
	while((chr=in.read()) != -1) {
	    read=true;
	    if (chr == 10) break;
	    sb.append(new String(new byte[]{(byte) chr},"ISO-8859-1"));
	}
	return read?sb.toString():null;
    }    

}
