package fiw.fcp;
import fiw.core.InputStreamBuilder;
import java.io.InputStream;
import java.io.IOException;
import java.net.Socket;
import java.util.List;

/**
 * An interface for {@link FCPConn} to make testing easier.
 * @author mihi
 */
public interface FCPConnection {

    /**
     * Try to connect all nodes of this connection (usually only one)
     * and return their "status messages".
     */
    public String handshake();

    /**
     * Generate a new SSK pair and return it.
     */
    public SSKPair getNewSSK();

    /**
     * Calculate a CHK key without inserting it.
     * @param metadata the metadata of the key.
     * @param data the data of the key.
     * @param dataLength the length of the data.
     * @param logname the name to use in error logs if an error occurs.
     * @return the CHK key
     */
    public String calcCHK(String metadata, InputStreamBuilder data,
			  long dataLength, String logname);

    /**
     * Insert a key (from a stream builder).
     * @param key the key to insert into ("CHK@" to insert into a CHK)
     * @param metadata the metadata of the key
     * @param data the data of the key
     * @param htl the Hops To Live of the key
     * @param dataLength the length of the data
     * @param logname ethe name to use in error logs if an error occurs
     */
    public FCPInsertResult insertStream (String key, String metadata,
					 InputStreamBuilder data,
					 int htl, long dataLength,
					 String logname);
	
    /**
     * Insert a key (from a stream).
     * @param key the key to insert into ("CHK@" to insert into a CHK)
     * @param metadata the metadata of the key
     * @param data the data of the key
     * @param htl the Hops To Live to use
     * @param dataLength the length of the data
     * @param logname ethe name to use in error logs if an error occurs
     */
    public FCPInsertResult insertStream (String key, String metadata,
					 InputStream data,
					 int htl, long dataLength,
					 String logname);

    
    /**
     * Fetch the metadata of a key.
     *
     * @param key the key to fetch
     * @param HTL the Hops To Live to use
     */
    public FCPGetResult fetchMetadata(String key, int HTL);
    
    /**
     * Fetch a "fingerprint" (hash) of a key
     *
     * @param key the key to fetch
     * @param HTL the Hops to Live to use
     * @param pluslocal whether to look in the local datastore as well
     * (if you set FIW to ignore local datastore)
     */
    public FCPGetResult fetchFingerprint(String key, int HTL,
				   boolean pluslocal);
    
    /**
     * Delete a key from your local datastore. Depending on your
     * Freenet node version, this may or may not work.
     *
     * @param key the key to remove
     * @return the plain text response of the freenet node
     */
    public String deleteKey(String key);

    /**
     * Return current network load (in percent).
     * @see #getNodeInfo()
     */
    public int getNetworkLoad();

    /**
     * Return current node info (If this is a round robin connection,
     * a random node is queried).
     */
    public String getNodeInfo();

    /**
     * Get an instance of the FEC decoder. This may return
     * <code>null</code> if the encoder is currently used.
     */
    public FECConnection getFECInstance();

    /**
     * Wait for an instance of the FEC decoder. If the encoder is
     * currently used, this method blocks until it is available.
     */
    public FECConnection waitForFECInstance();

    /**
     * Return the current FCP host. If this is a round robin FCP
     * connection, return the first host in the list.
     */
    public String getHost();

    /**
     * Return all current FCP hosts.
     */
    public List getHostList();

    /**
     * Return the current FCP port.
     */
    public int getPort();

    /**
     * Internal method.
     */
    public Socket makeSocket() throws IOException;
}
