package fiw.fcp;
import fiw.core.*;
import java.io.*;
import java.net.*;
import java.security.MessageDigest;
import java.util.*;

/**
 * The "new" FCP layer implementation.
 * @author thelema
 */
public class FCPConn implements FCPConnection {
    private String fcpHost = null;
    private int fcpPort = 8481;
    private List hostList = null;
    private int rrPos = 0;
	
    private FCPConn (String h, int p) {
	fcpHost = h;
	fcpPort = p;
    }

    private FCPConn (List hl, int port) {
	hostList = hl;
	fcpPort = port;
    }

    /**
     * creates an unconnected FCP Connection; every operation will
     * throw an IOException
     */
    private FCPConn () {}

    /**
     * Return the current FCP port.
     */
    public int getPort() {
	return fcpPort;
    }

    /**
     * Return the current FCP host. If this is a round robin FCP
     * connection, return the first host in the list.
     */
    public String getHost() {
	if (hostList != null && !hostList.isEmpty())
	    return (String)hostList.get(0);
	if (fcpHost != null) return fcpHost;
	return null;
    }

    /**
     * Return all current FCP hosts.
     */
    public List getHostList() {
	if (hostList != null) return Collections.unmodifiableList(hostList);
	List l = new Vector();
	if (fcpHost != null) l.add(fcpHost);
	return Collections.unmodifiableList(l);
    }

    /**
     * Create a new FCPConnection.
     */
    public static FCPConn fcpFactory (String host, int port) {	
	FIWSystem.setts().setProperty("fcp.host",host);
	FIWSystem.setts().setProperty("fcp.port",""+port);
	if (host.indexOf('|') != -1) {
	    StringTokenizer st = new StringTokenizer(host,"|");
	    Vector hl = new Vector();
	    while (st.hasMoreTokens()) {
		String h = st.nextToken();
		if (handshake(h, port) != null) {
		    hl.addElement(h);
		} else {
		    FIWSystem.log().println("Unreachable: "+h);
		    System.err.println("Unreachable: "+h);
		}
	    }
	    if (hl.isEmpty())
		return new FCPConn();
	    else 
		return new FCPConn (hl, port);
	} else {
	    if (handshake(host,port) != null) {
		return new FCPConn(host, port);
	    } else 
		return new FCPConn();
	}
    }

    // private here! this won't change current server.
    private static String handshake(String host, int port) {
	try {
	    Socket s=new Socket(host,port);
	    FCPMessage reply = oneMessageEach(s, new FCPMessage("ClientHello"));
	    return reply.toString();
	} catch (SocketException e) {
	    FIWSystem.log().println("CLIENTHELLO: "+e.toString());
	    if (FIWSystem.doFullLog)
		FIWSystem.getInstance().getFullLog()
		    .println("CLIENTHELLO: "+e.toString());
	    return null;
	} catch(IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }

    public String handshake() {
	if (hostList== null && fcpHost == null)
	    return null;
	List hostList = getHostList();
	try {
	    StringBuffer result=new StringBuffer();
	    for (Iterator it = hostList.iterator(); it.hasNext();) {
		String host = (String) it.next();
		Socket s=new Socket(host, fcpPort);
		FCPMessage reply =
		    oneMessageEach(s, new FCPMessage("ClientHello"));
		if (reply.getVal("HighestSeenBuild") != null)
		    result.append("** before uploading, please upgrade to "+
				  "the latest build of Freenet **\n\n");
		result.append(reply.toString());
		if (it.hasNext()) result.append("-----\n");
	    }
	    return result.toString();
	} catch (SocketException e) {
	    FIWSystem.log().println("CLIENTHELLO: "+e.toString());
	    if (FIWSystem.doFullLog)
		FIWSystem.getInstance().getFullLog()
		    .println("CLIENTHELLO: "+e.toString());
	    return null;
	} catch(IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }
	
    private FCPMessage oneMessageEach(FCPMessage out) throws IOException {
	return oneMessageEach(makeSocket(), out);
    }

    private static FCPMessage oneMessageEach(Socket s, FCPMessage out)
	throws IOException {
	
	BufferedOutputStream bos=new BufferedOutputStream
	    (s.getOutputStream());
	BufferedInputStream bi=new BufferedInputStream
	    (s.getInputStream());
	out.writeMessage(bos);
	FCPMessage lastReply = FCPMessage.readMessage(bi);
	bos.close();
	bi.close();
	s.close();
	return lastReply;
    }
	
    private boolean nolocal (String where) {
	return "true".equals(FIWSystem.setts().getProperty
			     ("tuning.ignorelocal."+where));
    }
    
    public SSKPair getNewSSK() {
	try {
	    FCPMessage reply = oneMessageEach(new FCPMessage("GenerateSVKPair"));
	    
	    if (reply.getCmd().equals("Success")) {
		return new SSKPair(reply.getVal("PrivateKey"),
				   reply.getVal("PublicKey"),
				   reply.getVal("CryptoKey"));
	    } else {
		System.err.println("--error4--"+reply.headToString());
		return new SSKPair("Unexpected reply message");
	    }
	} catch (IOException e) {
	    e.printStackTrace(FIWSystem.log());
	    e.printStackTrace();
	    return new SSKPair("Unexpected error - consult logfile");
	}
    }
    
    public String calcCHK(String metadata, InputStreamBuilder data,
			  long dataLength, String logname) {
	try {
	    FCPMessage genchk = new FCPMessage ("GenerateCHK");
	    genchk.put("MetadataLength", metadata.length());
	    genchk.put("DataLength", metadata.length()+dataLength);
	    genchk.addTrail(metadata);
	    genchk.setIS(data.buildStream());
	    
	    FCPMessage ret = oneMessageEach(genchk);
	    if (ret.getCmd().equals("Success")) 
		return ret.getVal("URI").substring(8);
	    else
		return null;
	} catch (IOException e) {
	    e.printStackTrace(FIWSystem.log());
	    e.printStackTrace();
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }

    
    public FCPInsertResult insertStream (String key, String metadata,
				InputStreamBuilder data,
				int htl, long dataLength,
				String logname) {
	try {
	    return insertStream(key, metadata, data.buildStream(), htl,
				dataLength, logname);
	} catch (IOException e) {
	    e.printStackTrace(FIWSystem.log());
	    e.printStackTrace();
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }

    public FCPInsertResult insertStream (String key, String metadata,
				InputStream data,
				int htl, long dataLength,
				String logname) {
	
	int dataLen = metadata.length()+(int)dataLength;
	if (dataLen==0) {
	    System.out.println("**** Empty keys are not allowed ****");
	    FIWSystem.log().println("**** Empty keys are not allowed ****");
	    return null;
	}

	try {
	    Socket s = makeSocket();
	    BufferedOutputStream out=new BufferedOutputStream
		(s.getOutputStream());
	    BufferedInputStream bi=new BufferedInputStream
		(s.getInputStream());

	    FCPMessage clientput = new FCPMessage("ClientPut");
	    if (htl!=0 && nolocal("insert")) {
		clientput.put("RemoveLocalKey","true");
	    }
	    clientput.put("MetadataLength", metadata.length());
	    clientput.put("URI", "freenet:"+key);
	    clientput.put("DataLength", dataLen);
	    clientput.put("HopsToLive", htl);
	    clientput.addTrail(metadata);
	    clientput.setIS(data);
	    
	    clientput.writeMessage(out);

	    try {
	      while(true) {
		FCPMessage reply = FCPMessage.readMessage(bi);

		if (reply.getCmd().equals("Success")) {
		    return new InsertSuccessResult
			(reply.getVal("URI").substring(8));
		} else if (reply.getCmd().equals("KeyCollision")) {
		    return new KeyCollisionResult
			(reply.getVal("URI").substring(8));
		} else if (reply.getCmd().equals("RouteNotFound")) {
		    return new RouteNotFoundResult
			((int)reply.getInt("Unreachable",0),
			 (int)reply.getInt("Restarted",0),
			 (int)reply.getInt("Rejected",0));
		} else if (reply.getCmd().equals("DataNotFound")) {
		    // may not appear here, does nevertheless;
		    // treat it as RNF
		    return new RouteNotFoundResult(42,42,42);
		} else if (reply.getCmd().equals("FormatError")) {
		    System.out.println(reply.headToString());
		    FIWSystem.log().println(reply.headToString());
		    return null;
		} else if (reply.getCmd().equals("Pending")) {
		    continue;
		} else if (reply.getCmd().equals("Restarted")) {
		    continue;
		} else {
		    System.out.println("--error3--"+reply.headToString());
		    FIWSystem.log().println("--error3--"+reply.headToString());
		    return null;
		}
	      }
	    } catch (SocketTimeoutException ex) {
		return new SocketTimeoutResult();
	    }
	} catch (IOException e) {
	    e.printStackTrace(FIWSystem.log());
	    e.printStackTrace();
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }

    public FCPGetResult fetchMetadata(String key, int HTL) {
	try {
	    StringBuffer metaData = new StringBuffer();
	    long dataLen=0;
	    int metaDataLen=-1;
	    FCPMessage get = new FCPMessage("ClientGet");
	    get.put("URI", "freenet:"+key);
	    get.put("HopsToLive", HTL);
	    Socket s = makeSocket();
	    BufferedOutputStream bos=new BufferedOutputStream
		(s.getOutputStream());
	    BufferedInputStream bi=new BufferedInputStream
		(s.getInputStream());
	    
	    get.writeMessage(bos);
	    int counter=0;
	    while (counter < metaDataLen || metaDataLen == -1) {
		FCPMessage reply = FCPMessage.readMessage(bi);
		
		if (reply.getCmd().equals("DataFound")) {
		    dataLen = reply.getInt("DataLength");
		    metaDataLen = (int) reply.getInt("MetadataLength");
		    counter = 0;
		} else if (reply.getCmd().equals("RouteNotFound")) {
		    return new RouteNotFoundResult
			((int)reply.getInt("Unreachable",0),
			 (int)reply.getInt("Restarted",0),
			 (int)reply.getInt("Rejected",0));

		} else if (reply.getCmd().equals("DataNotFound")) {
		    return new DataNotFoundResult();
		} else if (reply.getCmd().equals("URIError")) {
		    return new URIErrorResult(reply.getVal("Reason"));
		} else if (reply.getCmd().equals("Restarted")) {
		    continue;
		} else if (reply.getCmd().equals("FormatError")) {
		    return new GeneralErrorResult(reply.headToString());
		} else 
		    return new GeneralErrorResult(reply.headToString());
		
		metaData= new StringBuffer();
		while (counter < metaDataLen) {
		    FCPMessage data = FCPMessage.readMessage(bi);
		    if (data.getCmd().equals("DataChunk")) {
			metaData.append(data.getTrailing());
			counter += data.getInt("Length");
		    } else if (data.getCmd().equals("Restarted")) {
			break;
		    } else 
			return new GeneralErrorResult(data.headToString());
		}
	    }
	    s.close();
	    bos.close();
	    bi.close();
	    
	    dataLen-=metaDataLen;
	    metaData.setLength(metaDataLen);
	    return new MetadataResult(dataLen,metaData.toString());	    
	} catch (SocketTimeoutException ex) {
	    return new SocketTimeoutResult();
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return new GeneralErrorResult("");
	}
    }

    public FCPGetResult fetchFingerprint(String key, int HTL,
				   boolean pluslocal) {
	try {
	    MessageDigest fingerprint = null;
	    long dataLen=-1, metaDataLen=0;
	    FCPMessage get = new FCPMessage("ClientGet");
	    if (!pluslocal && HTL != 0 && nolocal("fetch"))
		get.put("RemoveLocalKey", "true");
	    get.put("URI", "freenet:"+key);
	    get.put("HopsToLive", HTL);
	    Socket s = makeSocket();
	    BufferedOutputStream bos=new BufferedOutputStream
		(s.getOutputStream());
	    BufferedInputStream bi=new BufferedInputStream
		(s.getInputStream());
	    
	    get.writeMessage(bos);
	    int counter=0;
	    while (counter < dataLen || dataLen == -1) {
		FCPMessage reply = FCPMessage.readMessage(bi);
		
		if (reply.getCmd().equals("DataFound")) {
		    dataLen = reply.getInt("DataLength");
		    metaDataLen = reply.getInt("MetadataLength",0);
		    counter = 0;
		} else if (reply.getCmd().equals("RouteNotFound")) {
		    return new RouteNotFoundResult
			((int)reply.getInt("Unreachable",0),
			 (int)reply.getInt("Restarted",0),
			 (int)reply.getInt("Rejected",0));
		} else if (reply.getCmd().equals("DataNotFound")) {
		    return new DataNotFoundResult();
		} else if (reply.getCmd().equals("URIError")) {
		    return new URIErrorResult(reply.getVal("Reason"));
		} else if (reply.getCmd().equals("Restarted")) {
		    continue;
		} else if (reply.getCmd().equals("FormatError")) {
		    FIWSystem.log().println("--FormatError1--"+reply.toString());
		    return new GeneralErrorResult(reply.toString());
		} else {
		    FIWSystem.log().println("--Error1--"+reply.toString());
		    return new GeneralErrorResult(reply.toString());
		}
		fingerprint = MessageDigest.getInstance("SHA1");
		while (counter < dataLen) {
		    FCPMessage data = FCPMessage.readMessage(bi);
		    if (data.getCmd().equals("DataChunk")) {
			String chunk=data.getTrailing().toString();
			fingerprint.update(chunk.getBytes("ISO-8859-1"));
			long chunkLen = data.getInt("Length");
			counter += chunkLen;
			if (chunkLen != chunk.length()) {
			    FIWSystem.log().println("Unexpected length:" +
						    chunkLen+ " != "+
						    chunk.length());
			    return new GeneralErrorResult("---Error9---");
			}
		    } else if (data.getCmd().equals("Restarted")) {
			break;
		    } else {
			FIWSystem.log().println("--Error2--"+data.toString());
			return new GeneralErrorResult(data.toString());
		    }
		}
	    }
	    if (counter != dataLen) {
		FIWSystem.log().println("--Error5: "+counter+" != "+dataLen);
		return new GeneralErrorResult("--Error5:--");
	    }
	    s.close();
	    bos.close();
	    bi.close();
	    return new FingerprintResult
		(metaDataLen+"/"+(dataLen-metaDataLen)+"/"+
		HashUtil.bytesToHex(fingerprint.digest()));
	} catch (SocketTimeoutException ex) {
	    return new SocketTimeoutResult();
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return new GeneralErrorResult("IOException");
	} catch (java.security.NoSuchAlgorithmException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return new GeneralErrorResult("NoSuchAlgoritmException");
	}
    }

    public String deleteKey(String key) {
	try {
	    FCPMessage out = new FCPMessage("ClientGet");
	    out.put("RemoveLocalKey", "true");
	    out.put("URI", "freenet:"+key);
	    out.put("HopsToLive", "0");
	    FCPMessage reply = oneMessageEach(out);
	    return reply.headToString();
	} catch (IOException e) {
	    e.printStackTrace();
	    return null;
	}
    }


    public synchronized int getNetworkLoad() {
	FCPMessage reply;
	if (lastNodeInfo != null) {
	    reply=lastNodeInfo;
	    lastNodeInfo=null;
	} else {
	    try {
		reply = oneMessageEach(new FCPMessage("ClientInfo"));
	    } catch (IOException e) {
		return -1;
	    }
	}
	return (int) reply.getInt("EstimatedLoad");
    }

    private FCPMessage lastNodeInfo=null;
    
    public synchronized String getNodeInfo() {
	try {
	    FCPMessage reply = oneMessageEach(new FCPMessage("ClientInfo"));
	    lastNodeInfo=reply;
	    return reply.headToString();
	} catch (SocketException e) {
	    return null;
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    if (FIWSystem.doFullLog)
		e.printStackTrace(FIWSystem.getInstance().getFullLog());
	    return null;
	}
    }

    
    /**
     * this method tries to catch "connect errors" and to solve them
     * (by retrying after a few seconds)
     */
    public Socket makeSocket() throws IOException {
	Socket s = makeSocket0();
	int soTimeout = Integer.parseInt
	    (FIWSystem.setts().getProperty("tuning.sotimeout"));
	if (s != null && soTimeout != 0) {
	    s.setSoTimeout(soTimeout);
	}
	return s;
    }

    private Socket makeSocket0() throws IOException {
	for(int i=0;i<10;i++) {
	    String host = null;
	    if (hostList == null) {
		host = fcpHost;
	    } else {
		host = (String) hostList.get(rrPos);
		rrPos = ( rrPos + 1 ) % hostList.size();
	    }
	    if (host == null) {
		throw new IOException("No FCP Connection set up");
	    }
	    try {
		return new Socket (host, fcpPort);
	    } catch (ConnectException e) {
		try {
		    Thread.sleep(1000);
		} catch (InterruptedException e2) {
		    e2.printStackTrace();
		    e2.printStackTrace(FIWSystem.log());
		    throw new IOException("Interrupted");
		}
	    }
	}
	throw new IOException("Connection refused by Freenet node");
    }

    public FECConnection getFECInstance() {
	return FECUtil.getInstance(this);
    }

    public FECConnection waitForFECInstance() {
	return FECUtil.waitForInstance(this);
    }
}
