package fiw.core.option;
import fiw.core.*;

/**
 * Tries to model an option of a project that can be changed. Intended
 * to make it easier writing new user interfaces (CLI, GUI) and make
 * them easier to maintain.
 * @author mihi
 */ 
public class Option {


    private int special=0;
    private String name;
    private String prop;
    private String def, disableddef;
    protected String value;
    private int panel;
    private OptionType type;
    private Enabler enabler;
    protected boolean modified = false, loaded=false;


    public String getName() { return name;}
    public String getProperty() { return prop;}
    public String getDefault() { return def;}
    public String getValue() {return value;}
    public OptionType getType() {return type;}
    public int getPanelNumber() {return panel; }
    public int getSpecialNumber() { return special;}
    public boolean isSpecial() { return special != 0;}
    public boolean isEnabled() { return enabler.isEnabled(); }

    private static final Enabler ALWAYS_ENABLED = new Enabler() {
	    public boolean isEnabled() {
		return true;
	    }
	};

    /**
     * Sets the value of this option
     * @param value the new value
     * @return true iff the value is valid
     */
    public boolean setValue(String value) {
	if (!validateValue(value)) return false;
	modified=true;
	this.value=value;
	return true;
    }

    /**
     * Checks if a value is valid
     * @param value the value
     * @return if it is valid
     */
    public boolean validateValue(String value) {
	if (isSpecial()) return false;
	return type.isValid(value);
    }

    /**
     * Returns why a value is invalid
     * @param value the value
     * @return the reason, or <code>null</code>, if the value is valid.
     */
    public String getInvalidReason(String value) {
	if (isSpecial()) return "Special options cannot be set.";
	return type.getInvalidReason(value);
    }

    /**
     * Loads the value from the project settings
     * @param project the Project (-42 means global option)
     */
    public void load(int project) {
	if (isSpecial()) return;
	if (project == -42) {
	    value=FIWSystem.setts().getProperty(prop);
	} else {
	    value=FIWSystem.setts().getProjectSetting(project,prop);
	}
	if (value==null && isEnabled()) value=def;
	if (value==null && !isEnabled()) value=disableddef;
	modified=false;
	loaded = true;
    }

    /**
     * Saves the value into the project settings
     * @param project the Project (-42 means global option)
     */
    public void save(int project) {
	if (isSpecial()) return;
	if (!modified) return;
	if (!loaded) throw new RuntimeException
			 ("Saving an option without loading it!");
	if (project == -42) {
	    FIWSystem.setts().setProperty(prop,value);
	} else {
	    FIWSystem.setts().saveProject(project,prop, value);
	}
	modified=false;
    }
    
    
    /** Creates a new "normal" option */
    public Option(String name, String prop, String def,
		   int panel, OptionType type, Enabler en) {
	this.name=name; this.prop=prop; this.def=def;
	this.disableddef=def;
	this.panel=panel; this.type=type;
	this.enabler=en;
    }

    /** Creates a new "normal" option. */
    public Option(String name, String prop, String def,
		   int panel, OptionType type) {
	this(name,prop,def,panel,type, ALWAYS_ENABLED);
    }

    /** Creates a new "normal" option. */
    public Option(String name, String prop, String def,
		   int panel) {
	this(name,prop,def,panel, new StringOptionType());
    }

    
    /** Creates a new "special" option */
    public Option(String desc, int specialnumber, int panel, Enabler e) {
	this.name=desc;
	this.special=specialnumber;
	this.panel=panel;
	this.enabler = e;
    }

    /** Creates a new "special" option */
    public Option(String desc, int specialnumber, int panel) {
	this(desc,specialnumber, panel, ALWAYS_ENABLED);
    }

    public void setDisabledDefault(String disdef){
	this.disableddef=disdef;
    }
	
    public String toString() {
	return prop+"="+value;
    }
}
    
