package fiw.core.option;
import fiw.core.*;

/**
 * Class containing all options that are independent from projects,
 * for example FCP settings and tuning variables.
 * @author mihi
 */
public class GlobalOptions extends AbstractOptions { 

    /** all available settings with values */
    private Option[] settings;

    private void initSettings() {
      settings = new Option[] {
	new Option(getI18n("OptionHost"),"fcp.host","127.0.0.1",0),
	new Option(getI18n("OptionPort"),"fcp.port","8481",0,
		   new IntOptionType(0,65535)),
	new Option(getI18n("OptionThreads"), "tuning.threadcount","20",1,
		   new IntOptionType(1,1000)),
	new Option(getI18n("OptionFECMin"), "tuning.fecminsize","1048576",1,
		   new IntOptionType(1024)),
	new Option(getI18n("OptionNoFECFor"), "tuning.nofecexts",
		   "txt htm html",1),	  
	new Option(getI18n("OptionTestVerify"),
		   "tuning.doverifytest","true",1, BOOL),	  
	new Option(getI18n("OptionVerifyFlags"), "tuning.verifyflags",
		   "11111111",-1),
	new Option(getI18n("OptionTestNoFEC"),
		   "tuning.testnofec","true",1, BOOL),
	new Option(getI18n("OptionTestModURL"),
		   "tuning.modifyurl","true",1, BOOL),		  
	new Option(getI18n("OptionSocketTimeout"),
		   "tuning.sotimeout","300000",1, new IntOptionType(0)),
	new Option(getI18n("OptionSHA1"), "tuning.newhash",
		   "false", 1, BOOL),
	new Option(getI18n("OptionDefPriority"), "tuning.priority","3", 1,
		   new IntOptionType(0)),
	new Option(getI18n("OptionThreadWait"), "tuning.threadwait","60",
		   1, new IntOptionType(0)),
	new Option("Geeky settings",1,1), // special
	new Option(getI18n("OptionVerify"),
		   "tuning.doverify","true",1, BOOL),	  
	new Option(getI18n("OptionInsertRetry"),
		   "tuning.inserttry","10",1, new IntOptionType(1,50)),
	new Option(getI18n("OptionFetchRetry"), "tuning.fetchtry","5",1,
		   new IntOptionType(1,50)),
	new Option(getI18n("OptionHTLInc"), "tuning.htlinc","8",1,
		   new IntOptionType(0,20)),
	new Option(getI18n("OptionRememberInserts"),
		   "tuning.checkprogressfile","true",1, BOOL),	  
	new Option(getI18n("OptionIgnoreLocalInsert"),
		   "tuning.ignorelocal.insert","true",1, BOOL),  
	new Option(getI18n("OptionIgnoreLocalFetch"),
		   "tuning.ignorelocal.fetch","true",1, BOOL),	  
	new Option(getI18n("OptionModifyTests"),
		   "tuning.modifytests","false",1, BOOL),
	new Option(getI18n("OptionNIMAdvance"), "nimgetadvance", "5", 10,
		   new IntOptionType(1)),
	new Option(getI18n("OptionNIMHTL"), "nimgethtl", "20",10,
		   new IntOptionType(0,100)),
	// options not available via GUI:
	new Option("Maximal size for one-key NIMs",
		   "tuning.maxnimsize","8192",-1),
	new Option("Number of projects", "proj.count","0",-1),
	new Option("Allow inserting into broken DBRs [No]",
		   "tuning.nodbrmismatch","false",-1, BOOL),
	new Option("FProxy URL","fproxy.url","http://127.0.0.1:8888/",-1),
	new Option("Locale","locale","",-1),
      };
    }

    /**
     * Returns all settings.
     */
    public Option[] getSettings() {
	return settings;
    }
    
    /** if the settings' values have been loaded from the config file */
    private boolean loaded = false;

    /**
     * Creates a new GlobalOptions object that loads all options from
     * the config file
     */
    public GlobalOptions() { this(true); }

    /**
     * Creates a new GlobalOptions object.
     * @param load boolean whether the options should be loaded from
     * the config file.
     */
    public GlobalOptions(boolean load) {
	this(load, true);
    }

    /**
     * Creates a new GlobalOptions object.
     * @param load whether the options should be loaded from the
     * config file.
     * @param useI18n whether to internationalize option
     * descriptions. This is a problem since the I18n class needs a
     * GlobalOptions object to determine which locale to use. This
     * would lead into an endless recursion.
     */
    public GlobalOptions(boolean load, boolean useI18n) {
	this.useI18n=useI18n;
	initSettings();
	if (load) load();    
    }

    private boolean useI18n;
    
    private String getI18n(String name) {
	if (useI18n)
	    return I18n.get(name);
	else
	    return "!No I18n available!";
    }

    
    /**
     * Loads all options from the config file.
     */
    public void load() {
	for (int i=0;i<settings.length;i++) {
	    settings[i].load(-42);
	}
	loaded=true;
    }

    /**
     * saves all modified options to the config file.
     */
    public void save() {
	if (!loaded) throw new RuntimeException
			 ("You must load before saving...");
	for (int i=0;i<settings.length;i++) {
	    settings[i].save(-42);
	}
    }
}
