package fiw.core.jobs;

import fiw.core.insert.*;
import fiw.core.insert.event.*;
import fiw.fcp.*;

/**
 * A job that tries to retrieve some data and metadata from a Freenet key
 */
public class RetrieveCheckJob extends Job {

    private String key, progressName;
    private String fingerprint;
    private int htl, fixNumber=-1;
    //   private InsertLogger logger;
    private InsertContext ic;
    private FCPConnection fc;

    /**
     * Creates a new retrieve check job.
     * @param key the key to retrieve
     * @param fingerprint the fingerprint of the data to retrieve
     * @param name a name for the data
     * @param progressName the name in the progress file
     * @param ic the insert context
     * @param dependencies the jobs this job depends on
     */
    public RetrieveCheckJob(String key,
			    String fingerprint,
			    String name,
			    String progressName,
			    InsertContext ic,
			    Job[] dependencies) {
	super(true, dependencies, name);
	this.key=key;
	this.htl=ic.fetchHTL;
	this.fingerprint=fingerprint;
	this.progressName=progressName;
	this.ic=ic;
	fc=ic.getFCPConnection();
    }

    /**
     * Returns the key to be retrieved.
     */
    public String getKey() { return key;}

    /**
     * Runs this job from outside a job scheduler. This is done from
     * an {@link InsertJob} when verifying is on.
     */
    public int foreignRun(int number, int fixNumber) {
	myNumber=number;
	this.fixNumber=fixNumber;
	return runInternal();
    }

    /**
     * Checks if this job should be run.
     */
    public boolean shouldRun() {
	Job[] deps = getDependencies();
	Job dep = null;
	if (deps != null && deps.length >0)
	    dep = deps[0];
	if (dep instanceof InsertJob) {
	    if (key == null) { // fetch key if needed
		key = ((InsertJob)dep).getResultKey();
	    }
	    if (fingerprint == null) {
		fingerprint = ((InsertJob)dep).getFingerprint();
	    }
	    if (progressName == null) {
		progressName= ((InsertJob)dep).getProgressName();
	    }
	}
	if ("!none!".equals(key)) return false;
	return super.shouldRun();
    }

    /**
     * Runs the job.
     */
    public int run() {
	ic.fireInsertEvent(new JobInitInsertEvent(myNumber));
	ic.fireInsertEvent(new JobChangeInsertEvent(myNumber, name,
						    ic.ACTIONS[ic.AC_START]));
	ic.fireInsertEvent(new JobGettingInsertEvent(myNumber, name,
						     -1));
	int result = runInternal();
	if (result != 0) {
	    if (progressName != null) {
		ic.dataNotInserted(progressName);
	    }
	    ic.fireInsertEvent(new JobFailureEvent(myNumber, name, null));
	    if (result == 3) {
		ic.setRetryFutile();
	    }
	} else {
	    ic.fireInsertEvent(new JobSuccessEvent(myNumber, name));
	}
	return result;
    }

    private int runInternal() {
	if (key == null)
	    throw new RuntimeException("assert: key not null");
	if (fingerprint == null)
	    throw new RuntimeException("assert: fingerprint not null");
	boolean exitDNF=false;
        for(int j=0;j<ic.RETRIEVERETRY;j++) {
	    int jj=(fixNumber==-1?j:fixNumber); // Number for thread status
	    String j2 = fixNumber==-1?"":"("+ic.num(j+1)+") ";
	    if (ic.shouldStop()) return 1;
            ic.addlog(myNumber,jj,ic.AC_GETTING,j2+name);
            FCPGetResult fetch=fc.fetchFingerprint(key,htl+(exitDNF?1:0),
						   false);
            if (fetch instanceof FingerprintResult) {
		ic.notAllUnreachable();
                ic.addlog(myNumber,jj,ic.AC_GOT,name);
		if(fingerprint.equals("-")) {
		    return 0;
		} else if (fetch.getPayload().equals(fingerprint)) {
		    ic.addlog(myNumber,jj,ic.AC_IDENT,name);
		    return 0;		    
		} else {
		    ic.addlog(myNumber,jj,ic.AC_DIFF,name);
		    ic.addlog("      GOT:    "+fetch.getPayload());
		    ic.addlog("      SHOULD: "+fingerprint);
		    break;
		}
            } else if (fetch instanceof RouteNotFoundResult) {
		RouteNotFoundResult rnfr=(RouteNotFoundResult)fetch;
                ic.addlog(myNumber,jj,ic.AC_RNF
			       ,name+ " "+rnfr.getReasonString());
		if (rnfr.isAllUnreachable()) {
		    if (ic.allUnreachable()) break;
		} else {
		    ic.notAllUnreachable();
		}
	    } else if (fetch instanceof SocketTimeoutResult) {
		ic.fireInsertEvent(new StructuredJobInsertEvent
				   (myNumber, jj, "***Timeout ", name));
		ic.notAllUnreachable();		
            } else if (fetch instanceof DataNotFoundResult) {
		ic.notAllUnreachable();
                ic.addlog(myNumber,jj,ic.AC_DNF,name);
		if (exitDNF || htl==0) break;
                exitDNF=true;
	    } else if (fetch instanceof URIErrorResult) {
		URIErrorResult uer = (URIErrorResult)fetch;
		ic.notAllUnreachable();
		if (uer.getReason()
		    .equals("Data found, but failed to decrypt with "+
			    "the decryption key given in the URI.")) {
		    ic.addlog("#"+myNumber+
			      ": [Error] Key already exists with different "+
			      "crypto key! Please insert into another key.");
		} else {
		    ic.addlog("#"+myNumber+
			      "[Error] URI error while fetching.");
		    ic.addlog("Reason: "+uer.getReason());
		}
		return 3;
            } else {
		ic.addlog("[ERROR] #"+ic.num(myNumber)+
		       ": internal error - consult logfile");
            }
        }
	return 2;
    }
}
