package fiw.core.jobs;

import java.util.*;

/**
 * A ThreadProducer that does not produce new threads, but reuses them
 * in a thread pool. When there is no idle thread left, a new one is created. When a thread is idle for some time, it is destroyed.
 * @author mihi
 */
public class PooledThreadProducer implements ThreadProducer {

    private long keepDelay;
    private volatile boolean stopAll=false;
    private List freeThreads;
    private Object syncObject;

    /**
     * Creates a new pooled thread producer.
     * @param keepDelay number of milliseconds a thread may be idle
     * before it is destroyed.
     */
    public PooledThreadProducer(long keepDelay) {
	this.keepDelay=keepDelay;
	this.freeThreads = new ArrayList();
	this.syncObject=new Object();
    }

    /**
     * Initializes the thread producer.
     */
    public void initialize() {
	stopAll = false;
    }

    /**
     * Checks if the pool has an idle thread and used that one if
     * any. Otherwise it creates a new thread.
     * @param j the job to run
     */
    public void produceThread(Job j) {
	synchronized(syncObject) {
	    if (freeThreads.isEmpty()) {
		PooledThread t = new PooledThread();
		t.setJob(j);
		t.start();
	    } else {
		PooledThread t = (PooledThread) freeThreads.get(0);
		synchronized(t) {
		    t.setJob(j);
		    freeThreads.remove(t);
		    t.notifyAll();
		}
	    }
	}
    }

    /**
     * Destroys all threads in the pool.
     */
    public void finish() {
	synchronized(syncObject) {
	    stopAll=true;
	    // beware of ConcurrentModificationExceptions; so use
	    // a copy to iterate on.
	    for (Iterator it = new ArrayList(freeThreads).iterator();
		 it.hasNext();) {
		PooledThread t = (PooledThread) it.next();
		synchronized(t) {
		    t.notifyAll();
		}
	    }
	}
    }

    private class PooledThread extends Thread {
	private Job jobToRun = null;

	public synchronized void setJob(Job j) {
	    if (jobToRun != null)
		throw new IllegalStateException("There is already a job.");
	    jobToRun=j;
	}
	
	public void run() {
	    synchronized(this) {
		if (jobToRun==null)
		    throw new IllegalStateException ("Nothing to run.");
	    }
	    try {
		while(true) {
		    Job j;
		    synchronized(this) {
			if (jobToRun == null) break;
			j=jobToRun;
			jobToRun=null;
		    }
		    j.run0();
		    if (stopAll) break;
		    long now = System.currentTimeMillis();
		    synchronized(syncObject) {
			freeThreads.add(this);
		    }
		    synchronized(this) {
			while(jobToRun==null && !stopAll) {
			    if (keepDelay==0) {
				wait();
			    } else {
				wait(keepDelay);
				if (System.currentTimeMillis()-now<keepDelay)
				    break;
			    }
			}
		    }
		    synchronized(syncObject) {
			freeThreads.remove(this);
		    }
		}
	    } catch (InterruptedException ex) {
		ex.printStackTrace();
	    }
	}
    }
}
