package fiw.core.jobs;


/**
 * Responsible for starting {@link Job}s in correct order.
 */
public interface JobScheduler {


    /**
     * Called by a job to notify the scheduler that is has finished.
     * @param j the calling job
     */
    public void notifyFinished(Job j);

    /**
     * Called by a job to notify the scheduler that the job is not
     * run. The job has to call {@link #notifyFinished} nevertheless.
     * @param j the calling job
     */
    public void notifySkipping(Job j);

    /**
     * Called by a job to notify the scheduler that an error
     * occurred. The job has to call {@link #notifyFinished}
     * nevertheless.
     * @param stopAll if the error should stop the whole insertion process.
     */
    public void notifyError(boolean stopAll);

    /**
     * Called by a job to request being run as a thread. Usually a job
     * scheduler delegates this method to a thread producer.
     * @param j the calling job
     */
    public void runAsThread(Job j);

    /**
     * Informs the job scheduler that it should stop all jobs as soon
     * as possible.
     */
    public void doStop();

    /**
     * Starts the job scheduler in a new thread.
     */
    public void start();

    /**
     * Starts the job scheduler and waits until it has finished.
     */
    public void startAndWait();

    /**
     * Checks if all jobs have been scheduled.
     * @return true, iff all jobs have been scheduled
     */
    public  boolean hasFinished();

    /**
     * Waits until the job scheduler has finished.
     */
    public void waitForFinished();

    /**
     * Checks if a problem occured during run.
     * @return true, iff the run was successful.
     */
    public  boolean wasSuccessful();
}
