package fiw.core.jobs;


/**
 * A separate part of a process that can be scheduled separately. Each
 * job can depend on other jobs (which must be executed first) and can
 * stop other jobs depending on it to be run. Jobs can be threaded
 * (executed in its own thread) or not (executed in the JobScheduler's
 * thread). A {@link JobScheduler} is used to schedule jobs.
 * @author mihi
 */
public abstract class Job {
    private boolean threaded;
    private Job[] dependencies;
    private boolean finished=false;
    private boolean successful=true;
    private JobScheduler scheduler;

    /**
     * The number assigned to the job by the job scheduler
     */
    protected int myNumber;

    /**
     * The name of the job
     */
    protected String name;

    /**
     * Creates a new job.
     * @param threaded whether the job is run in an own thread
     * @param dependencies jobs this job depends on
     * @param name a name for the job
     */
    public Job (boolean threaded, Job[] dependencies, String name) {
	this.threaded = threaded;
	this.dependencies = dependencies;
	this.name=name;
    }

    /**
     * Returns the dependencies of this job.
     */
    protected Job[] getDependencies() {
	return dependencies;
    }

    /**
     * returns the only dependency of this Job, or <code>null</code>
     * if this job has no or more than one dependency.
     */
    protected Job getDependency() {
	if (dependencies != null && dependencies.length == 1) {
	    return dependencies[0];
	} else {
	    return null;
	}
    }

    /**
     * Returns if this job is threaded.
     */
    public boolean isThreaded() { return threaded;}

    /**
     * Checks if this job has been finished.
     */
    public synchronized boolean hasFinished() { return finished;}

    /**
     * Checks if this job's execution was successful.
     */
    public synchronized boolean wasSuccessful() { return successful;}

    /**
     * Checks if this job is ready to run. A job is ready to run if it
     * has not been finished and all of its dependencies have been
     * finished.
     */
    public boolean isReadyToRun() {
	if (finished) return false;
	for (int i=0;i<dependencies.length;i++) { //check dependencies
	    if (!dependencies[i].hasFinished()) {
		return false;
	    }
	}
	return true;
    }

    private boolean shouldrun;

    /**
     * Starts this job
     * @param sched the JobScheduler that starts this thread
     * @param myNumber the internal job number
     */
    public void start(JobScheduler sched, int myNumber) {
	if (finished) {
	    throw new IllegalStateException("Job already finished");
	}
	this.myNumber=myNumber;
	this.scheduler=sched;
	for (int i=0;i<dependencies.length;i++) { //check dependencies
	    if (!dependencies[i].hasFinished()) {
		throw new IllegalStateException("Dependency error: "+this+
						" depends on "+
						dependencies[i]);
	    }
	    if (!dependencies[i].wasSuccessful()) {
		sched.notifySkipping(this);
		finished=true;
		successful = false;
		sched.notifyFinished(this);
		return;
	    }
	}
	shouldrun = shouldRun();
	if (shouldrun && isThreaded()) {
	    sched.runAsThread(this);
	} else {
	    run0();
	}
    }

    /**
     * Internal run method.
     */
    public final void run0() {
	int s;
	if (shouldrun) {
	    s = run();
	} else {
	    s=0;
	}
	synchronized(this) {
	    finished=true;
	    switch (s) {
	    case 0: // properly run		
		successful=true;
		break;
	    case 1: // stop depending tasks
		successful=false;
		break;
	    case 2: // error & stop depending tasks
		successful= false;
		scheduler.notifyError(false);
		break;
	    case 3: // error & stop all tasks
		successful=false;
		scheduler.notifyError(true);
	    }
	}
	scheduler.notifyFinished(this);
    }

    /**
     * Returns the name of this job.
     */
    public String toString() {
	return name;
    }

    /**
     * Does the actual job.
     * @return 0, iff everything was OK;
     *         1, iff depending tasks should not be run
     *         2, iff an error occurs and depending tasks should not be run
     *         3, iff an error occurs and no tasks should be run
     */
    public abstract int run();

    /**
     * Determines if the job should run at all. If this method returns
     * <code>false</code>, this job is treated as run successfully
     * without being run at all. This allows skipping data without
     * needing to start a new thread for each of them first.  As this
     * method is run in the main Scheduler thread, it should return
     * quickly. The default implementation just returns <code>true</code>.
     */
    public boolean shouldRun() {
	return true;
    }    
}
	
