package fiw.core.jobs;

import fiw.core.*;
import fiw.core.insert.*;
import fiw.core.insert.event.*;
import fiw.fcp.*;

/**
 * A job that inserts some data and metadata into a Freenet key - or
 * just creates the CHK for it.
 * @author mihi
 */
public class InsertJob extends Job {

    private String key,progressName, metadata;
    private String fingerprint = null, keycollisionwarning = null;
    private InputStreamBuilder data;
    private long dataLength;
    private InsertContext ic;
    private FCPConnection fc;
    private boolean keyCollision=false, keyCollisionOkay=false;
    private KeyFallback fallback;
    
    /**
     * The resulting public key for this insert.
     */
    protected String resultKey;
    
    /**
     * The number of this thread for the logger.
     */
    protected boolean loggerStarted = false;

    /**
     * whether to only calculate the key.
     */
    protected boolean calcOnly = false;

    /**
     * A KeyFallback that does not fallback at all.
     */
    private static KeyFallback defaultFallback = new NoKeyFallback();

    /**
     * Creates a new insert job.
     * @param key the key to insert into 
     * @param metadata the metadata for this key
     * @param data an input stream builder to get the data from
     * @param dataLength the length of the data
     * @param name the name of the inserted thing
     * @param progressName the name to use in the progress file
     * @param ic the insert context
     * @param dependencies jobs this job depends on
     */
    public InsertJob(String key,
		     String metadata,
		     InputStreamBuilder data,
		     long dataLength, String name, String progressName,
		     InsertContext ic,
		     Job[] dependencies) {
	super(true, dependencies, name);
	this.key=key;
	this.metadata=metadata;
	this.data=data;
	this.dataLength = dataLength;
	this.progressName=progressName;
	this.ic=ic;
	fc=ic.getFCPConnection();
	this.fallback=defaultFallback;
    }

    /**
     * Returns the thread number for this insert from the logger.
     */
    protected void getLoggerNumber() { // FIXME: kill that method
	if (!loggerStarted) {
	    ic.fireInsertEvent(new JobInitInsertEvent(myNumber));
	    loggerStarted=true;
	}
    }

    /**
     * Sets a key fallback.
     */
    public void setFallback(KeyFallback f) {
	fallback = f;
    }

    /**
     * Sets this job to only calculate the CHK, not do an actual insert.
     */
    public void setCalcOnly() {
	if (!"CHK@".equals(key)) {
	    throw new RuntimeException("assert: calcCHK only for CHKs!");
	}
	calcOnly = true;
    }

    /**
     * Specifies that key collisions are no errors on that (non-CHK)
     * key.
     */
    public void setKeyCollisionOkay() {
	keyCollisionOkay=true;
    }

    /**
     * Sets the warning that should be shown in case of a key
     * collision.
     */
    public void setKeyCollisionWarning(String warning) {
	keycollisionwarning = warning;
    }

    /**
     * Sets the metadata.
     * @param metadata the new metadata
     */
    public void setMetadata(String metadata) {
	this.metadata=metadata;
    }

    /**
     * Sets the progress name.
     */
    public void setProgressName(String progressName) {
	this.progressName=progressName;
    }

    /**
     * Gets the progress name.
     */
    public String getProgressName() {
	return progressName;
    }

    /**
     * Sets the data.
     */
    public void setData(InputStreamBuilder data, long dataLength) {
	this.data=data;
	this.dataLength=dataLength;
    }

    /**
     * Gets the "fingerprint" (hash of metadata and data) of that key.
     */
    public String getFingerprint() {
	return fingerprint;
    }

    /**
     * Gets the resulting public key of this insert.
     */
    public String getResultKey() {
	return resultKey;
    }

    /**
     * Checks whether there was a key collision.
     */
    public boolean isKeyCollision() {
	return keyCollision;
    }

    /**
     * Checks whether this file should be inserted.
     */
    public boolean shouldRun() {
	if (metadata==null)
	    throw new RuntimeException("assert: metadata for insert present");
	if (data == null)
	    throw new RuntimeException("assert: data for insert present");
	if (progressName == null)
	    throw new RuntimeException
		("assert: proressName for insert present");
	if (calcOnly) {
	    progressName="Calc."+progressName;
	} else {
	    // build fingerprint
	    fingerprint = HashUtil.makeFingerprint(metadata,
						   data,
						   dataLength);
	    if (fingerprint.equals("")) {
		ic.addlog("[Warning] Could not generate fingerprint of "+
			  this);
	    }
	}
	getLoggerNumber();
	String res;
	if ((res=ic.alreadyDone(progressName)) != null) {
	    resultKey = res;
	    if (keyCollisionOkay) {
		fingerprint="-"; // no check on verify
	    }
	    ic.fireInsertEvent(new JobSkippedInsertEvent(myNumber, name));
	    return false;
	}
	ic.fireInsertEvent(new JobChangeInsertEvent
			   (myNumber, name, ic.ACTIONS[ic.AC_START]));
	return true;
    }

    /**
     * Runs the job.
     */
    public int run() {
	if (metadata==null || data == null || progressName == null)
	    throw new RuntimeException("assert: data for insert present");
	getLoggerNumber();
	if (calcOnly) {
	    ic.fireInsertEvent(new JobChangeInsertEvent
			       (myNumber,name,ic.ACTIONS[ic.AC_CALCCHK]));
	    resultKey = fc.calcCHK(metadata, data, dataLength, name);
	    if (resultKey == null) {

		ic.fireInsertEvent
		    (new JobFailureEvent
		     (myNumber, name,
		      "[Error] unexpected error - consult logfile"));
		return 2;
	    }
	    ic.dataInserted(progressName, resultKey);
	    ic.fireInsertEvent(new JobFinishedEvent(myNumber, name));
	    return 0;
	}
	// insert
	for (int i=0;i<ic.INSERTRETRY;i++) {
	    if (ic.shouldStop() || ic.isAllUnreachable()) break;
	    ic.fireInsertEvent(new JobUppingInsertEvent(myNumber, name, i));
	    FCPInsertResult res=fc.insertStream(key, metadata, data,
						ic.htl,
						dataLength,"INSERT("+name+")");
	    if (res==null) {
		ic.addlog("[ERROR] #"+ic.num(myNumber)+ ": "+name+
			  ": internal error - consult logfile");
		resultKey=null;
		continue;
	    } else if (res instanceof RouteNotFoundResult) {
		RouteNotFoundResult rnfr = (RouteNotFoundResult) res;
		ic.addlog(myNumber,i,ic.AC_RNF,
			  name+" "+rnfr.getReasonString());
		if (rnfr.isAllUnreachable()) {
		    if (ic.allUnreachable()) {
			break;
		    }
		} else {
		    ic.notAllUnreachable();
		}
		resultKey=null;
		continue;
	    } else if (res instanceof SocketTimeoutResult) {
		ic.fireInsertEvent(new StructuredJobInsertEvent
				   (myNumber, i, "***Timeout ", name));
		ic.notAllUnreachable();
		resultKey=null;
		continue;
	    } else if (res instanceof KeyCollisionResult) {
		ic.notAllUnreachable();
		ic.fireInsertEvent(new UploadedJobInsertEvent
				   (myNumber, i, true, name, res.getKey()));
		if (!key.equals("CHK@")) {
		    if (ic.shouldStop()) break;
		    ic.addlog(myNumber,i,ic.AC_EXAM,name);
		    FCPGetResult comp = fc.fetchFingerprint(res.getKey(),
							    ic.fetchHTL,
							    true);
		    if (comp == null ||
			!(comp instanceof FingerprintResult)) {
			ic.addlog("[Error] could not get colliding key!");
		    } else if (comp.getPayload().equals(fingerprint)) {
			ic.addlog(myNumber,i,ic.AC_IDENT,name);
		    } else {
			ic.addlog(myNumber,i,ic.AC_DIFF,name);
			if (keyCollisionOkay) {
			    fingerprint="-"; // no check on verify
			    keyCollision=true;
			    ic.addlog("[Warning] Key collision on "+
				      "non-CHK file!");
			} else {
			    String nkey = fallback.nextKey(key);
			    if (nkey == null) {
				keyCollision=true;
				ic.fireInsertEvent
				    (new JobFailureEvent
				     (myNumber, name,
				      "[Error] Key collision on non-CHK file!"
				      ));
				if (keycollisionwarning != null) {
				    ic.addlog(keycollisionwarning);
				}
				ic.setRetryFutile();
				return 3;
			    } else {
				key=nkey;
				ic.addlog("#"+ic.num(myNumber)+
					  " ** now trying: "+key);
				i=-1;
				continue;
			    }
			}
		    }
		}		
	    } else if (res instanceof InsertSuccessResult) {
		ic.notAllUnreachable();
		ic.fireInsertEvent(new UploadedJobInsertEvent
				   (myNumber, i, false, name, res.getKey()));
	    } else {
		throw new RuntimeException("Unknown result: "+res.getClass());
	    }
	    resultKey=res.getKey();
	    if (!keyCollision) {
		ic.dataInserted(progressName, resultKey);
	    }
	    if (!(ic.doVerify(false))) break;
	    ic.fireInsertEvent(new JobGettingInsertEvent(myNumber, name, i));
	    RetrieveCheckJob r = buildChecker(false, resultKey, null);
	    int cfa =  r.foreignRun(myNumber, i);
	    if (cfa == 0) break;
	    if (cfa == 3) {
		ic.fireInsertEvent(new JobFailureEvent(myNumber, name, null));
		ic.setRetryFutile();
		return 3;
	    }
	    resultKey = null;
	    ic.dataNotInserted(progressName);
	}
	if (ic.shouldStop()) {
	    resultKey=null;
	    ic.fireInsertEvent(new JobAbortEvent(myNumber, name));
	    return 1;
	} else if (resultKey == null || ic.isAllUnreachable()) {
	    ic.fireInsertEvent(new JobFailureEvent(myNumber, name, null));
	    return 2;
	} else {
	    ic.fireInsertEvent(new JobSuccessEvent(myNumber, name));
	    return 0;
	}
    }

    /**
     * Builds a check job that verifies this job's insert.
     * @param depending whether the new job should depend on this job
     * @param key the key where the file is inserted to (if the job is
     * not depending)
     * @param extraJob another job the new job depends on when depending.
     */
    public RetrieveCheckJob buildChecker(boolean depending,
					 String key,
					 Job extraJob) {
	if (calcOnly) {
	    throw new RuntimeException ("assert: cannot verify chk "+
					"calculation of "+this);
	}
	Job[] deps = (depending)?new Job[] {this, extraJob} : new Job[0];
	return new RetrieveCheckJob(key, fingerprint,
				    name, progressName, ic, deps);
    }
}
