package fiw.core.jobs;
import fiw.core.insert.*;
import fiw.core.*;

import java.io.*;

/**
 * Inserts a part of a file (e.g. a FEC chunk) into Freenet.
 * @author mihi
 */
public class FilePartInsertJob extends InsertJob
    implements InputStreamBuilder {

    private File f;
    private boolean checkChunk;
    private String filename, hash = null;
    private long start, length;
    private int number;
    private InsertContext ic;

    /**
     * Creates a new FilePartInsertJob.
     * @param f the file to insert a part of
     * @param filename the name of that file (in Freenet)
     * @param start the start offset in the file
     * @param length the length of the chunk
     * @param number the number of the chunk
     * @param checkChunk wheher it is a check chunk
     * @param ic the insert context
     * @param deps the jobs this job depends on
     */
    public FilePartInsertJob(File f, String filename, long start, long length,
			     int number, boolean checkChunk,
			     InsertContext ic, Job[] deps) {
	super("CHK@", "",null, 0, filename+" (Ch"+(checkChunk?"eck ch":"")+
	      "unk "+ number+")", null, ic,deps);
	this.f=f;
	this.start=start;
	this.length=length;
	this.checkChunk=checkChunk;
	this.filename=filename;
	this.ic=ic;
	this.number=number;
    }

    /**
     * Creates a copy of a FilePartInsertJob
     * @param toClone the job to copy
     */
    public FilePartInsertJob(FilePartInsertJob toClone) {
	this(toClone.f, toClone.filename, toClone.start, toClone.length,
	     toClone.number, toClone.checkChunk, toClone.ic,
	     toClone.getDependencies());
    }

    /**
     * Checks whether this chunk must be inserted.
     */
    public boolean shouldRun() {
	if (f==null) {
	    Job dep = getDependency();
	    if (dep instanceof FileProducing) {
		f = ((FileProducing)dep).produceFile();
	    }
	}
	if (hash == null) {
	    Job dep = getDependency();
	    if (dep instanceof HashProducing) {
		hash = ((HashProducing)dep).produceHash();
	    }
	}
	setData(this, length);
	if (hash != null) { 
	    setProgressName((checkChunk?"Check":"")+"Chunk."+hash+"."+start);
	}
	return super.shouldRun();
    }

    /**
     * Builds a buffered input stream for that chunk.
     * @return this stream
     */
    public InputStream buildStream() throws IOException {
	return new BufferedInputStream
	    (new PaddingInputStream(new FileInputStream(f), start, length));
    }
}
