package fiw.core.jobs;
import fiw.core.insert.*;
import fiw.core.*;

import java.io.*;


/**
 * This job inserts a file into Freenet.
 * @author mihi
 */
public class FileInsertJob extends InsertJob implements MetadataProducingEx,
							InputStreamBuilder {


    private File f;
    private boolean isDefault;
    private String ctype, filename, hash;

    /**
     * Creates a new file insert job that inserts into a CHK.
     * @param f the file to insert
     * @param filename the name of that file (in Freenet)
     * @param ctype the content type of that file (may be null)
     * @param hash the MD5 hash of that file
     * @param isDefault whether it is the default file for a freesite
     * (index.html)
     * @param modifyMetadata whether the metadata should get a "Test
     * insert" comment
     * @param ic the insert context
     * @param deps the jobs this job depends on
     */
    public FileInsertJob(File f, String filename, String ctype, String hash,
			 boolean isDefault,
			 boolean modifyMetadata,
			 InsertContext ic, Job[] deps) {
	this(f, filename,ctype, hash, isDefault, "CHK@",
	     "Version\nRevision=1\nEnd\n"+
	     (modifyMetadata?"Comment=FIW_Test\n":""), ic, deps);
    }

    /**
     * Creates a new file insert job.
     * @param f the file to insert
     * @param filename the name of that file (in Freenet)
     * @param ctype the content type of that file (may be null)
     * @param hash the MD5 hash of that file
     * @param isDefault whether it is the default file for a freesite
     * (index.html)
     * @param key the key where to insert the file to
     * @param metadata the metadata to use
     * @param ic the insert context
     * @param deps the jobs this job depends on
     */
    public FileInsertJob(File f,String filename, String ctype, String hash,
			 boolean isDefault,
			 String key, String metadata,
			 InsertContext ic, Job[] deps) {
	super(key, metadata,null, 0, filename, null,
	      ic,deps);
	this.f=f;
	this.isDefault=isDefault;
	this.ctype=ctype;
	this.filename=filename;
	this.hash=hash;
    }

    /**
     * Checks whether this file must be inserted.
     */
    public boolean shouldRun() {
	if (f==null) {
	    Job dep = getDependency();
	    if (dep instanceof FileProducing) {
		f = ((FileProducing)dep).produceFile();
	    }
	}
	if (hash == null) {
	    Job dep = getDependency();
	    if (dep instanceof HashProducing) {
		hash = ((HashProducing)dep).produceHash();
	    }
	}
	setData(this, f.length());
	if (hash != null) { 
	    setProgressName("File."+hash);
	}
	return super.shouldRun();
    }

    /**
     * Builds a buffered FileInputStream for this file.
     * @return the stream
     */
    public InputStream buildStream() throws IOException {
	return new BufferedInputStream(new FileInputStream(f));
    }
	      
	      
    /**
     * Produces the metadata to redirect to this file.
     * @return the metadata
     */
    public String produceMetadata () {
	if (!isDefault) {
	    return "";
	}
	return "EndPart\nDocument\n"+
            "Redirect.Target=freenet:"+getResultKey()+
	    (ctype==null?"":"\nInfo.Format="+ctype)+"\n";
    }

    /**
     * Produces the metadata used in a mapfile to redirect to this file.
     * @return the metadata
     */
    public String produceMetadataEx() {
	return "EndPart\nDocument\nRedirect.Target=freenet:"+
	    getResultKey()+"\nName="+
	    filename+(ctype==null?"":"\nInfo.Format="+ctype)+"\n";	
    }
}
