package fiw.core.jobs;

import fiw.core.insert.*;
import fiw.fcp.*;

import java.io.*;
import java.util.*;

/**
 * A Job that builds FEC chunks.
 * @author mihi
 */
public class FECBuilderJob extends Job implements FileProducing,
						  HashProducing {

    private InsertContext ic;
    private FCPConnection fc;
    private File file, fecDataDir, checkfile = null;
    private String fecToken, filename;
    private String sha1 = null, hash=null;
    private FECMetrics fecMetrics;
    private Properties fecProgress;

    /**
     * Creates a new FEC builder job.
     * @param file the file to FEC
     * @param filename the name of the file
     * @param fecProgress the progress stats to update
     * @param fecToken the token to mark current FEC processes
     * @param fecDataDir the dir to place the FEC chunks in
     * @param fecMetrics sizes of the chunks
     * @param ic the insert context
     * @param dependencies the jobs this job depends on
     */
    public FECBuilderJob(File file, String filename, Properties fecProgress,
			 String fecToken, File fecDataDir,
			 FECMetrics fecMetrics,
			 InsertContext ic, Job[] dependencies) {
	super(true, dependencies, "FEC encoding "+filename);
	this.file=file;
	this.filename=filename;
	this.fecProgress=fecProgress;
	this.fecToken=fecToken;
	this.fecDataDir = fecDataDir;
	this.fecMetrics = fecMetrics;
	this.ic=ic;
	fc=ic.getFCPConnection();
    }

    /**
     * Checks if this file needs to be FECced.
     */
    public boolean shouldRun() {
	HashCalcJobEx dep = (HashCalcJobEx)getDependency();
	hash = dep.produceHash();
	sha1 = dep.getSHA1();
	checkfile = new File(fecDataDir,hash);
	String dest = fecProgress.getProperty(hash);
	if (dest !=null) {
	    ic.addlog("#"+ic.num(myNumber)+
		      " FEC blocks for "+filename+" up to date.");
	    fecProgress.setProperty(hash, fecToken);
	    return false;
	} else {
	    ic.addlog("#"+ic.num(myNumber)+
		      " Starting FEC encoding for "+filename);
	    return true;
	}
    }

    /**
     * Runs the job.
     */
    public int run() {
	FECConnection encoder=fc.getFECInstance();
	try {
	    if (encoder==null) {
		ic.addlog("#"+ic.num(myNumber)+
			  " Waiting until FEC encoder is free..");
		encoder=fc.waitForFECInstance();
	    }
	    ic.addlog("#"+ic.num(myNumber)+
		      " Generating check chunks for "+filename);
	    long now=System.currentTimeMillis();
	    boolean okay = encoder.fecEncodeFile(file, checkfile, fecMetrics);
	    long secs = (System.currentTimeMillis()-now)/1000;
	    if (!okay) {
		ic.addlog("[Error] unexpected error while building "+
			  "check chunks!");
		return 2;
	    }
	    ic.addlog("#"+ic.num(myNumber)+
		      " "+filename+": Finished generating "+
		      "check chunks (" +secs +" seconds for "+
		      file.length()+ " bytes)");
	    fecProgress.setProperty(hash, fecToken);
	} finally {
	    if (encoder !=null) {
		encoder.releaseInstance();
	    }
	}
	return 0;
    }

    /**
     * Returns the file that contains the check chunks.
     */
    public File produceFile() {
	return checkfile;
    }

    /**
     * Returns the MD5 hash of the file
     */
    public String produceHash() {
	return hash;
    }

    /**
     * Returns the SHA1 hash of the file
     */
    public String getSHA1() {
	return sha1;
    }
}
