package fiw.core.jobs;

import fiw.core.*;


/**
 * A simple JobScheduler implementation that does no reordering of jobs.
 * @author mihi
 */
public class DefaultJobScheduler implements JobScheduler {

    private Thread schedThread;
    private ThreadCounter myCounter;
    private boolean success = true, finished = false, criticalError=false,
	shouldStop = false;
    private ThreadProducer tp;
 
    
    /**
     * The jobs to schedule.
     */
    protected Job[] jobs;

    /**
     * A synchronisation object.
     */
    protected Object syncJobs = new Object();

    /**
     * Creates a new DefaultJobScheduler.
     * @param jobs the jobs to schedule
     * @param maxThreads the maximal number of threads to use
     * @param l a logger to log errors
     * @param tp the ThreadProducer instance to use for threaded jobs
     */
    public DefaultJobScheduler(Job[] jobs, int maxThreads, MyLogger l,
			       ThreadProducer tp) {
	for (int i=0;i<jobs.length;i++) { // sanity check (no pun intended!)
	    Job[] deps = jobs[i].getDependencies();
	    middle:
	    for (int j=0;j<deps.length;j++) {
		for(int k=0;k<i;k++) {
		    if (jobs[k] == deps[j]) continue middle;
		}
		throw new IllegalArgumentException
		    ("Job "+deps[j]+ " not scheduled before "+jobs[i]);
	    }
	}
	myCounter = new ThreadCounter(maxThreads,l);
	this.jobs=jobs;
	this.tp=tp;
    }

    /**
     * Starts the scheduler in a new thread.
     */
    public void start() {
	schedThread = new Thread() {
		public void run() {
		    runScheduler();
		}
	    };
	schedThread.start();
    }

    /**
     * Starts the scheduler and waits until all threads are scheduled.
     */
    public void startAndWait() {
	runScheduler();
    }

    /**
     * Called by a job to notify the scheduler that is has finished.
     * @param j the calling job
     */
    public synchronized void notifyFinished(Job j) {
	synchronized(syncJobs) {
	    myCounter.releaseThread();
	    syncJobs.notify();
	}
    }
    
    /**
     * Called by a job to notify the scheduler that the job is not
     * run. The job has to call {@link #notifyFinished} nevertheless.
     * @param j the calling job
     */
    public void notifySkipping(Job j) {}
    
    /**
     * Called by a job to notify the scheduler that an error
     * occurred. The job has to call {@link #notifyFinished}
     * nevertheless.
     * @param stopAll if the error should stop the whole insertion process.
     */
    public synchronized void notifyError(boolean stopAll) {
	synchronized(myCounter) {
	    success = false;
	    criticalError=stopAll;
	}
    }
    
    /**
     * Called by a job to request being run as a thread. This method
     * is delegated to the thread producer.
     * @param j the calling job
     */
    public synchronized void runAsThread(Job j) {
	tp.produceThread(j);
    }
    
    /**
     * Checks if all jobs have been scheduled.
     * @return true, iff all jobs have been scheduled
     */
    public boolean hasFinished() {
	synchronized (myCounter) {
	    return finished;
	}
    }

    /**
     * Requests the scheduler to stop its jobs as soon as possible.
     */
    public void doStop() {
	synchronized(myCounter) {
	    shouldStop = true;
	}
    }

    /**
     * Waits until all jobs have been scheduled.
     */
    public void waitForFinished() {
	try {
	    schedThread.join();
	} catch (InterruptedException ex) {
	    ex.printStackTrace();
	}
    }

    /**
     * Checks if an error occured during run.
     * @return true, iff the run was successful.
     */
    public synchronized boolean wasSuccessful() {
	synchronized(myCounter) {
	    return success;
	}
    }

    private int currentJob = 0;

    /**
     * Does the actual scheduler task - maybe in a new thread.
     */
    protected void runScheduler() {
	tp.initialize();
	try {
	    int nextJob;
	    while ((nextJob = getNextJob()) != -1) { 
		synchronized(myCounter) {
		    if (criticalError || shouldStop) {
			break;
		    }
		    myCounter.startThread();
		}
		jobs[nextJob].start(DefaultJobScheduler.this, nextJob+1);
	    }
	    myCounter.waitForAllThreads();
	} catch (InterruptedException ex) {
	    ex.printStackTrace();
	}
	tp.finish();
	finished=true;
    }

    /**
     * Returns the next job to be scheduled. The job returned must be
     * ready to run. May be overridden by subclasses to provide
     * reordering.
     * @return the index of the next job in the jobs array
     */
    protected int getNextJob() throws InterruptedException { // no reordering
	if (currentJob>=jobs.length)  {
	    currentJob = -1;
	} else {
	    synchronized(syncJobs) {
		while (!jobs[currentJob].isReadyToRun()) {
		    syncJobs.wait();
		}
	    }
	}
	return currentJob++;
    }   
}
