package fiw.core.insert.event;

/**
 * An event that can occur while inserting.
 * @author mihi
 */
public interface InsertEvent {

    /**
     * Get the job number firing this event, or 0 if it is a global
     * event.
     */
    public int getJobNumber();

    /**
     * Get the state change of a job. This can be
     * {@link #JOB_STATE_STARTED}, {@link #JOB_STATE_ENDED},
     * {@link #JOB_STATE_STARTSTOP} or {@link #JOB_STATE_NO_ACTION}.
     */
    public int getJobStateChange();

    /**
     * Check whether the job has started. This is true iff
     * {@link #getJobStateChange()} returns {@link #JOB_STATE_STARTED} or
     * {@link #JOB_STATE_STARTSTOP}.
     */
    public boolean isJobStarted();

    /**
     * Check whether the job has started. This is true iff
     * {@link #getJobStateChange()} returns JOB_STATE_ENDED or
     * JOB_STATE_STARTSTOP.
     */
    public boolean isJobEnded();

    /**
     * Returns the textual representation of that event that should be
     * appear in logs.
     */
    public String toLogString();

    /**
     * Returns the Freenet key attached to this event, or
     * <code>null</code> if none is attached.
     */
    public String getKey();

    /**
     * Returns <code>true</code> iff the log string should be treated
     * as partial, e. g. the next string should go into the same
     * line. Use of this is discouraged since there is no reliable way
     * to determine which event is its successor - so it may only be
     * used when only a single thread is creating events.
     */
    public boolean isPartialLogString();

    /**
     * Returns a string that may be shown next to a job in a graphical
     * progress window. This might also be used to be logged into a
     * "job log" - one log for each job.
     */
    public String toJobLogString();

    /**
     * Represents that the job has just started.
     * @see #getJobStateChange()
     */
    public static final int JOB_STATE_STARTED=1;
    
    /**
     * Represents that the job has just ended.
     * @see #getJobStateChange()
     */
    public static final int JOB_STATE_ENDED=2;
    
    /**
     * Represents that the job state is unchanged.
     * @see #getJobStateChange()
     */
    public static final int JOB_STATE_NO_ACTION=0;

    /**
     * Represents that the job has just started and ended as
     * well. This will be the only event ever sent by this job.
     * @see #getJobStateChange()
     */
    public static final int JOB_STATE_STARTSTOP=
	JOB_STATE_STARTED + JOB_STATE_ENDED;
}
