package fiw.core.insert;
import fiw.core.*;
import fiw.core.jobs.*;
import fiw.fcp.*;

import java.io.*;
import java.util.*;
import fiw.Main;

/**
 * Class that does the data gathering and initializes the job queue
 * for inserting a project.
 * @author mihi
 */
public class ProjectInsertController extends InsertController {

    private int project;
    private boolean clearProgress, useParser, noRealInsert;
    private List insertables = null;
    private FileClass[] fileClasses =null;
    private Thread myThread;
    private File basedir, metadir;
    private Settings ss;
    private boolean isDBR;
    private boolean fecNeeded=false;
    private InsertJob resultKeyJob = null, testResultKeyJob = null;

    /**
     * An empty job array used to specify no dependencies.
     */
    private static final Job[] NO_DEPS = new Job[0];

    /**
     * Creates a new ProjectInsertController and starts it.
     * @param ic the InsertContext used
     * @param project number of the project
     * @param useParser whether the parser should be used
     * @param noRealInsert whether the insert should be stopped before
     * files are inserted
     */
    public ProjectInsertController(InsertContext ic, int project,
				   boolean clearProgress, boolean useParser,
				   boolean noRealInsert) {
	super(ic);
	this.project=project;
	this.clearProgress=clearProgress;
	this.useParser=useParser;
	this.noRealInsert=noRealInsert;
	ss=FIWSystem.setts();
	basedir = ss.getProjectFile(project);
	metadir = FileUtil.fiwDir(basedir);
	isDBR = ss.getProjectSetting(project,"dbr").equals("true");

	myThread = new Thread(this);
	myThread.start();
    }
    
    /**
     * Inserts the project.
     * @return true, iff the project was inserted successfully
     */
    public boolean insert() {
	if (!greetAndCheck()) return false;
	File progressFile=new File(metadir, ic.testinsert?"testprogress.ini":
				   "progress.ini");
	if (!ic.loadProgress(progressFile, clearProgress)) {
	    ic.setRetryFutile();
	    return false;
	}
	ic.addlog("*** Loading mimetype.conf...");
	String htmlCharset=ss.getProjectSettingDef(project, "charset", "");
	ContentTypeManager ctm=null;
	try {
	    ctm = new ContentTypeManager(htmlCharset);
	} catch (FileNotFoundException ex) {
	    ic.addlog("[Error] Unable to load mimetype.conf.");
	    return false;
	}
	File parseFile = new File(metadir,"parse.ini");
	HTMLParser hp = null;
	if (parseFile.exists()) {
	    if (useParser) {
		hp= new HTMLParser(project, ic.testinsertmod,
				   ic.getMyLogger());
		ic.addlog("*** parsing files...");
		hp.parseAll();
		if (hp.getErrors()) {
		    ic.addlog ("--- failed parsing.");
		    return false;
		}
		ic.addlog("--- finished parsing.");
		useParser=false; // do not parse on retry
	    }
	} else {
	    parseFile = null;
	}
	if (!readFileClasses(parseFile)) return false;
	if (ic.shouldStop()) return false;
	String prepMap = getPreparedMapfileEntries(ctm);
	if (prepMap == null) return false;
	if (ic.shouldStop()) return false;
	ic.addlog("*** Searching files...");
	fecNeeded=false;
	Properties fecsettings = new Properties();
	String fecToken=FileUtil.dateString();
	File fecDataDir = new File(metadir, "fecdata");	
	File fecsetf = new File(metadir,"fec.ini");
	if (fecsetf.exists()) {
	    if (!FileUtil.loadProperties(fecsettings,fecsetf)) {
		ic.addlog("[Error] Unexpected error - consult logfile");
		return false;
	    }
	    fecsettings.setProperty("Version","2");
	}
	if (ic.shouldStop()) return false;
	if (!parseFiles(fecsettings, ctm.getIgnoredFileNames())) return false;
	if (ic.shouldStop()) return false;
	if (fecNeeded) {
	    if (!buildFECMetrics(fecDataDir)) return false;
	}
	if (ic.shouldStop()) return false;
	resultKeyJob=null;
	testResultKeyJob = null;
	Job[] jobs = buildJobs(fecsettings, fecToken, fecDataDir, ctm,
			       prepMap);
	if (jobs == null) return false;	
	ic.newChanceForUnreachable();
	// let the game begin ...
	boolean fail = !runJobList(jobs);
	if (ic.shouldStop()) return false;
	// some FEC cleanup
	if (fecNeeded && !fail) {
	    fecCleanup(fecsettings, fecsetf, fecDataDir, fecToken);
	}
	if (ic.shouldStop()) return false;
	if (ic.isAllUnreachable()) {
	    ic.addlog("[Warning] Some of your requests could not even make "+
		      "it off your node. Please check your Internet "+
		      "connection.");
	}
	if (fail) {
	    ic.addlog("\n+++ FAILED +++");
	    return false;
	}
	if (noRealInsert) {
	    ic.addlog("\n+++ FINISHED +++ (you selected to stop before "+
		      "inserting files)");
	    return true;
	}
	ic.addlog("\n+++ FINISHED +++");
	if (ic.testinsert) {
	    ic.addlog("***  Now you can test your freesite locally.");
	} else {
	    ic.addlog("*** Now you have to propagate your URI "+
		      "(e.g. send it to TFE).\n"+
		      "    Another good idea would be asking the guys in "+
		      "#freenet on IIP\n    (http://www.invisiblenet.net/iip/)"+
		      " if they can get your \n"+
		      "    freesite (as requesting it helps "+
		      "propagating it).");
	}
	if (testResultKeyJob != null &&
	    testResultKeyJob.getResultKey() != null) {
	    resultKeyJob = testResultKeyJob;
	}
	ic.addlog("\nYour URI:\n\n"+resultKeyJob.getResultKey()+
		  "//\n\nHave fun!\n\n");
	return true;
    }

    /**
     * shows the "header" of the insert and performs several checks.
     * In other words, these checks do not have any complex influence
     * on the process later - either they are passed or not.
     * @return true, iff all checks were successful
     */
    private boolean greetAndCheck() {
	ic.addlog("=*======== Inserting project (FIW "+
		  Main.VERSION+") ========*=\n"+
		  "Name: "+ss.getProjectSetting(project,"name")+"\n"+
		  "HTL: "+ic.htl+"\n"+
		  (isDBR?"  First slot: "+GUIHelper.getInstance().getDateString
		   (Integer.parseInt(ss.getProjectSetting(project,"dbr.first"))
		    *1000L)+"\n"+
		   "  Last slot:  "+GUIHelper.getInstance().getDateString
		   (Integer.parseInt(ss.getProjectSetting(project,"dbr.last"))
		    *1000L)+"\n":
		   "  Edition: "+
		   ss.getProjectSetting(project,"editionx")+"\n")+
		  "Public URI: "+
		  ss.getPublicURI(project,ic.testinsertmod,0)+"//\n"+
		  "Start of insert: "+FileUtil.dateString2()+"\n"+
		  "FCP host/port: "+fc.getHost()+":"+ fc.getPort()+"\n"+
		  (ic.testinsert?"[Test insert]\n":"")+
		  (noRealInsert?"[Only preparations]\n":"")+
		  "=*==============================================*=\n\n");
	if (ic.shouldStop()) return false;
	if (!basedir.exists()) {
	    ic.addlog("[Error] Directory does not exist.");
	    ic.setRetryFutile();
	    return false;
	}
	if (useParser && new File(basedir,"__index.html").exists()) {
	    ic.addlog("[Error] __index.html not supported any more.");
	    return false;
	}
	if (useParser && new File(basedir,"__mapfile").exists()) {
	    ic.addlog("[Error] __mapfile not supported any more.");
	    return false;
	}
	ic.addlog("*** Performing general checks:");
	boolean testsok = FIWSystem.setts().doTest(ic.getFCPConnection(),
						   project,
						   ic.getMyLogger());
	if (!testsok) {
	    ic.addlog("[Error] Checks failed. Stopping.");
	    return false;
	}
	ic.addlog("--- All checks finished.\n");
	return true;
    }

    /**
     * reads a file "mapfile.ini" in your .fiw directory and parses
     * mapfile entries from it.
     * @return these mapfile entries, or "" if the file does not
     * exist, or <code>null</code> if an error occurs
     */
    private String getPreparedMapfileEntries(final ContentTypeManager ctm) {
	if (new File(metadir,"mapfile.ini").exists()) {
	    StringBuffer prepmap=new StringBuffer();
	    ic.addlog("*** Reading mapfile.ini...");
	    Properties mf = FileUtil.loadMapFile(basedir);
	    if (mf==null) {
		ic.addlog("[Error] Could not load mapfile.ini");
		return null;
	    }
	    Iterator it=new EnumerationIterator(mf.propertyNames());
	    while(it.hasNext()) {
		String filename=(String)it.next();
		String fileCHK=mf.getProperty(filename);
		int posi;
		if ((posi=filename.indexOf('#')) != -1) {
		    prepmap.append("EndPart\nDocument\n"+
				   "Redirect.Target=freenet:"+
				   fileCHK+"\nName="+
				   filename.substring(0,posi)+
				   "\nInfo.Format="+
				   ctm.parseEncoding
				   (filename.substring(posi+1))+"\n");
		} else {
		    String ct = ctm.getCType(filename);
		    prepmap.append("EndPart\nDocument\n"+
				   "Redirect.Target=freenet:"+
				   fileCHK+"\nName="+filename+
				   "\nInfo.Format="+ct+"\n");
		}
	    }
	    ic.addlog("*** Read mapfile.ini");
	    return prepmap.toString();
	}
	return "";
    }

    /**
     * Determines the file class responsible for a given file name.
     * @param name the file name
     * @return the file class
     */
    private FileClass getFileClass(String name) {
	for (int j=0;j<fileClasses.length;j++) {
	    if (fileClasses[j].matchesFile(name)) {
		return fileClasses[j];
	    }
	}
	throw new RuntimeException("Bug somewhere in FileClass");
    }

    /**
     * Iterates through the project's files and checks how to process
     * them. The results are placed in the <code>insertables</code>
     * List.
     * @return true, iff no errors occurred
     */
    private boolean parseFiles(final Properties fecsettings,
			       final List ignfiles) {
	SubFile[] fl = SubFile.listSubFiles(basedir, ignfiles);
	for(int i=0;i<fl.length;i++) {
	    String name=fl[i].getSubName();
	    if (fl[i].isDirectory()) {
		// not really necessary, as there should not be any
		// directories in the list anyway
		ic.addlog ("[Warning] Ignoring directory "+name);
		continue;
	    }
	    boolean useFEC = false;
	    String prop=fecsettings.getProperty("usefec."+fl[i].getName());
	    if ("yes".equals(prop) || "true".equals(prop)) {
		useFEC= true;
	    } else if (!("no".equals(prop) || "false".equals(prop))) {
		if (fl[i].length() >=ic.FECMINSIZE) {
		    useFEC=true;
		    for(int j=0;j<ic.NOFECEXTS.length;j++) {
			if (fl[i].getName().endsWith(ic.NOFECEXTS[j]))
			    useFEC=false;
		    }
		}
	    }
	    FileClass fc = getFileClass(name);
	    if (fc.isContainer()) {
		if (useFEC) {
		    ic.addlog("[Error] cannot put FEC file "+name+
			      " into a container!");
		    return false;
		}
		fc.getContainer().addFile(fl[i]);
	    } else if (useFEC && ic.testinsert && ic.TESTNOFEC) {
		ic.addlog("*** Skipping FEC file "+ fl[i].getSubName());
	    } else {
		insertables.add(new InsertableFile
				(fl[i],fc.getPriority(), useFEC));
		if (useFEC) {
		    fecNeeded=true;
		}
	    }
	}
	return true;
    }

    /**
     * builds the FEC metrics for all files that need FEC.
     * @return true, iff no errors occurred
     */
    private boolean buildFECMetrics(final File fecDataDir) {
	if (!fecDataDir.exists()) fecDataDir.mkdir();
	FECConnection encoder=fc.getFECInstance();
	try {
	    if (encoder==null) {
		ic.addlog("*** waiting until FEC encoder is free..");
		    encoder=fc.waitForFECInstance();
		}
		ic.addlog("*** Generating FEC metrics for all files... ");
		for (Iterator it = insertables.iterator();it.hasNext();) {
		    Insertable ins = (Insertable) it.next();
		    if (ins instanceof InsertableFile) {
			InsertableFile insf = (InsertableFile) ins;
			if (insf.getUseFEC()) {
			    FECMetrics fecm=
				encoder.getFECMetrics(insf.getFile().length());
			    if (fecm == null) {
				ic.addlog ("[Error] Unexpected error while "+
					   "calculating FEC metrics");
				return false;
			    }
			    insf.setFECMetrics(fecm);
			}
		    }
		}
		ic.addlog("--- FEC metric generation finished.");
	    } finally {
		if (encoder !=null) {
		    encoder.releaseInstance();
		}
	    }
	return true;
    }

    /**
     * Parses the file classes from the given file. The classes are
     * stored into the fileClasses array.
     * @param parseFile the file to parse the data from
     * @return true, iff no errors occurred
     */
    private boolean readFileClasses(File parseFile) {
	ArrayList classes = new ArrayList();
	insertables = new ArrayList();
	if (parseFile != null) {
	    try {
		ic.addlog("*** Reading file classes");
		BufferedReader br=new BufferedReader(new FileReader(parseFile));
		String line;
		while((line=br.readLine())!=null) {
		    // :container:container0.zip:23
		    // :containerinsert:container1.zip:74
		    // :exact:index.html:container1.zip
		    if (line.startsWith(":")) {
			String[] parts = StringSplitter.split
			    (line.substring(1),":");
			if (parts.length != 3) {
			    ic.addlog("[Error] Cannot parse \""+line+"\"");
			    return false;
			}
			String type = parts[0];
			String match = parts[1];
			String destination = parts[2];
			int numdest;
			FreenetContainer container = null;
			try {
			    numdest = Integer.parseInt(destination);
			    if (numdest <0) numdest = -1;
			} catch (NumberFormatException ex) {
			    numdest = -1;
			}
			if (numdest == -1) { // maybe a container?
			    for (Iterator it = insertables.iterator();
				 it.hasNext();) {
				Insertable elem = (Insertable) it.next();
				if (elem instanceof FreenetContainer &&
				    ((FreenetContainer)elem).getName()
				    .equals(destination)) {
				    container = (FreenetContainer)elem;
				    numdest  = container.getPriority();
				    break;
				}
			    }
			    if (numdest == -1) {
				ic.addlog("[Error] Unknown container: "+
					      destination);
				return false;
			    }
			}
			if (type.equals("containerphantom")) {
			    if (container != null) {
				ic.addlog("[Error] Containers inside "+
					      "containers are not supported");
				return false;
			    }
			    insertables.add
				(new FreenetContainer(numdest, match, false,
						      new File(metadir,
							       match)));
			} else if (type.equals("container")) {
			    if (container != null) {
				ic.addlog("[Error] Containers inside "+
					  "containers are not supported");
				return false;
			    }
			    insertables.add
				(new FreenetContainer(numdest, match, true,
						      new File(metadir,
							       match)));
			} else if (type.equals("exact")) {
			    classes.add
				(new FileClass(FileClass.EXACT_MATCH, match,
					       numdest, container));
			} else if (type.equals("ending")) {
			    classes.add
				(new FileClass(FileClass.END_MATCH, match,
					       numdest, container));
			} else if (type.equals("regex")) {
			    classes.add
				(new FileClass(FileClass.REGEX_MATCH, match,
					       numdest, container));
			} else {
			    ic.addlog("[Error] Cannot parse \""+line+"\"");
			    return false;
			}
		    }
		}
		br.close();
		ic.addlog("*** Finished reading file classes");
	    } catch(IOException e) {
		ic.addlog("[Error] internal error, consult logfile");
		e.printStackTrace();
		e.printStackTrace(FIWSystem.log());
		return false;
	    }
	}
	classes.add(new FileClass(FileClass.END_MATCH,"",
				  ic.DEFAULT_PRIORITY,null));
	fileClasses = (FileClass[])
	    classes.toArray(new FileClass[classes.size()]);
	return true;
    }

    /**
     * uses the data collected to build the Job array of things to
     * do. Now this is the really long monster method (over 200
     * lines) - but it should be easily understandable.
     * @return the resulting job array
     */
    private Job[] buildJobs(final Properties fecsettings,
			    final String fecToken,
			    final File fecDataDir,
			    final ContentTypeManager ctm,
			    final String prepMap) {
    	ArrayList firstpass = new ArrayList(),
	    splitfiles = new ArrayList(),
	    secondpass = new ArrayList(),
	    toVerify = new ArrayList(),
	    mapfileNeeded= new ArrayList();
	ic.addlog("*** building job tree");
	Insertable[] insis = (Insertable[])
	    insertables.toArray(new Insertable[insertables.size()]);
	Arrays.sort(insis);
	int increment=0;
	if (isDBR && !noRealInsert) {
	    increment=Integer.parseInt(ss.getProjectSetting
				       (project,"dbr.increment"));
	    int offset=Integer.parseInt(ss.getProjectSetting
					(project,"dbr.offset"));
	    String key=ss.getPrivateURI(project,ic.testinsertmod);
	    String pubkey=ss.getPublicURI(project,ic.testinsertmod,-1);
	    InsertJob j = new DBRInsertJob(key, increment, offset, pubkey,
					   ic, NO_DEPS);
	    if (ic.NODBRMISMATCH) {
		j.setKeyCollisionOkay();
	    } else {
		j.setKeyCollisionWarning
		    ("[Error] Your current DBR and the DBR already "+
		     "available are different.\n        If you used another "+
		     "software for uploading them, add \n\n        "+
		     "tuning.nodbrmismatch=true\n\n        "+
		     "to your fiw.conf.");
	    }
	    firstpass.add(j);
	    toVerify.add(j);
	    resultKeyJob=j;
	}
	for (int i=0;i<insis.length;i++) {
	    Insertable in = insis[i];
	    if (in instanceof FreenetContainer) {
		FreenetContainer inc = (FreenetContainer)in;
		String filename = inc.getContainerFile().getName();
		ContainerBuilderJob bj = new ContainerBuilderJob
		    (inc, ctm, ic, NO_DEPS);
		firstpass.add(bj);
		if (!noRealInsert) {
		    HashCalcJob hj = new HashCalcJob(inc.getContainerFile(),
						     filename,
						     ic, NO_DEPS);
		    String metadata = "Version\nRevision=1\nEndPart\n"+
			"Document\nInfo.Format=application/zip\nEnd\n"+
			(ic.modifyMetadata?"Comment=FIW_Test\n":"");
		    FileInsertJob fj = new FileInsertJob
			(inc.getContainerFile(), filename, null,
			 null, false,
			 "CHK@", metadata, ic, new Job[] {hj});
		    firstpass.add(hj);
		    firstpass.add(fj);
		    if (inc.isInsertedAsFile()) {
			mapfileNeeded.add(fj);
		    }
		    MetadataMaskFillerJob mmf = new MetadataMaskFillerJob
			(bj, fj, ic, new Job[] {bj, fj});
		    firstpass.add(mmf);
		    mapfileNeeded.add(mmf);
		    if (in.getPriority() < 10) { // before mapfile
			toVerify.add(fj);
		    } else {
			fj.setCalcOnly();
			FileInsertJob fj2 =  new FileInsertJob
			    (inc.getContainerFile(), filename, null,
			     null, false,
			     "CHk@", metadata, ic, new Job[] {hj});
			secondpass.add(fj2);
			toVerify.add(fj2);
		    }
		}
	    } else if (in instanceof InsertableFile) {
		InsertableFile inf = ((InsertableFile) in);
		String filename = inf.getFile().getSubName();		    
		if (inf.getUseFEC()) {
		    int[] metrics = inf.getFECMetrics()
			.getLegacyFECMetrics();
		    boolean wasZero=false;
		    long accu = 0;
		    int cnt = 0;
		    List partJobs = new ArrayList();
		    Job fecBuilder = null;
		    HashCalcJobEx hj = new HashCalcJobEx(inf.getFile(),
							 filename,
							 ic, NO_DEPS);
		    firstpass.add(hj);
		    for (int j=0;j<metrics.length;j++) {
			FilePartInsertJob jjj;
			cnt++;
			if(metrics[j]==0) {
			    cnt = 0;
			    accu = 0;
			    wasZero=true;
			    fecBuilder = new FECBuilderJob
				(inf.getFile(), filename, fecsettings,
				 fecToken, fecDataDir, inf.getFECMetrics(),
				 ic, new Job[]{hj});
			    firstpass.add(fecBuilder);
			    continue;
			} else if (noRealInsert) {
			    if (wasZero) break; //no need for inserting
			    continue;
			} else if (wasZero) {
			    // check chunk
			    jjj = new FilePartInsertJob
				(null, filename, accu, metrics[j],
				 cnt,true, ic, new Job[] {fecBuilder});
			} else {
			    // data chunk
			    jjj = new FilePartInsertJob
				(inf.getFile(), filename, accu, metrics[j],
				 cnt,false, ic, new Job[] {hj});        
			}
			accu+=metrics[j];
			partJobs.add(jjj);
			firstpass.add(jjj);
			if (inf.getPriority() < 10) {
			    toVerify.add(jjj);
			} else {
			    jjj.setCalcOnly();
			    Job jjj2= new FilePartInsertJob(jjj);
			    secondpass.add(jjj2);
			    toVerify.add(jjj2);
			}
		    }
		    if (!noRealInsert) {
			partJobs.add(0,fecBuilder);
			Job[] pJobs = (Job[]) partJobs.toArray
			    (new Job[partJobs.size()]);
			Job fc = new SplitfileBuilderJob
			    (inf.getFile().length(), filename,
			     ctm.getCType(filename), ic, pJobs);
			firstpass.add(fc);
			InsertJob fci = new SplitFileInsertJob
			    ("CHK@", filename, ctm.getCType(filename),
			     filename.equals("index.html"),
			     ic, new Job[] {fc});
			splitfiles.add(fci);
			mapfileNeeded.add(fci);
			if (inf.getPriority() < 10) {
			    toVerify.add(fci);
			} else {
			    fci.setCalcOnly();
			    InsertJob fci2 = new SplitFileInsertJob
				("CHK@", filename, ctm.getCType(filename),
				 filename.equals("index.html"),
				 ic, new Job[] {fc});
			    secondpass.add(fci2);
			    toVerify.add(fci2);
			}    
		    }
		} else if (!noRealInsert) {
		    HashCalcJob hj = new HashCalcJob(inf.getFile(),
						     filename,
						     ic, NO_DEPS);
		    FileInsertJob fj = new FileInsertJob
			(inf.getFile(), filename, ctm.getCType(filename),
			 null, filename.equals("index.html"),
			 ic.modifyMetadata, ic, new Job[] {hj});
		    firstpass.add(hj);
		    firstpass.add(fj);
		    mapfileNeeded.add(fj);
		    if (inf.getPriority() < 10) { // before mapfile
			toVerify.add(fj);
		    } else {
			fj.setCalcOnly();
			FileInsertJob fj2 =  new FileInsertJob
			    (inf.getFile(), filename, ctm.getCType(filename),
			     null, filename.equals("index.html"),
			     ic.modifyMetadata, ic, new Job[] {hj});
			secondpass.add(fj2);
			toVerify.add(fj2);
		    }
		}
	    } else {
		ic.addlog("[Error] Internal error: cannot handle "+
			  in.getClass().getName());
		return null;
	    }
	}
	firstpass.addAll(splitfiles);
	if (ic.shouldStop()) return null;
	if (!noRealInsert) {
	    String extra = "";
	    if (!"0".equals(ss.getProjectSetting(project,"editionx"))) {
		ic.addlog("    (Adding .next link to mapfile)");
		extra+=("EndPart\nDocument\nRedirect.Target=freenet:"+
			ss.getPublicURI(project,ic.testinsertmod,1)+"\n"+
			"Name=.next\nInfo.Format="+
			ctm.parseEncoding("text/html")+"\n");
	    }
	    if (prepMap.length() >0) {
		ic.addlog ("    (Adding mapfile.ini entries to mapfile)");
		extra+=prepMap;
	    }
	    Job mfj2 = new MapfileBuilderJob(extra,ic,
					     (Job[]) mapfileNeeded.toArray
					     (new Job[mapfileNeeded.size()]));
	    firstpass.add(mfj2);
	    InsertJob mfj = new IndirectMapfileInsertJob(ic,true,
							 new Job[]{mfj2});
	    firstpass.add(mfj);
	    if (getFileClass("indirect.mapfile").getPriority() <10) {
		toVerify.add(mfj);
	    } else {
		Job mfj3 = new IndirectMapfileInsertJob(ic, false,
							new Job[]{mfj2});
		mfj.setCalcOnly();
		firstpass.add(mfj3);
		toVerify.add(mfj3);
	    }
	    ArrayList mapfileJobs = new ArrayList();
	    if (isDBR) {
		int fromslot=Integer.parseInt(ss.getProjectSetting
					      (project,"dbr.first"));
		int toslot=Integer.parseInt(ss.getProjectSetting
					    (project,"dbr.last"));
		for(int slot=fromslot; slot<=toslot;slot+=increment) {
		    String key = ss.getPrivateURI(project,ic.testinsertmod,
						  slot);
		    String title = "slot ["+
			GUIHelper.getInstance().getDateString(slot*1000L)+"]";
		    Job j = new MetadataInsertJob(key, null, title, ic,
						  new Job[] {mfj});
		    firstpass.add(j);
		    toVerify.add(j);
		    mapfileJobs.add(j);
		}
		ic.addlog("*** To test DBRs, use the "+
			  "`?date=yyyymmdd-hh:mm:ss' parameter");
	    } else {
		String key=ss.getPrivateURI(project,ic.testinsertmod);
		InsertJob j2 = new MetadataInsertJob(key, null, "mapfile",
						     ic, new Job[] {mfj});
		firstpass.add(j2);
		toVerify.add(j2);
		resultKeyJob = j2;
		mapfileJobs.add(j2);
	    }
	    if (ic.testinsert) {
		Job[] allmapfiles = (Job[]) mapfileJobs.toArray
		    (new Job[mapfileJobs.size()]);
		Job coll = new CollisionFallbackJob((MetadataProducing)mfj,
						    ic, allmapfiles);
		firstpass.add(coll);
		testResultKeyJob = new MetadataInsertJob
		    ("CHK@",null, "CHK mapfile", ic, new Job[] {coll});
		firstpass.add(testResultKeyJob);
		toVerify.add(testResultKeyJob);
	    }
	    if (ic.doVerify(true)) {
		DummyJob dj;
		secondpass.add(dj = new DummyJob
			       ("=============================\n"+
				"Fetching inserted data again:",
				ic, (Job[])toVerify.toArray
				(new Job[toVerify.size()])));
		for (Iterator it = toVerify.iterator(); it.hasNext();) {
		    InsertJob j = (InsertJob) it.next();
		    secondpass.add(j.buildChecker(true, null, dj));
		}
	    }
	}
	firstpass.addAll(secondpass);
	return (Job[]) firstpass.toArray(new Job[firstpass.size()]);
    }
    /**
     * Stores the FEC states (which files were successfully encoded)
     * into fec.ini and deletes any obsolete FEC cache files.
     */
    private void fecCleanup(final Properties fecsettings,
			    final File fecsetf,
			    final File fecDataDir,
			    final String fecToken) {
		    Iterator iter = fecsettings.keySet().iterator();
	    while (iter.hasNext()) {
		String name = (String) iter.next();
		if (!"Version".equals(name) && !name.startsWith("usefec.")) {
		    String value = (String) fecsettings.getProperty(name);
		    if (!fecToken.equals(value)) {
			ic.addlog("*** Deleting FEC cache file "+name);
			new File(fecDataDir, name).delete();
			iter.remove(); 
		    }
		}
	    }
	    FileUtil.saveProperties(fecsettings,fecsetf,
				    "FEC settings cache");
    }
}
