package fiw.core.insert;
import fiw.core.*;
import fiw.core.jobs.*;
import fiw.fcp.FCPConnection;

/**
 * Controls the insert of anything (like a project, a file).
 * This class contains everything that is independent of
 * what should be inserted.
 */

public abstract class InsertController implements Runnable, Stoppable {

    protected InsertContext ic;
//    protected InsertLogger logger;
    protected FCPConnection fc;
    private boolean failure = false;
    private Thread sleeping= null;
    private JobScheduler js;

    private static final boolean USE_DEBUGGING_FRIENDLY_SCHEDULER = false;
    
    public InsertController (InsertContext ic) {
	this.ic=ic;
//	logger = ic.getLogger();
	fc=ic.getFCPConnection();
    }
    
    public void run () {
	ic.fireIAmHere(this);
	int retry=0;
	while(!ic.shouldStop()) {
	    failure=false;
	    ic.fireRestart();
	    if (insert()) {
		ic.fireSetStatus(I18n.get("Finished_Insertion"));
		break;
	    } else {
		ic.fireSetStatus(I18n.get("Stopped_Insertion"));
		if (!failure) break;
		if (ic.isRetryFutile())
		    break;
	    }
	    if (ic.shouldStop()) break;
	    ic.addlog("\n\n ******* Retrying insert in 1 minute *******\n\n");
	    ic.fireSetStatus(I18n.get("RetryStopInsert"));
	    sleeping = Thread.currentThread();
	    try {
		Thread.sleep(60000);
	    } catch (InterruptedException e) {
		//nothing
	    }
	    sleeping = null;
	    ic.fireSetStatus(I18n.format("RetryInsert",new Integer(++retry)));
	}
	if (ic.shouldStop()) {
	    ic.addlog("+++ Aborted +++ ");
	    ic.fireSetStatus(I18n.get("StatusAborted"));
	}
	ic.fireDone();
    }

    /** a method to stop all the insert threads. */
    public synchronized void stopRunning() {
	ic.doStop();
	if (js != null) {
	    js.doStop();
	}
	if (sleeping != null)
	    sleeping.interrupt();
	ic.addlog("========= will try to abort... =========");
	ic.fireSetStatus(I18n.get("StatusAborting"));
    }

    protected void reportFailure() {
	failure = true;
    }

    protected boolean hasFailed() {
	return failure;
    }

    protected boolean runJobList(Job[] jobs) {
	ic.addlog("*** Processing "+jobs.length+" jobs...");
	if (USE_DEBUGGING_FRIENDLY_SCHEDULER) {
	    js = new DefaultJobScheduler(jobs, ic.THREADS, ic.getMyLogger(),
					 new SimpleThreadProducer());
	} else {
	    int waittime = Integer.parseInt(FIWSystem.setts().getProperty
					    ("tuning.threadwait"));
	    ThreadProducer tp;
	    if (waittime==-1) {
		tp = new SimpleThreadProducer();
	    } else {
		tp = new PooledThreadProducer(waittime*1000L);
	    }
	    js = new ReorderingJobScheduler(jobs, ic.THREADS, ic.getMyLogger(),
					    tp);
	}
	js.startAndWait();
	if (!js.wasSuccessful()) {
	    reportFailure();
	    js = null;
	    return false;
	}
	js = null;
	return true;
    }

    
    public abstract boolean insert();
    
}
