package fiw.core;

/**
 * Counts threads and blocks if more than a given number of threads
 * have been "created". Can wait for all this threads as well. It's
 * the job of the threads to inform this class when they have
 * finished.
 * @author mihi
 */

public class ThreadCounter {

    private int maxThreads, freeThreads;
    private MyLogger logger;

    /**
     * Creates a new ThreadCounter.
     * @param max the maximum number of threads to run.
     * @param logger a logger to log errors.
     */
    public ThreadCounter (int max, MyLogger logger) {
	maxThreads=max;
	freeThreads=max;
	this.logger=logger;
    }

    /**
     * Increases the number of threads. If no more threads are
     * allowed, wait until one thread has finished.
     */
    public void startThread() {
        try {
            synchronized(this) {
                while (freeThreads<=0) {
                    wait();
                }
                freeThreads--;
            }
        } catch(InterruptedException e) {
            e.printStackTrace();
            logger.addlog("FATAL!");
        }
    }

    /**
     * Waits until all threads have finished.
     */
    public void waitForAllThreads() {
        synchronized(this) {
            try {
                while (freeThreads < maxThreads) {
                    wait();
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
                logger.addlog("FATAL!");
            }
        }
    }

    /**
     * Called by the thread to notify this counter that a thread has
     * finished.
     */
    public void releaseThread() {
	synchronized(this) {
	    freeThreads++;
	    notifyAll();
	}
    }

    /**
     * Checks if there are running threads.
     * @return <code>true</code> iff there are running threads.
     */
    public synchronized boolean isRunning() {
	return  freeThreads < maxThreads;
    }
}

	
