package fiw.core;
import fiw.fcp.*;
import fiw.core.option.*;

import java.io.*;
import java.util.*;

/** This file contains code for holding all the project settings and
 * contains most of the file I/O code */

public class Settings {

    private static Settings inst;
    private Properties prop;

    /**
     * To be used only by {@link FIWSystem}.
     */
    protected static synchronized Settings getInstance() {
	if (inst == null) {
	    inst=new Settings();
	    inst.checkSettings();
	}
	return inst;
    }
    
    private Settings() {
	Properties defprop=new Properties();
	GlobalOptions gos = new GlobalOptions(false, false);
	Option[] sets = gos.getSettings();
	for (int i=0;i<sets.length;i++) {
	    Option o = sets[i];
	    if (!o.isSpecial() && o.getProperty() != null) {
		defprop.setProperty(o.getProperty(),o.getDefault());
	    }
	}
	prop=new Properties(defprop);
	try {
	    InputStream in=new FileInputStream("fiw.conf");
	    prop.load(in);
	    in.close();
	} catch (FileNotFoundException e) {
	    // nothing
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }

    public void flush() {
	try {
	    OutputStream out=new FileOutputStream("fiw.conf");
	    prop.store(out,"FIW settings");
	    out.flush();
	    out.close();
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }
    
    public int getProjCount() {
	return Integer.parseInt(prop.getProperty("proj.count"));
    }

    public void setProjCount(int count) {
	prop.setProperty("proj.count",""+count);
	flush();
    }

    
    public String[] getProjectNames() {
	int cnt=getProjCount();
	String[] result=new String[cnt];
	for (int i=0;i<cnt;i++) {
	    result[i]=prop.getProperty("proj."+i+".name");
	}
	return result;
    }

    public static String getEditionString(int number) {
	String ss=""+number;
	if (number >= 11000 && number < 12000) { // 334 edition number format
	    return ss.substring(ss.length()-3);
	}
	return ss;
    }
    
    public String getPrivateURI(int project, boolean testing) {
	return getPrivateURI(project, testing,0);
    }
    
    public String getPrivateURI(int project, boolean testing, int slot) {
	Settings ss=this;
	String uri;
	int edition=Integer.parseInt
	    (ss.getProjectSetting(project,"editionx"));
	if (ss.getProjectSetting(project, "pubkey").equals("KSK")) {
	    uri="KSK@"+(testing?"htl0test/":"")+
		ss.getProjectSetting(project, "keydir")+
		(edition==0?"":"/"+getEditionString(edition));
	} else if (ss.getProjectSetting(project,"pubkey").equals("CHK")) {
	    uri="CHK@";
	} else {
	    uri="SSK@"+ss.getProjectSetting(project,"privkey")+
		ss.getMaybeEntropy(project)+ "/"+
		(slot==0?"":Integer.toHexString(slot)+"-")+
		(testing?"htl0test/":"")+ss.getProjectSetting(project,"keydir")+
		(edition==0?"":"/"+getEditionString(edition));
	}
	return uri;
    }
    
    public String getMaybeEntropy(int project) {
	String entropy = getProjectSettingDef(project, "entropy", "");
	return (entropy.length()==0?"":","+entropy);
    }

    public String getPublicURI(int project, boolean testing, int editioninc) {
	Settings ss=this;
	String uri;
	int edition=0;
	if(editioninc==-1) { // no edition number at all please
	    edition=0;
	} else if (editioninc != 0) { // future edition number please
	    edition=Integer.parseInt
		(ss.getProjectSetting(project,"editionx"))+editioninc;
	} else {
	    edition=Integer.parseInt(ss.getProjectSetting
				     (project,"editionx"));
	}
	if (ss.getProjectSetting(project,"pubkey").equals("KSK")) {
	    uri="KSK@"+(testing?"htl0test/":"")+
		ss.getProjectSetting(project,"keydir")+
		(edition==0?"":"/"+getEditionString(edition));
	} else if (ss.getProjectSetting(project,"pubkey").equals("CHK")) {
	    uri="CHK@";
	} else {
	    uri="SSK@"+ss.getProjectSetting(project,"pubkey")+"PAgM"+
		getMaybeEntropy(project)+"/"+
		(testing?"htl0test/":"")+ss.getProjectSetting(project,"keydir")+
		(edition==0?"":"/"+getEditionString(edition));
	}
	return uri;
    }
    
    public File getProjectFile(int no) {
	return new File(getProjectSetting(no,"path"));
    }
    
    public void setProperty(String name, String val) {
	prop.setProperty(name,val);
	flush();
    }

    public String getProperty(String name) {
	return prop.getProperty(name);
    }

    public void renameProperties(String from, String to) {
	copyProperties(from, to, true);
    }

    public void copyProperties(String from, String to, boolean move) {
	Iterator i = new EnumerationIterator(prop.propertyNames());
	List al=new ArrayList();
	while (i.hasNext()) {
	    String el=(String)i.next();
	    if (el.startsWith(from))
		al.add(el);
	}
	i=al.iterator();
	while(i.hasNext()) {
	    String key=(String)i.next();
	    String newKey= to + key.substring(from.length());
	    String value=prop.getProperty(key);
	    prop.setProperty(newKey,value);
	    if (move) {
		prop.remove(key);
	    }
	}
	flush();
    }
    
    public void deleteProperties(String prefix) {
	Iterator it=new EnumerationIterator(prop.propertyNames());
	List al=new ArrayList();
	while (it.hasNext()) {
	    String el=(String)it.next();
	    if (el.startsWith(prefix))
		al.add(el);
	}
	it=al.iterator();
	while(it.hasNext()) {
	    String key=(String)it.next();
	    prop.remove(key);
	}
	flush();
    }

    public void saveProject(int num, String property, String value) {
	if ( num >=getProjCount()) {
	    prop.setProperty("proj.count",""+(num+1));
	}
	prop.setProperty("proj."+num+"."+property,value);
	flush();
    }
	
    public String getProjectSetting(int pro, String name) {
	return prop.getProperty("proj."+pro+"."+name);
    }

    public String getProjectSettingDef(int pro, String name, String defaul) {
	String res = prop.getProperty("proj."+pro+"."+name);
	return res==null?defaul:res;
    }
                
    public boolean doTest(FCPConnection fcpConn, int project,
			  MyLogger logger) {
	logger.addlogNoNewline("Checking if path exists...");
	File path=getProjectFile(project);
	if (!path.exists()) {
	    logger.addlog("FAILED!");
	    return false;
	}
	logger.addlog("OK");
	logger.addlogNoNewline("Checking if URI is valid...");
	boolean b = checkURI(fcpConn, project);
	if (!b) {
	    logger.addlog("FAILED!");
	    if (getProjectSetting(project,"pubkey").endsWith("PAgM")) {
		logger.addlog("Perhaps you should remove the "+
			      "PAgM from the end of your public key?");
	    }
	    return false;
	}
	logger.addlog("OK");
	logger.addlogNoNewline("Checking for index.html...");
	if (!new File(path,"__index.html").exists() &&
	    !new File(path,"index.html").exists()) {
	    logger.addlog("FAILED!");
	    return false;
	}
	logger.addlog("OK");
	String cset = getProjectSettingDef(project, "charset","");
	if (cset.length() != 0) {
	    logger.addlogNoNewline("Checking charset...");
	    try {
		"".getBytes(cset);
	    } catch (UnsupportedEncodingException e) {
		logger.addlog("FAILED!");
		return false;
	    }
	    logger.addlog("OK");
	}
	return true;
    }
    
    private boolean checkURI(FCPConnection fcpConn, int project) {
	String priv=getPrivateURI(project,true);
	for (int i=0;i<priv.length();i++) {
	    if (priv.charAt(i)>126 || priv.charAt(i) < 43)
		return false;
	}
	String pub=getPublicURI(project,true,0);
	String rand=""+new Random().nextInt(100000000);
	if (pub.equals("CHK@")) return true;
	FCPInsertResult ins= fcpConn.insertStream
	    (priv+"--testing"+rand,
	     "Version\nRevision=1\nComment=SyntaxCheck\nEnd\n",
	     new ByteArrayInputStream(new byte[0]), 0, 0, "CHECKINSERT");
	if (ins == null || !(ins instanceof InsertSuccessResult)) return false;
	return ins.getKey().equals(pub+"--testing"+rand);
    }

    public static String makeTime(int seconds) {
	return twoDigits(seconds/3600)+":"+twoDigits((seconds/60)%60) + ":" +
	    twoDigits(seconds % 60);
    }
    
    public static String twoDigits(int i) {
	return ((i<10)?"0":"")+i;
    }
    
    public static int parseTime(String timeValue) {
	try {
	    StringTokenizer time=new StringTokenizer(timeValue,":");
	    int value=3600, result = 0;
	    while(time.hasMoreTokens()) {
		result+=Integer.parseInt(time.nextToken())*value;
		value/=60;
	    }
	    return result;
	} catch (NumberFormatException e) {
	    return -1;
	}
    }

    public static String loadFileRaw(File f) {
	try {
	    DataInputStream fis=new DataInputStream(new FileInputStream(f));
	    byte[] b = new byte[(int)f.length()];
	    fis.readFully(b);
	    return new String(b,"ISO-8859-1");
	} catch (FileNotFoundException e) {
	    return null;
	} catch (IOException e) {
	    e.printStackTrace();
	    return null;
	}
    }

    public int updateNimNumber(int project) {
	int nimnum=Integer.parseInt
	    (getProjectSettingDef(project,"nextnim","1"));
	String fetchednims=getProjectSettingDef(project,"checkednims","");
	while(fetchednims.indexOf("("+nimnum+")") != -1) {
	    nimnum++;
	}
	return nimnum;
    }
    
    private void checkSettings() {
	Set done=new HashSet();
	Iterator it;
	int cnt = getProjCount();
	for (int i=0;i<cnt;i++) {
	    Options opts = new Options(i);
	    Option[] sets = opts.getSettings();
	    for (int j=0;j<sets.length;j++) {
		Option o = sets[j];
		if (!o.isSpecial()) {
		    done.add("proj."+i+"."+o.getProperty());
		}
	    }
	}
	Option[] gopts = new GlobalOptions(true).getSettings();
	for(int i=0;i<gopts.length;i++) {
	    Option o = gopts[i];
	    if (!o.isSpecial()) {
		done.add(o.getProperty());
	    }
	}
	it=new EnumerationIterator(prop.propertyNames());
	while (it.hasNext()) {
	    String next = (String) it.next();
	    if (!done.contains(next)) {
		System.out.println("*** obsolete/unknown setting: "+
				   next+"="+prop.getProperty(next));
	    }		
	}
    }
    
    public void makeReport() {
	Set done=new HashSet();
	StringBuffer report=new StringBuffer("Report about FIW.conf dated "+
					     FileUtil.dateString2()+"\n");
	Iterator it;
	report.append("FCP address: "+prop.getProperty("fcp.host")+":"+
		      prop.getProperty("fcp.port")+"\n");
	done.add("fcp.host");
	done.add("fcp.port");
	report.append("\nProject listing:\n===============\n");
	done.add("proj.count");
	int cnt = getProjCount();
	for (int i=0;i<cnt;i++) {
	    report.append("\nProject ").append(i).append(": ")
		.append(getProjectSetting(i,"name")).append("\n\n");
	    Option[] opts = new Options(i).getSettings();
	    for (int j=0;j<opts.length;j++) {
		Option o = opts[j];
		if (!o.isSpecial()) {
		    report.append(o.getName()).append("=")
			.append(o.getValue())
			.append("\n");
		    done.add("proj."+i+"."+o.getProperty());
		}
	    }
	}
	report.append("\nTuning options:\n==============\n\n");
	Option[] gopts = new GlobalOptions(true).getSettings();
	for(int i=0;i<gopts.length;i++) {
	    Option o = gopts[i];
	    if (!o.isSpecial()) { 
		report.append(o.getName()).append("=")
		    .append(o.getValue())
		    .append("\n");
		done.add(o.getProperty());
	    }
	}
	report.append("\nStrange settings:\n=================\n\n");
	it=new EnumerationIterator(prop.propertyNames());
	while (it.hasNext()) {
	    String next = (String) it.next();
	    if (!done.contains(next)) {
		    report.append(next).append("=")
			.append(prop.getProperty(next)).append("\n");
	    }		
	}
	report.append("+++ finished +++");
	FileUtil.saveFile(new File("fiw-conf-report.log"),report.toString());
    }    
}
