package fiw.core;

import java.io.*;
import java.util.*;

/**
 * An InputStream that loads only a part from another input stream. If
 * the input stream does not have enough data, the rest is filled with
 * zero bytes.
 */
public class PaddingInputStream extends InputStream {

    /**
     * The number of bytes that remain to be read.
     */
    private long remaining;

    /**
     * The underlying input stream.
     */
    private InputStream in;

    /**
     * Creates a new PaddingInputStream.
     * @param in the underlying input stream.
     * @param skip the number of bytes to skip from it
     * @param length the number of bytes to read from it
     */
    public PaddingInputStream(InputStream in, long skip, long length)
	throws IOException {
	this.in=in;
	long toskip=skip, skipped=-1;
	while (toskip > 0)  {
	    long lastskipped = skipped;
	    skipped = in.skip(toskip);
	    if (skipped == 0 && lastskipped == 0) 
		break;
	    toskip -= skipped;
	}
	remaining=length;
    }

    /**
     * Reads a byte.
     */
    public int read() throws IOException {
	if (remaining==0) return -1;
	int b = in.read();
	if (b==-1) b=0;
	remaining--;
	return b;
    }

    /**
     * Reads multiple bytes.
     */
    public int read(byte[] b, int off, int len) throws IOException {
	if (remaining <0) throw new RuntimeException
			      ("Remaining = "+remaining);
	if (remaining == 0) return -1;
	int length = len;
	if (length>remaining) length=(int)remaining;
	int result = in.read(b,off,length);
	if (result == -1) {
	    result = length;
	    Arrays.fill(b,off,length,(byte)0);
	}
	remaining -= result;
	return result;
    }

    /**
     * Closes this stream.
     */
    public void close() throws IOException {
	in.close();
    }
}
