package fiw.core;
import fiw.core.insert.*;
import fiw.core.jobs.*;
import fiw.Main;

import java.util.*;
import java.io.*;

/**
 * Controls NIM checks.
 */
public class NimCheckerController extends InsertController {

    private InsertContext ic;
    private int advance;

    /**
     * Creates a new NIM checker controller
     * @param ic the InsertContext
     * @param advance the number of NIMs to fetch from each project
     */
    public NimCheckerController(InsertContext ic,
				int advance) {
	super(ic);
	this.ic = ic;
	this.advance=advance;
	ic.fireSetStatus(I18n.get("NIMProgress"));
	new Thread(this).start();
    }

    /**
     * Overrides the superclass method to prevent repetition of the check.
     */
    public void run () {
	ic.fireIAmHere(this);
	insert();
	ic.fireDone();
    }

    /**
     * Does the actual check. The stupid name "insert" is caused by
     * the fact that this class extends an InsertController that
     * should have been named better as well.
     * @return whether the check was successful
     */
    public boolean insert() {
	ic.addlog("======== Checking your NIMs (FIW "+
		  Main.VERSION+") =========\n"+
		  "Advance: "+advance+"\n"+
		  "HTL: "+ic.fetchHTL+"\n"+
		  "Start of check: "+FileUtil.dateString2()+"\n"+
		  "FCP host/port: "+fc.getHost()+":"+ fc.getPort()+"\n"+
		  "================================================\n\n");
	Settings ss = FIWSystem.getInstance().getSettings();
	int count = ss.getProjCount();
	ArrayList prefixes = new ArrayList(),
	    numbers=new ArrayList();
	ic.addlog("*** Checking projects...");
	int total = 0;
	for (int i=0;i<count;i++) {
	    String prefix = ss.getProjectSetting(i,"nimprefix");
	    if (prefix != null && !"".equals(prefix)) {
		String checkednims = ss.getProjectSettingDef
		    (i, "checkednims", "");
		int num;
		try {
		    num = Integer.parseInt
			(ss.getProjectSetting(i,"nextnim"));
		} catch (NumberFormatException e) {
		    num=1;
		}
		int found=0;
		int[] nimlist=new int[advance];
		while (found < advance) {
		    if (checkednims.indexOf("("+num+")") == -1) {
			nimlist[found]=num;
			found++;
		    }
		    num++;
		}
		ic.addlog("KSK@"+prefix+"-:"+printList(nimlist));
		prefixes.add(prefix);
		numbers.add(nimlist);
		total++;
	    }
	}
	Job[] NO_DEPS=new Job[0];
	Job[] jobs = new Job[total*advance];
	for (int i=0;i<advance;i++) {
	    for (int j=0;j<total;j++) {
		String key = "KSK@"+(String)prefixes.get(j)+"-"+
		    ((int[])numbers.get(j))[i];
		jobs[i*total+j]= new RetrieveCheckJob
		    (key, "-",key, "",ic, NO_DEPS);
	    }
	}
	ic.addlog("*** Finished checking projects");
	ic.addlog("*** Trying to fetch NIMs...");
	if (!ic.shouldStop()) {
	    runJobList(jobs);
	}
	if (ic.shouldStop()) {
	    ic.addlog("+++ Aborted +++");
	    ic.fireSetStatus(I18n.get("StatusAborted"));
	    ic.fireDone();
	    return false;
	}
	List successes=new Vector();
	for (int i=0;i<advance*total;i++) {
	    if (jobs[i].hasFinished() && jobs[i].wasSuccessful()) {
		successes.add(((RetrieveCheckJob)jobs[i]).getKey());
	    }
	}
	ic.addlog("*** "+successes.size()+" NIMs found:");
	Iterator it=successes.iterator();
	String FPROXY_URL=FIWSystem.setts().getProperty("fproxy.url");
	try {
	    BufferedWriter htmlw=new BufferedWriter
		(new FileWriter("foundnims.html"));
	    BufferedWriter fw=new BufferedWriter
		(new FileWriter("foundnims.log",true));
	    htmlw.write("<html><head><title>FIW "+Main.VERSION+" - found NIMs"+
			"</title></head><body bgcolor=\"#BBBBBB\">"+
			"<h1>Found NIMs</h1>");
	    htmlw.newLine();
	    while(it.hasNext()) {
		String next=(String) it.next();
		ic.addlog(next);
		fw.write(next);
		fw.newLine();
		htmlw.write("<h2>"+next+"</h2><iframe width=\"100%\" height"+
			    "=200 src=\""+FPROXY_URL+next+"\"></iframe>");
		htmlw.newLine();
		int pos=next.indexOf("-");
		String prefix=next.substring(4,pos);
		String tail=next.substring(pos+1);
		for (int i=0;i<count;i++) {
		    if (prefix.equals(ss.getProjectSetting(i,"nimprefix"))) {
			String old=ss.getProjectSettingDef(i,"checkednims","");
			ss.saveProject(i, "checkednims", old+"("+tail+")");
			
		    }
		}     
	    }
	    fw.flush();
	    fw.close();
	    htmlw.write("</body></html>");
	    htmlw.newLine();
	    htmlw.flush();
	    htmlw.close();
	    ic.addlog("+++ FINISHED +++");
	    ic.addlog("\nfoundnims.log contains all NIMs ever found "+
		      "(until you delete them).");
	    ic.addlog("foundnims.html contains iframes for all NIMs "+
		      "found in this run.");
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    ic.addlog("[Error] unexpected error - check logfile");
	}
	ic.fireSetStatus(I18n.format("NIMsFound",
				     new Integer(successes.size())));
	return true;
    }

    /**
     * Converts an int array to a string.
     * @param the int array
     * @return the string
     */
    private String printList(int[] list) {
	String result=" ";
	for (int i=0;i<list.length;i++) {
	    result+=list[i]+" ";
	}
	return result;
    }    
}
