package fiw.core;

import java.util.*;
import java.io.*;
import java.text.*;
import java.net.*;

/**
 * a "new" HTML parser for edition headers and more.
 * @author mihi
 */

public class HTMLParser {

    private Properties vars, defVars;
    private Properties templates=new Properties(); // cache
    private File srcDir, dstDir, incDir;
    private String charset;
    private MyLogger ml;
    private boolean hasErrors=false;
    private Date baseDate;

    /**
     * The (platform dependent) line separator.
     */
    public static final String NEWLINE = System.getProperty("line.separator");

    /**
     * Returns if earlier parse processes caused errors. This
     * information is not reset when this method is called.
     */
    public boolean getErrors() {
	return hasErrors;
    }

    /**
     * Creates a new HTML parser.
     * @param projectnum the number of the project
     * @param test if the files should be parsed for a test insert
     * @param ml a Logger to log errors
     */
    public HTMLParser (int projectnum, boolean test, MyLogger ml) {
	this.ml=ml;
	Settings ss = FIWSystem.getInstance().getSettings();
	defVars=new Properties();
	defVars.setProperty("ps.ednum",
			 ss.getProjectSetting(projectnum,"editionx"));
	defVars.setProperty("ps.projname",
			 ss.getProjectSetting(projectnum,"name"));
	defVars.setProperty("ps.nimpref",
			 ss.getProjectSettingDef(projectnum,"nimprefix",
						 ""));
	defVars.setProperty("ps.nextnim",
			 ss.getProjectSettingDef(projectnum,"nextnim", ""));
	defVars.setProperty("ps.actlink",ss.getProjectSetting(projectnum,
							    "activelink"));
	defVars.setProperty("ps.keyonly","/SSK@"+ss.getProjectSetting
			    (projectnum, "pubkey")+"PAgM"+
			    ss.getMaybeEntropy(projectnum));
	defVars.setProperty("ps.pubkey",ss.getProjectSetting
			    (projectnum, "pubkey"));
	/* uncomment the following two lines to add support for the
	 * #$#ps.privkey; tag. THIS IS ON YOUR OWN RISK, if you use
	 * some third party template that uses this command
	 * afterwards, and so publish your private key, it is entirely
	 * your fault! */
// 	defVars.setProperty("ps.privkey",
// 			    ss.getProjectSetting(projectnum,"privkey"));
	defVars.setProperty("ps.keydir", (test?"htl0test/":"")+
			    ss.getProjectSetting(projectnum, "keydir"));
	defVars.setProperty("ps.key","/"+ss.getPublicURI(projectnum, test,-1));
	defVars.setProperty("ps.keynoslash",ss.getPublicURI(projectnum, test,-1));
	defVars.setProperty("tags.title",htmlescape(ss.getProjectSetting
						    (projectnum,"title")));
	defVars.setProperty("tags.category", htmlescape
			    (ss.getProjectSetting (projectnum,"category")));
	defVars.setProperty("tags.author", htmlescape
			    (ss.getProjectSetting(projectnum,"author")));
	defVars.setProperty("tags.description", htmlescape
			    (FileUtil.loadFile(new File
					      (ss.getProjectFile(projectnum),
					       "description.txt"))));
	srcDir=ss.getProjectFile(projectnum);
	dstDir=new File(FileUtil.fiwDir(srcDir),"parsed");
	if (!dstDir.exists()) dstDir.mkdirs();
	incDir=new File(FileUtil.fiwDir(srcDir),"include");
	if (!incDir.exists()) incDir.mkdir();
	vars=new Properties(defVars);
	charset=ss.getProjectSettingDef(projectnum, "charset", "");
	if (charset.length()==0) charset="ISO-8859-1";
	String lockedTime = ss.getProjectSettingDef
	    (projectnum,"lockedtime", "");
	if (lockedTime.length() == 0) {
	    baseDate = new Date();
	} else {
	    baseDate = new Date(Long.parseLong(lockedTime));
	}
    }

    
    /**
     * Parses all files that must be parsed for the current project.
     */
    public void parseAll() {
	try {
	    File f=new File(FileUtil.fiwDir(srcDir),"parse.ini");
	    BufferedReader br=new BufferedReader(new FileReader(f));
	    String line;
	    while((line=br.readLine())!=null) {
		if (!line.startsWith("#") &&
		    !line.startsWith(":") &&
		    line.length() != 0 &&
		    new File(srcDir,line).exists()) {
		    parse(line);
		}
	    }
	    br.close();
	} catch(IOException e) {
	    ml.addlog("[Error] internal error, consult logfile");
	    hasErrors=true;
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	}
    }

    /**
     * Parses one file.
     * @param filename the file to parse
     */
    public synchronized void parse(String filename) {
	ml.addlog("     parsing "+filename);
	try {
	    BufferedReader br=new BufferedReader
		(new InputStreamReader
		 (new FileInputStream(new File(srcDir,filename)),
		 charset));
	    BufferedWriter bw=new BufferedWriter
		(new OutputStreamWriter
		 (new FileOutputStream(new File(dstDir,filename)),
		 charset));
	    String line;
	    StringBuffer all=new StringBuffer();
	    while ((line=br.readLine()) != null) {
		all.append(line).append(NEWLINE);
	    }
	    bw.write(parseString(all.toString()));
	    bw.flush();
	    bw.close();
	    br.close();
	} catch (IOException e) {
	    ml.addlog("[Error] internal error, consult logfile");
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    hasErrors=true;
	}
    }

    /**
     * Parses a string.
     * @param the string to parse
     * @return the parsed string
     */
    private String parseString(String data) throws IOException {
	StringBuffer done=new StringBuffer(), to=done;
	int index=0;
	String todef=null;
	if (data==null) return null;
	while(index < data.length()) {
	    int p = data.indexOf("#$#", index);
	    if (p==-1) break;
	    if (data.charAt(p+3) == '#') {
		to.append(data.substring(index,p+2));
		index=p+3;
		continue;
	    }
	    int p2 = data.indexOf(";",p);
	    if (p2==-1) break;
	    to.append(data.substring(index,p));
	    String between = data.substring(p+3,p2);
	    if (between.startsWith("def:")) {
		if (todef != null) {
		    vars.setProperty(todef,to.toString());
		    if (todef.startsWith("add:") &&
			vars.getProperty(todef.substring(4)) == null)
			vars.setProperty(todef.substring(4),to.toString());
		    todef=null;
		}
		if (between.equals("def:end")) { // end of definition
		    to=done;
		} else { // start of definition
		    to=new StringBuffer();
		    todef=between.substring(4);
		}
	    } else {
		to.append(parseTag(between));
	    }
	    index=p2+1;
	}
	to.append(data.substring(index));
	if (todef != null) {
	    ml.addlog("[Parse Error] Definition of #$#"+todef+"; not ended.");
	    todef=null;
	    hasErrors=true;
	}		
	return done.toString();
    }

    /**
     * Parses a tag (without leading <code>#$#</code> and trailing
     * <code>;</code>).
     * @param tag the tag
     * @return the parsed tag
     */
    private String parseTag(String tag) throws IOException {
	int p;
	if(tag.equals("/")) {
	    return "/";
	} else if (tag.startsWith("e:")) {
	    try {
		int num = Integer.parseInt(parseTag(tag.substring(2)));
		if (num <1) num=1;
		return Settings.getEditionString(num);
	    
	    } catch (NumberFormatException e) {
		return Settings.getEditionString(1);
	    }
	} else if (tag.startsWith("url:")) {
	    try {
		BufferedReader br=new BufferedReader
		(new InputStreamReader
		 (new URL(tag.substring(4)).openStream(),
		  "ISO-8859-1"));
		String line;
		StringBuffer inc=new StringBuffer();
		while ((line=br.readLine()) != null) {
		    inc.append(line).append(NEWLINE);
		}
		return inc.toString();
	    } catch (IOException e) {
		e.printStackTrace(FIWSystem.log());
		ml.addlog("[Parse Error] Problems with "+tag+
			  ", consult logfile.");
		hasErrors=true;
		return "";
	    }
	} else if (tag.startsWith("file:")) {
	    try {
		BufferedReader br=new BufferedReader
		    (new InputStreamReader
		     (new FileInputStream(new File(incDir, tag.substring(5))),
		      charset));
		String line;
		StringBuffer inc=new StringBuffer();
		while ((line=br.readLine()) != null) {
		    inc.append(line).append(NEWLINE);
		}
		return inc.toString();
	    } catch (FileNotFoundException e) {
		ml.addlog("[Parse Error] "+tag+" not found");
		hasErrors=true;
		return "";
	    }
	} else if (tag.startsWith("date:")) {
	    String formatString=tag.substring(5);
	    if (formatString.length()==0) {
		formatString = "EEEE, yyyy-MM-dd HH:mm:ss";
	    }
	    SimpleDateFormat sdf= new SimpleDateFormat(formatString,
						       Locale.US);
	    sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
	    return sdf.format(baseDate);
	} else if ((p=tag.indexOf("+")) != -1) {
	    try {
		return ""+(Integer.parseInt(parseTag(tag.substring(0,p)))+
			   Integer.parseInt(parseTag(tag.substring(p+1))));
	    } catch (NumberFormatException e) {
		ml.addlog("[Parse error] not a number: \""+tag+"\"");
		hasErrors=true;
		return "#NaN#";
	    }
	}
	if (isInteger(tag)) return tag;
	String value=vars.getProperty(tag.toLowerCase());
	if (value!=null) return value;
	value=parseString(getTemplate(tag.toLowerCase()));
	if (value!=null) return value;
	ml.addlog ("[Parse error] undefined tag \""+tag+"\"");
	hasErrors=true;
	return "#undefined#";
    }

    /**
     * Checks if a given string is an integer literal.
     * @param text the string
     * @return <code>true</code> iff it is an integer literal
     */
    private boolean isInteger(String text) {
	boolean ret=true;
	try {
	    Integer.parseInt(text);
	} catch (NumberFormatException e) {
	    ret=false;
	}
	return ret;
    }

    /**
     * Loads a template (from templates.html).
     * @param name the name of the template
     * @return the template
     */
    private String getTemplate(String name) throws IOException {
	String value = templates.getProperty(name);
	if (value != null) return value;
	try {
	    FileInputStream fis;
	    try {
		fis=new FileInputStream("templates.html");
	    } catch (FileNotFoundException e) {
		fis=new FileInputStream("../templates.html");
	    }
	    BufferedReader br= new BufferedReader
		(new InputStreamReader
		 (fis, "ISO-8859-1"));
	    String line;
	    boolean add=false;
	    StringBuffer tmpl = new StringBuffer();
	    while((line=br.readLine()) != null) {
		if (line.equals("<hR><h2>#$#"+name+";</h2>")) {
		    add=true;
		} else if (add) {
		    if (line.startsWith("<hR>")) break;
		    if (line.startsWith("<!-- hR")) break;
		    tmpl.append(line).append(NEWLINE);
		}
	    }
	    br.close();
	    if (add) {
		templates.setProperty(name,tmpl.toString());
		return tmpl.toString();
	    } else {
		return null;
	    }
	} catch (FileNotFoundException e) {
	    return null;
	}
    }

    /**
     * Escapes a string (especially <code>&lt;</code>, <code>&gt;</code>
     * and <code>&amp;</code>) for HTML.
     * @param toescape the string to escape
     * @return the escaped string
     */
    private String htmlescape(String toescape) {
	if (toescape == null) return "";
	while (toescape.startsWith("\n")) {
	    toescape=toescape.substring(1);
	}
	while (toescape.endsWith("\n")) {
	    toescape=toescape.substring(0,toescape.length()-1);
	}
	StringBuffer b = new StringBuffer();
	for (int i=0;i<toescape.length();i++) {
	    String replacement;
	    switch (toescape.charAt(i)) {
	    case '>': 
		replacement="&gt;"; break;
	    case '<':
		replacement="&lt;"; break;
	    case '&':
		replacement="&amp;"; break;
	    case '\n':
		replacement="<br />"; break;
	    default:
		replacement= ""+toescape.charAt(i);
	    }
	    b.append(replacement);
	}
	return b.toString();
    }
}
