package fiw.core;

import java.io.*;
import java.util.*;
import java.text.*;

/**
 * This class contains static methods for file i/o.
 * @author mihi
 */
public class FileUtil {
    
    /**
     * Returns the <code>.fiw</code> folder below the given 
     * <code>parent</code> dir. This folder is used to store internal data.
     * If a <code>__fiw</code> folder exists, this one is renamed to 
     * <code>.fiw</code> for compatibility with versions before 0.06.
     * If neither folder exists, a <code>.fiw</code> folder is created.
     * @param parent the folder that contains the .fiw folder.
     * @return the fiw filder.
     */
    public static File fiwDir (File parent) {
	File res= new File(parent,".fiw");
	File resOld = new File(parent,"__"+"fiw");
	if (resOld.exists()) {
	    resOld.renameTo(res);
	}
	if (!res.exists()) res.mkdir();
	return res;
    }
    
    /**
     * oads the "mapfile.ini" file that contains additional mapfile
     * entries.
     * @param dir the directory of that project
     * @return the "mapfile.ini" properties
     */
    public static Properties loadMapFile(File dir) {
	Properties p = new Properties();
	if (!loadProperties(p,new File(fiwDir(dir), "mapfile.ini"))) {
	    return null;
	}
	return p;
    }

    /**
     * Loads the descriptions shown in the left part of the wizard.
     * @param no the internal number of the wizard step
     * @return the description
     */
    public static String getWizardDesc(int no) {
	
	    InputStream is=Settings.class.getResourceAsStream
		("/res/"+I18n.get("HelpPrefix")+"-"+no+".txt");
	    if (is==null) return "[Description not found]";
	    return loadFileFromStream(is);
    }

    /**
     * Loads the data from an InputStream into a String. The default 
     * character encoding is used.
     * @param is the InputStream
     * @return the String
     */
    public static String loadFileFromStream(InputStream is) {
	try {
	    BufferedReader br = new BufferedReader(
		new InputStreamReader(is));
	    StringBuffer s=new StringBuffer();
	    String line;
	    while((line=br.readLine())!=null) {
		s.append(line).append("\n");
	    }
	    br.close();
	    return s.toString();
	} catch (IOException e) {
		return "";
	}
    }
    
    /**
     * Saves a String into a file. Old content is deleted.
     * @param fl the file
     * @param s the string
     */
    public static void saveFile(File fl, String s) {
	try {
	    BufferedWriter bw = new BufferedWriter(new FileWriter(fl));
	    bw.write(s);
	    bw.flush();
	    bw.close();
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }

    /**
     * Loads a text file into a string.
     * @param fl the file
     * @return the string
     */
    public static String loadFile(File fl) {	
	try {
	    return loadFileFromStream(new FileInputStream(fl));
	} catch (FileNotFoundException e) {
	    return "";
	}
    }

    /**
     * Loads Properties from a file into a Properties object
     * @param p an empty Properties object to fill
     * @param f the file
     * @return <code>true</code>, iff the file was successfully loaded.
     */
    public static boolean loadProperties (Properties p, File f) {
	try {
	    InputStream in = new FileInputStream(f);
	    p.load(in);
	    in.close();
	    return true;
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return false;
	}
    }

    /**
     * Saves Properties into a file.
     * @param p the Properties object
     * @param f the destination file
     * @param desc a description for the file
     * @return <code>true</code>, iff the file could be saved properly
     */
    public static boolean saveProperties (Properties p, File f, String desc) {
	try {
	    OutputStream out=new FileOutputStream(f);
	    p.store(out,desc);
	    out.flush();
	    out.close();
	    return true;
	} catch (IOException e) {
	    e.printStackTrace();
	    e.printStackTrace(FIWSystem.log());
	    return false;
	}
    }

    /**
     * Creates a date string from current date (like 20030420-132201). 
     * This format is used for log filenames.
     */
    public static String dateString() {
	DateFormat df=new SimpleDateFormat("yyyyMMdd-HHmmss");
	return df.format(new Date());
    }
    
    /**
     * Creates a date string from current date 
     * (like 2003-04-20 13:22:01)
     */
    public static String dateString2() {
	DateFormat df=new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	return df.format(new Date());
    }
}
