package fiw.cli;
import fiw.core.*;
import fiw.fcp.*;
import fiw.core.option.*;

import java.io.*;

/**
 * An interactive command line interface for FIW.
 * @author mihi
 */
public class Cli {

    private BufferedReader stdIn;
    private int project;

    /**
     * Constructor.
     */
    public Cli() {
	try {
	    while (System.in.available() != 0) System.in.read();
	} catch (IOException ex) {
	    ex.printStackTrace();
	}
	stdIn = new BufferedReader(new InputStreamReader(System.in));
    }

    /**
     * Starts the Cli. Asks for an options and runs it afterwards
     * until one enters 'q' to quit.
     */
    public void start() {
	try {
	    if (initialize()) {
		while(true) {
		    switch(mainSelect()) {
		    case -1: System.out.println("Unknown command");
			break;
		    case 0: return;
		    case 1: editProject(); break;
		    case 2: deleteProject(); break;
		    case 3: insertProject(); break;
		    case 4: moveCloneProject(); break;
		    }
		}
	    }
	    System.out.println("\n\nThanks for using FIW!");
	} catch (IOException ex) {
	    ex.printStackTrace();
	}
    }

    /**
     * Shows welcome message and sets up the FCP connection
     * @return <code>true</code> if the FCP connection was set up properly.
     */
    private boolean initialize() throws IOException {
	createHelpFile();
	System.out.println("Welcome to the FIW command line interface!\n"+
			   "You can find help in the file \"fiwcli.txt\".\n"+
			   "This file is divided into sections - each "+
			   "section is about one 'step' \nin FIW cli.\n");
	System.out.println("------------------");
	System.out.println("Step 0 - configure FCP connection");
	FCPConnection fc = FIWSystem.getInstance().createFCPConnection();
	while(true) {
	    System.out.println("\nCurrent config: "+fc.getHost()+
			       ":"+fc.getPort());
	    String s=fc.handshake();
	    if (s==null)  {
		System.out.println ("Could not connect to host.\n\n"+
				    "Press return to retry with same "+
				    "settings");
		
	    } else {
		System.out.println("Connection successful. Answer from "+
				   "fred:\n\n"+s+"\n\nPress Return to go "+
				   "to the next step\nPress 'a' for advanced "+
				   "settings");
	    }
	    System.out.println("Enter new host:port pair to "+
			       "change config\nEnter 'q' to quit");
	    System.out.print("fiw>");
	    String line = stdIn.readLine();
	    if (line==null) return false;
	    if (line.equals("") && s != null) return true;
	    if (line.equals("")) continue;
	    if (line.equals("q")) return false;
	    if (line.equals("a") && s != null) {
		advancedSettings();
	    } else if (line.indexOf(":") == -1) {
		System.out.println("Unknown line.");
	    } else {
		int pos = line.indexOf(":");
		try {
		    int port = Integer.parseInt(line.substring(pos+1));
		    String host = line.substring(0,pos);
		    fc = FIWSystem.getInstance()
			.createFCPConnection(host, port);
		} catch (NumberFormatException ex) {
		    System.out.println("Invalid port number.");
		}
	    }
	}
    }

    /**
     * Shows the main menu.
     * @return the number of the option selected.
     */
    private int mainSelect() throws IOException {
	System.out.println("------------------");
	System.out.println("Step 2 - Select what to do\n");
	String[] pr=FIWSystem.setts().getProjectNames();
	for(int i=0;i<pr.length;i++) {
	    System.out.println("["+(i+1)+"] "+pr[i]);
	}
	System.out.println("\nActions:\n[C]reate new, [E]dit #, [I]nsert #, "+
			   "[D]elete #, [M]ove/clone, [Q]uit");
	System.out.println("Enter the letter of of the desired action plus "+
			   "the number of the\nproject, e.g. e3 to edit "+
			   "Project 3");
	System.out.print("fiw>");
	String line = stdIn.readLine();
	if (line == null || line.length() <1) return -1;
	if (line.length() >1) {
	    try {
		project = Integer.parseInt(line.substring(1));
		if (project<1 || project >pr.length) return -1;
	    } catch (NumberFormatException ex) {
		return -1;
	    }
	} else {
	    project = -1;
	}
	switch(line.toLowerCase().charAt(0)) {
	case 'q': return 0;
	case 'c': project = pr.length+1; return 1;
	case 'e': if (project==-1) return -1;
	    return 1;
	case 'd': if (project == -1) return -1;
	    return 2;
	case 'i': if (project == -1) return -1;
	    return 3;
	case 'm': return 4;
	default: return -1;
	}
    }

    /**
     * Edits a project.
     */
    private void editProject() throws IOException {
	System.out.println("------------------");
	System.out.println("Step 3 - create/edit a project\n");
	System.out.println("NOTE: Most options are not checked here - so "+
			   "do not complain \nif a project created w/ cli "+
			   "throws an exception...\n");
	int currentStep =3;
	Options os = new Options(project-1);
	Option[] oss = os.getSettings();
	outer:
	for (int i=0;i<oss.length;i++) {
	    Option o = oss[i];
	    int panel = o.getPanelNumber();
	    if ((panel<3 || panel > 6) && panel != 1001) continue;
	    if (o instanceof ButtonOption) {
		System.out.print(o.getName()+" [y/N]:");
		String line = stdIn.readLine();
		if (line != null && line.equals("q")) {
		    break outer;
		} else if (line != null && line.equals("y")) {
		    ((ButtonOption)o).handleAction();
		}
	    } else if (o.isSpecial()) {
		switch (o.getSpecialNumber()) {
		case 1:
		    System.out.println("\nTo generate a new key pair, just "+
				       "leave these values as is and enter "+
				       "'y' in \nthe following question.");
		    break;
		case 3: // site description
		    System.out.println("\nCreate a file called "+
				       "description.txt in your project "+
				       "folder and add the\ndescription "+
				       "for this project there.\n");
		    break;
		case 4:
		    System.out.println("\nAdd the files you want to parse "+
				       "(including index.html, if you want to"+
				       " parse\nthat file) into "+
				       ".fiw/parse.ini.\n");
		    break;
		case 5:
		    //FIXME: add a check settings thingy...
		    break;
		case 6:
		    if (!os.getOption("dbr").getValue().equals("true")) {
			i+=2;
		    }
		    break;
		default:
		    System.out.println("ERROR: Special number invalid "+
				       "in Option #"+i);
		}
	    } else {
		if (o.getPanelNumber() != currentStep &&
		    o.getPanelNumber() != 1001) {
		    currentStep=o.getPanelNumber();
		    System.out.println("\n------------------");
		    System.out.println("Step "+currentStep+
				       " - create/edit a project\n");
		}
		while (true) {
		    System.out.print(o.getName()+
				     ((o.getType() instanceof
				       DayTimeOptionType) ? " (seconds)" : "")+
				     " ["+o.getValue()+"]: ");
		    String line = stdIn.readLine();
		    if (line == null || line.equals("")) {
			break;
		    } else if (line.equals("q")) {
			break outer;
		    } else {
			boolean ok = o.setValue(line);
			if (!ok) {
			    System.out.println(o.getInvalidReason(line));
			} else {
			    break;
			}
		    }
		}
	    }
	}
	System.out.println("--- saving settings ---");
	os.save();
	System.out.print("--- settings saved - press return ---");
	stdIn.readLine();
    }

    /**
     * Allows to set up advanced ("tuning") options.
     */
    private void advancedSettings() throws IOException {
	System.out.println("------------------");
	System.out.println("Step 1 - advanced settings\n");
	System.out.println("NOTE: Most options are not checked here\n");
	GlobalOptions os = new GlobalOptions();
	Option[] oss = os.getSettings();
	outer:
	for (int i=0;i<oss.length;i++) {
	    Option o = oss[i];
	    int panel = o.getPanelNumber();
	    if (!(panel==1)) continue;
	    if (o.isSpecial()) {
		switch (o.getSpecialNumber()) {
		case 1:
		    System.out.println("Configure geeky settings?");
		    System.out.print("y/n>");
		    String line = stdIn.readLine();
		    if (line != null && !line.equals("y")) {
			break outer;
		    }
		    break;
		default: System.out.println("ERROR: Special number invalid "+
					    "in Option #"+i);
		}
	    } else {
		while (true) {
		    System.out.print(o.getName()+" ["+o.getValue()+"]: ");
		    String line = stdIn.readLine();
		    if (line == null || line.equals("")) {
			break;
		    } else if (line.equals("q")) {
			break outer;
		    } else {
			boolean ok = o.setValue(line);
			if (!ok) {
			    System.out.println(o.getInvalidReason(line));
			} else {
			    break;
			}
		    }
		}
	    }
	}
	os.save();
    }

    /**
     * Asks for confirmation, then deletes a project.
     */
    private void deleteProject() throws IOException {
	System.out.println("------------------");
	System.out.println("Step 7 -delete a project\n");
	int p = project -1;
	System.out.println("Deleting a project cannot be undone.\n"+
			   "Do you want to proceed?\n");
	System.out.println("Project name: '"+
			   FIWSystem.setts().getProjectSetting(p,"name")+"'"+
			   "\npublic key: "+
			   FIWSystem.setts().getProjectSetting(p,"pubkey")+
			   "\nprivate key: "+
			   FIWSystem.setts().getProjectSetting(p,"privkey"));
	System.out.print("y/n>");
	String line = stdIn.readLine();
	if (line != null && line.toLowerCase().equals("y")) {
	    FIWSystem.setts().deleteProperties("proj."+p+".");
	    for (int i=p;i<FIWSystem.setts().getProjCount();i++) {
		FIWSystem.setts().renameProperties
		    ("proj."+(i+1)+".","proj."+(i)+".");
	    }
	    FIWSystem.setts().setProperty("proj.count",
				      ""+(FIWSystem.setts().getProjCount()-1));
	}
    }

    /**
     * not implemented yet. Use {@link CliInsert}.
     */
    private void insertProject() throws IOException {
	System.out.println("------------------");
	System.out.println("Step 8 -insert a project\n");
	System.out.println("Please use the `fiw cliins' command!");
	System.out.print("\nPress return. ");
	stdIn.readLine();
	//FIXME
    }

    /**
     * not implemented yet. Use {@link CliInsert}.
     */
    private void moveCloneProject() throws IOException {
	while (true) {
	    System.out.println("------------------");
	    System.out.println("Step 2b - move/clone projects\n");
	    String[] pr=FIWSystem.setts().getProjectNames();
	    for(int i=0;i<pr.length;i++) {
		System.out.println("["+(i+1)+"] "+pr[i]);
	    }
	    System.out.println("\nMove [u]p #, Move [d]own #, [C]lone #, "+
			       "[Q]uit");
	    System.out.print("fiw>");
	    String line=stdIn.readLine();
	    if (line == null) return;
	    if (line.length() <1) {
		project=-2;
	    } else if (line.length() >1) {
		try {
		    project = Integer.parseInt(line.substring(1));
		    if (project<1 || project >pr.length) project=-2;;
		} catch (NumberFormatException ex) {
		    project = -2;
		}
	    } else {
		project = -1;
	    }
	    if (project==-2) {
		System.out.println("Unknown command");
		continue;
	    }
	    switch(line.toLowerCase().charAt(0)) {
	    case 'q': return;
	    case 'c':
		if (project==-1) {
		    System.out.println("Unknown command");
		} else {
		    int newnum = pr.length, projectnum = project-1;
		    FIWSystem.setts().setProjCount(newnum+1);
		    FIWSystem.setts().copyProperties
			("proj."+projectnum+".", "proj."+newnum+".", false);
		}		
		break;
	    case 'u':
		if (project==-1) {
		    System.out.println("Unknown command");
		} else if (project==1) {
		    System.out.println("Project 1 cannot be moved up.");
		} else {
		    int projectnum=project-1;
		    FIWSystem.setts().renameProperties
			("proj."+projectnum+".", "proj.TEMP.");
		    FIWSystem.setts().renameProperties
			("proj."+(projectnum-1)+".","proj."+projectnum+".");
		    FIWSystem.setts().renameProperties
			("proj.TEMP.","proj."+(projectnum-1)+".");
		}
		break;
	    case 'd':
		if (project==-1) {
		    System.out.println("Unknown command");
		} else if (project == pr.length) {
		    System.out.println("Project "+project+
				       " cannot be moved down.");
		} else {
		    int projectnum = project -1;
		    FIWSystem.setts().renameProperties
			("proj."+projectnum+".", "proj.TEMP.");
		    FIWSystem.setts().renameProperties
			("proj."+(projectnum+1)+".","proj."+projectnum+".");
		    FIWSystem.setts().renameProperties
			("proj.TEMP.","proj."+(projectnum+1)+".");
		}
		break;
	    default: 
	    }
	}
    }
    
    /**
     * Creates a "help" file containing all the wizard step help texts.
     */
    private void createHelpFile() throws IOException {
	File dest = new File("fiwcli.txt");
	if (dest.exists()) return;
	try {
	    PrintWriter w = new PrintWriter(new BufferedWriter
					    (new FileWriter(dest)));
	    for (int i=0;i<100;i++) {
		String desc = FileUtil.getWizardDesc(i);
		if (desc.startsWith("[")) break;
		w.println("======================");
		w.println("STEP "+i);
		w.println("======================");
		w.println(desc);
		w.println();
	    }
	    w.flush();
	    w.close();
	} catch (IOException ex) {
	    ex.printStackTrace();
	}
    }
}
