package fiw.addon;
import fiw.fcp.*;
import fiw.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.text.*;

/**
 * Freenet MetaData Browser allows to view the metadata of a given
 * key and to remove that key from the local datastore.
 * @author mihi
 */
public class MetaDataBrowser extends Frame implements ActionListener,
						      KeyListener {

    
    /**
     * The string shown at startup.
     */
    private static final String GREETING =
	"Welcome to FMDB "+Main.VERSION+"\n\n"+
	"Enter an URI into the text box below and press Return to view.\n"+
	"Place the Cursor in a line that contains a redirect URI and "+
	"press Return to follow it.\n"+
	"If that URI is a DateRedirect, you can set the date you want to "+
	"use for redirection in the text field below.\n"+
	"You can also remove files from your local datastore. For that, "+
	"visit the file and click the \"Zap\" button.\n"+
	"\n"+
	"Have Fun!\n\n\n\nFred says:\n";
    
    private String serverState;
    private TextArea view;
    private TextField uri, htl, date;
    private Button back, zap;
    private ArrayList oldUris = new ArrayList();
    private FCPConnection fcpConn;

    /**
     * Creates a new MetaData Browser window.
     * @param standalone if this metadata browser runs outside FIW.
     */
    public MetaDataBrowser(FCPConnection fcpConn, boolean standalone) {
	this(fcpConn, "SSK@rBjVda8pC-Kq04jUurIAb8IzAGcPAgM/TFE", standalone);
    }

    /**
     * Creates a new MetaData Browser window for the given URI.
     * @param startUri the URI to start with.
     * @param standalone if this metadata browser runs outside FIW.
     */
    public MetaDataBrowser(FCPConnection fcpConn,String startUri,
			   final boolean standalone) {
	super(fcpConn.getHost()+":"+fcpConn.getPort()+
	      " - Freenet MetaData Browser "+Main.VERSION+" (c) 2003 by mihi");
	this.fcpConn=fcpConn;
	Panel p,p2,p3,p4;
	Font font = new Font("Monospaced",Font.PLAIN,12);
	setIconImage(java.awt.Toolkit.getDefaultToolkit()
		     .createImage(Main.class.getResource("/res/icon2.gif")));
	add(view = new TextArea
	    ("",30,100,
	     TextArea.SCROLLBARS_VERTICAL_ONLY),BorderLayout.CENTER);
	view.setEditable(false);
	view.addKeyListener(this);
	view.setFont(font);
	setBackground(Color.lightGray);
	add(p=new Panel(new BorderLayout()),BorderLayout.SOUTH);
	p.add(p2=new Panel(new GridLayout(2,1)),BorderLayout.EAST);
	p2.add(new Label("Date:"));
	DateFormat df=new SimpleDateFormat("yyyy-MM-dd HH:mm");
	df.setTimeZone(TimeZone.getTimeZone("GMT"));
	p2.add(date=new TextField(df.format(new Date())));
	p.add(p2=new Panel(new GridLayout(2,1)),BorderLayout.CENTER);
	p2.add(p3=new Panel(new BorderLayout()));
	p3.add(back=new Button("Back"),BorderLayout.WEST);
	back.addActionListener(this);
	p3.add(p4=new Panel(new BorderLayout()),BorderLayout.CENTER);
	p4.add(zap=new Button("Zap"),BorderLayout.WEST);
	zap.addActionListener(this);
	p4.add(new Label("URI:"),BorderLayout.CENTER);
	p2.add(uri=new TextField("about:"));
	uri.addActionListener(this);
	p.add(p2=new Panel(new GridLayout(2,1)),BorderLayout.WEST);
	p2.add(new Label("HTL:"));
	p2.add(htl=new TextField("0"));
	htl.addActionListener(this);
	addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent evt) {
		    if (standalone) 
			System.exit(0);
		    else
			dispose();
		}
	    });
	pack();
	// this value is used in the status screen:
	serverState=fcpConn.handshake();
	if (serverState != null) {
	    followLink();
	    uri.setText(startUri);
	} else {
	    uri.setEditable(false);
	    uri.setText("noconn:");
	    followLink();
	}
	show();
    }

    public void actionPerformed (ActionEvent evt) {
	if (evt.getSource() == back) {
	    if (oldUris.size() > 1) {
		oldUris.remove(oldUris.size()-1);
		uri.setText((String)oldUris.get(oldUris.size()-1));
		oldUris.remove(oldUris.size()-1);
		followLink();
	    }
	} else if (evt.getSource()==zap) {
	    String urii=uri.getText();
	    if (urii.indexOf("@") != -1 && !urii.startsWith("invalid:")) {
		fcpConn.deleteKey(urii);
		oldUris.remove(oldUris.size()-1);
		followLink();
	    }
	} else {
	    followLink();
	}
    }

    public void keyPressed (KeyEvent evt) {
	if (evt.getKeyCode()==KeyEvent.VK_ENTER) {
	    String line = getLine(0);
	    if (line.startsWith("Redirect.Target=")) {
		uri.setText(line.substring(16));
		followLink();
	    } else if (line.startsWith("DateRedirect.Target=")) {
		String to = line.substring(20);
		long inc=86400, offs=0;
		for (int i=1;i<3;i++) {
		    String ll = getLine(-i);
		    if (ll.startsWith("DateRedirect.Increment=")) {
			inc=Integer.parseInt(ll.substring(23),16);
		    } else if (ll.startsWith("DateRedirect.Offset=")) {
			offs=Integer.parseInt(ll.substring(20),16);
		    } else {
			break;
		    }
		}
		for (int i=1;i<3;i++) {
		    String ll = getLine(i);
		    if (ll.startsWith("DateRedirect.Increment=")) {
			inc=Integer.parseInt(ll.substring(23),16);
		    } else if (ll.startsWith("DateRedirect.Offset=")) {
			offs=Integer.parseInt(ll.substring(20),16);
		    } else {
			break;
		    }
		}
		long now=getDateSeconds(date.getText())-offs;
		now =((now/inc)*inc) + offs;
		String slotnum = Integer.toHexString((int)now)+"-";
		if (to.startsWith("KSK@")) {
		    to="KSK@"+slotnum+to.substring(4);
		} else if (to.startsWith("freenet:KSK@")) {
		    to="KSK@"+slotnum+to.substring(4+8);
		} else if (to.startsWith("SSK@") ||
			   to.startsWith("freenet:SSK@")) {
		    int pp = to.indexOf("/");
		    to=to.substring(0,pp+1)+slotnum+to.substring(pp+1);
		} else {
		    to="/";
		}
		uri.setText(to);
		followLink();
	    } else if (line.indexOf("CHK@") != -1) {
		int p= line.indexOf("CHK@");
		uri.setText(line.substring(p));
		followLink();
	    } else {
		uri.setText("/");
		followLink();
	    }
	}
    }
    
    public void keyTyped (KeyEvent evt) {}
    public void keyReleased (KeyEvent evt) {}

    /**
     * Fetches a line of metadata
     * @param offset diffenence between current line (caret position)
     * and the line to fetch.
     * @return this line.
     */
    private String getLine(int offset) {
	int caret=view.getCaretPosition()+4;
	String line = "\n\n\n\n"+view.getText()+"\n\n\n\n";
	int pos = line.substring(0,caret).lastIndexOf("\n");
	for (int i=0;i>offset;i--) {
	    pos=line.substring(0,pos).lastIndexOf("\n");
	}
	line = line.substring(pos+1);
	for (int i=0;i<offset;i++) {
	    pos=line.indexOf("\n");
	    line=line.substring(pos+1);
	}
	pos = line.indexOf("\n");
	line = line.substring(0,pos);
	return line;
    }

    /**
     * Follows the link in the URI text box.
     */
    private void followLink() {
	String urii = uri.getText();
	view.setText("");
	if (urii.startsWith("freenet:")) {
	    urii=urii.substring(8);
	    uri.setText(urii);
	}
	oldUris.add(urii);
	if (urii.length()==0) {
	    view.setText("No URI specified");
	    return;
	} else if (urii.equals("/")) {
	    view.setText("No URI found. Move your cursor to lines "+
			 "containing the word \"Target\".");
	    return;
	} else if (urii.equals("about:")) {
	    view.setText(GREETING+serverState);
	    return;
	} else if (urii.equals("noconn:")) {
	    view.setText("Error: could not connect to fred.");
	    return;
	} else if (urii.startsWith("invalid:")) {
	    view.setText("Error: URIs start with SSK@, CHK@, KSK@ or SVK@");
	    return;
	} else if (!urii.startsWith("SSK@") && !urii.startsWith("KSK@") &&
		   !urii.startsWith("CHK@") && !urii.startsWith("SVK@")) {
	    uri.setText("invalid:"+urii);
	    view.setText("Error: URIs start with SSK@, CHK@, KSK@ or SVK@");
	    return;
	}
	
	int htlvalue;
	try {
	    htlvalue =  Integer.parseInt(htl.getText());
	} catch (NumberFormatException e) {
	    htlvalue=-1;
	}
	if (htlvalue < 0 || htlvalue > 100) {
	    view.setText("Error: Incorrect HTL.");
	    return;
	}
	long ttime=System.currentTimeMillis();
	FCPGetResult result = fcpConn.fetchMetadata(urii, htlvalue);
	ttime=System.currentTimeMillis()-ttime;

	if (result instanceof RouteNotFoundResult) {
	    RouteNotFoundResult rnfr = (RouteNotFoundResult) result;
	    view.setText("Error: RouteNotFound\n"+
			 "\nUnreachable: "+rnfr.getUnreachable()+
			 "\nRestarted: "+rnfr.getRestarted()+
			 "\nRejected: "+rnfr.getRejected());
	} else if (result instanceof DataNotFoundResult) {
	    view.setText("Error: DataNotFound");
	} else if (result instanceof URIErrorResult) {
	    view.setText("Error: URIError\nReason:-3\n"+
			 ((URIErrorResult)result).getReason());
	} else if (result instanceof GeneralErrorResult) {
	   view.setText("Error: Unexpected error:\n"+
			((GeneralErrorResult)result).getReason());
	} else if (result instanceof SocketTimeoutResult) {
	    view.setText("Error: Socket Timeout");
	} else if (!(result instanceof MetadataResult)) {
	    view.setText("Error: Unexpected error: check logfile.\n\n"+
			 "Result type: "+result.getClass().getName());
	} else {
	    String mdata = result.getPayload();
	    long len = ((MetadataResult)result).getDataLength();
	    if (mdata.length() == 0) {
		mdata="[no metadata]";
	    }
	    view.setText(mdata+ "\n----------\nRawDataLength="+ len); 
	}
	view.insert("Time taken: "+ttime+"\n\n",0);
    }

    /**
     * Calculates the number of seconds since the epoch for a given
     * Date object.
     * @param date The date.
     * @return the seconds.
     */
    private long getDateSeconds(String date) {
	boolean timePresent = (date.indexOf(" ") != -1);
	DateFormat df=new SimpleDateFormat(timePresent ? "yyyy-MM-dd HH:mm"
					               : "yyyy-MM-dd");
	df.setTimeZone(TimeZone.getTimeZone("GMT"));
	df.setLenient(true);
	try {
	Date dd =df.parse(date);
	return dd.getTime() / 1000;
	} catch (ParseException e) {
	    return 0;
	}
    }
}
