package fiw.addon;
import fiw.fcp.*;
import fiw.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

/**
 * This window shows current load status and statistics. Load status
 * is shown both as text and as charts. Freenet load percentage is
 * shown in the title bar as well.
 * @author mihi
 */
public class LoadStatus extends Frame implements Runnable {

    private GraphCanvas mainGraph, slowGraph;
    private TextArea disp, records;
    private FCPConnection fcpConn;

    /** Creates a new LoadStatus window
     * @param standalone if the utility is run outside of FIW
     */
    public LoadStatus(FCPConnection fcpConn, final boolean standalone) {
	super("Freenet Load Status");
	this.fcpConn=fcpConn;
	setIconImage(java.awt.Toolkit.getDefaultToolkit()
		     .createImage(Main.class.getResource("/res/icon1.gif")));
	setBackground(Color.lightGray);
	Panel p;
	add(disp=new TextArea(25,40),BorderLayout.CENTER);
	add(p=new Panel(new BorderLayout()),BorderLayout.WEST);
	p.add(new Label("Last 10 minutes:"),BorderLayout.NORTH);
	p.add(p=new Panel(new BorderLayout()),BorderLayout.CENTER);
	p.add(mainGraph = new GraphCanvas(),BorderLayout.NORTH);
	p.add(p=new Panel(new BorderLayout()),BorderLayout.CENTER);
	p.add(new Label("Last 5 hours:"),BorderLayout.NORTH);
	p.add(p=new Panel(new BorderLayout()),BorderLayout.CENTER);
	p.add(slowGraph = new GraphCanvas(),BorderLayout.NORTH);
	p.add(records=new TextArea(4,4),BorderLayout.CENTER);
	disp.setEditable(false);
	records.setEditable(false);
	new Thread(this).start();
	addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    if (standalone) 
			System.exit(0);
		    else
			dispose();
		}
	    });
	pack();
	show();
    }

    /**
     * Updates the status.
     */
    public void run() {
	try {
	    while(true) {
		String nodeInfo=fcpConn.getNodeInfo();
		int load=fcpConn.getNetworkLoad();
		if (load==-1) {
		    setTitle ("Not running - Freenet Load Status");
		    disp.setText("");
		    addValue(-1);
		} else {
 		    setTitle(load +"% - Freenet Load Status");
		    disp.setText(nodeInfo);
		    addValue(load);
		}
		Thread.sleep(2000);
	    }
	} catch (InterruptedException e) {
	    setTitle(e.toString());
	    e.printStackTrace();
	}
    }

    /**
     * Some vars to store statistics
     */
    private int valCount30=0, valMin=100, valMax=0;
    private long valSum=0, valCount=0;
    private int valMin30=100, valMax30=0;

    /**
     * adds a value to the statistics
     */
    private void addValue(int val) {
	int val2=val==-1?100:val;
	mainGraph.addValue(val2);
	    valCount30++;
	    if (valMin30>val2) valMin30=val2;
	    if (valMax30<val2) valMax30=val2;
	if (valCount30 == 30) { // 30 * 2 secs = 1 min
	    slowGraph.addValue(valMin30,valMax30);
	    valCount30=valMax30=0;
	    valMin30=100;
	}
	valSum+=val2;
	valCount++;
	if (val != -1) {
	    if (val<valMin) valMin=val;
	    if (val>valMax) valMax=val;
	}
	records.setText("Number of NodeInfo samples: "+valCount+"\n"+
			(valMin<=valMax?("Minimum load:     "+valMin+"%\n"+
			"Maximum load:     "+valMax+"%\n"):"")+
			"Average load:     "+(valSum/valCount)+"%\n"+
			"FCP host/port: "+fcpConn.getHost()+":"+
			fcpConn.getPort());
    }

    /**
     * A canvas used for drawing load charts
     */
    class GraphCanvas extends Canvas {

	private BufferedImage img;
	private int[] values=new int[300];
	private int[] values2=new int[300];
	private int pointer=0;
	private int lastPointer=0;

	/**
	 * Adds a value to the chart
	 * @param val the value
	 */
	public void addValue(int val) {
	    addValue(val,val);
	}

	/**
	 * Adds a value range to the chart. This is represented by a
	 * vertical line in the chart.
	 * @param val1 the minimum of the range
	 * @param val2 the maximum of the range
	 */
	public void addValue(int val1, int val2) {
	    values[pointer]=val1;
	    values2[pointer]=val2;
	    pointer++;
	    pointer%=300;
	    repaint();
	}

	/**
	 * overridden to avoid flickering
	 */
	public void update(Graphics g) {
	    paint(g);
	}

	/**
	 * Draws the chart. This uses double buffering to avoid
	 * flickering.
	 */
	public void paint(Graphics g) {
	    if (img==null) {
		img=new BufferedImage(310,110,BufferedImage.TYPE_INT_RGB);
		Graphics gg = img.getGraphics();
		gg.setColor(Color.lightGray);
		gg.fillRect(0,0,310,110);
		gg.setColor(Color.blue);
		gg.drawRect(4,4,301,102);
		gg.setColor(Color.white);
		gg.fillRect(5,5,300,100);
		gg.setColor(Color.black);
		gg.drawLine(5,105,304,105);
		gg.setColor(Color.green);
		gg.drawLine(5,5,304,5);
		gg.drawLine(5,30,304,30);
		gg.drawLine(5,55,304,55);				
		gg.drawLine(5,80,304,80);				
	    }
	    paint2(img.getGraphics());
	    g.drawImage(img,0,0,this);
	}

	/**
	 * Draws the chart onto the double buffer.
	 */
	private void paint2(Graphics g) {
	    g.setColor(Color.red);
	    g.drawLine(pointer+5,5,pointer+5,105);
	    while (lastPointer!= pointer) {
		int llp = lastPointer-1;
		if (lastPointer ==0) llp=0;
		g.setColor(Color.white);
		g.drawLine(lastPointer+5,5,lastPointer+5,105);
		g.setColor(Color.green);
		g.drawLine(lastPointer+5,5,lastPointer+5,5);
		g.drawLine(lastPointer+5,30,lastPointer+5,30);
		g.drawLine(lastPointer+5,55,lastPointer+5,55);
		g.drawLine(lastPointer+5,80,lastPointer+5,80);
		g.drawLine(lastPointer+5,105,lastPointer+5,105);
		g.setColor(Color.black);
		g.drawLine(llp+5,105-values[llp],
			   lastPointer+5,105-values[lastPointer]);
		g.drawLine(llp+5,105-values2[llp],
			   lastPointer+5,105-values2[lastPointer]);
		g.drawLine(lastPointer+5,105-values[lastPointer],
			   lastPointer+5,105-values2[lastPointer]);

		lastPointer=(lastPointer+1) % 300;
	    }
	}

	/**
	 * Returns the chart size.
	 * @return the chart size.
	 */
	public Dimension getPreferredSize() {
	    return new Dimension(310,110);
	}
    }
}
