//PROFILE-NO
unit SHA1;

// *****************************************************************************
// * Copyright 2003-2006 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

// this is based on the sample code provided in RFC 3174

interface

uses SysUtils,Classes;

type
  ESHA1Error = class (Exception);

  TSHA1Context = record
    H:        Array [0..4] of Cardinal;
    // Len_Lo:   Cardinal;
    // Len_Hi:   Cardinal;
    Length:   Int64;
    MsgIndex: Integer;
    MsgBlock: packed Array [0..63] of Byte;
    Computed: Boolean;
  end;
  PSHA1Context = ^TSHA1Context;

  TSHA1Digest = packed Array [0..19] of Byte;

procedure SHA1Reset(var Context: TSHA1Context);
procedure SHA1Input(var Context: TSHA1Context; pMsg: PChar; MsgLen: Cardinal);
procedure SHA1Result(var Context: TSHA1Context; var Digest: TSHA1Digest);
procedure SHA1Continue(var Context: TSHA1Context);

function  SHA1Test: String;
function  SHA1DigestToLoHexString(var Digest: TSHA1Digest): String;

procedure Util_CalcSHA1Digest(pData: Pointer; DataLen: Integer; var Digest: TSHA1Digest);
procedure Util_RollingHashPad(pOut: Pointer; PadLen: Cardinal; sha: TSHA1Context);

implementation

uses FuqidMemoryManager;

procedure SHA1Reset(var Context: TSHA1Context);
begin
  Context.H[0] := $67452301;
  Context.H[1] := $EFCDAB89;
  Context.H[2] := $98BADCFE;
  Context.H[3] := $10325476;
  Context.H[4] := $C3D2E1F0;
  // Context.Len_Lo   := 0;
  // Context.Len_Hi   := 0;
  Context.Length   := 0;
  Context.MsgIndex := 0;
  Context.Computed := False;
end;

function  SHA1RotL(const n: Byte; const val: Cardinal): Cardinal;
begin
  Result := (val shl n) or (val shr (32-n));
end;

function SHA1RotL_1(const val: Cardinal): Cardinal;
begin Result := (val shl 1) or (val shr 31); end;

function SHA1RotL_5(const val: Cardinal): Cardinal;
begin Result := (val shl 5) or (val shr 27); end;

function SHA1RotL_30(const val: Cardinal): Cardinal;
begin Result := (val shl 30) or (val shr 2); end;

procedure SHA1ProcessMessageBlock(var Context: TSHA1Context);
const
  K0 = $5A827999;
  K1 = $6ED9EBA1;
  K2 = $8F1BBCDC;
  K3 = $CA62C1D6;
var
  t:    Integer; 
  temp: Cardinal;
  W:    Array [0..79] of Cardinal;
  A,B,C,D,E: Cardinal;
begin
  for t := 0 to 15 do
    W[t] := Context.MsgBlock[t*4  ] shl 24
         or Context.MsgBlock[t*4+1] shl 16
         or Context.MsgBlock[t*4+2] shl  8
         or Context.MsgBlock[t*4+3];

  for t := 16 to 79 do
    W[t] := SHA1RotL_1(W[t-3] xor W[t-8] xor W[t-14] xor W[t-16]);

  A := Context.H[0];
  B := Context.H[1];
  C := Context.H[2];
  D := Context.H[3];
  E := Context.H[4];

  for t := 0 to 19 do begin
    temp := (Int64(SHA1RotL_5(A)) + ((B and C) or ((not B) and D)) + E + W[t] + K0) and $FFFFFFFF;
    E := D; D := C; C := SHA1RotL_30(B); B := A; A := temp;
  end;
  for t := 20 to 39 do begin
    temp := (Int64(SHA1RotL_5(A)) + (B xor C xor D) + E + W[t] + K1) and $FFFFFFFF;
    E := D; D := C; C := SHA1RotL_30(B); B := A; A := temp;
  end;
  for t := 40 to 59 do begin
    temp := (Int64(SHA1RotL_5(A)) + ((B and C) or (B and D) or (C and D)) + E + W[t] + K2) and $FFFFFFFF;
    E := D; D := C; C := SHA1RotL_30(B); B := A; A := temp;
  end;
  for t := 60 to 79 do begin
    temp := (Int64(SHA1RotL_5(A)) + (B xor C xor D) + E + W[t] + K3) and $FFFFFFFF;
    E := D; D := C; C := SHA1RotL_30(B); B := A; A := temp;
  end;

  Context.H[0] := (Int64(Context.H[0]) + A) and $FFFFFFFF;
  Context.H[1] := (Int64(Context.H[1]) + B) and $FFFFFFFF;
  Context.H[2] := (Int64(Context.H[2]) + C) and $FFFFFFFF;
  Context.H[3] := (Int64(Context.H[3]) + D) and $FFFFFFFF;
  Context.H[4] := (Int64(Context.H[4]) + E) and $FFFFFFFF;
  Context.MsgIndex := 0;
end;

procedure SHA1Input(var Context: TSHA1Context; pMsg: PChar; MsgLen: Cardinal);
begin
  Assert(not Context.Computed);
  while MsgLen > 0 do begin
    Context.MsgBlock[Context.MsgIndex] := Byte(pMsg^);
    inc(Context.MsgIndex);
    inc(Context.Length,8);
    if Context.MsgIndex = 64 then SHA1ProcessMessageBlock(Context);
    inc(pMsg); dec(MsgLen);
  end;
end;

procedure SHA1PadMessage(var Context: TSHA1Context);
var
  bits: packed Array [0..7] of Byte;
  i:    Integer;
  b:    Byte;
begin
  for i := 0 to 7 do
    bits[i] := (Context.Length shr ((7-i) shl 3)) and $FF;

  b := $80; SHA1Input(Context, @b, 1);
  b := $00; while Context.MsgIndex <> 56 do SHA1Input(Context, @b, 1);
  SHA1Input(Context, @bits, 8);
  {
  if Context.MsgIndex > 55 then begin
    Context.MsgBlock[Context.MsgIndex] := $80; inc(Context.MsgIndex);
    while Context.MsgIndex < 64 do begin
      Context.MsgBlock[Context.MsgIndex] := 0; inc(Context.MsgIndex);
    end;
    SHA1ProcessMessageBlock(Context);
  end else begin
    Context.MsgBlock[Context.MsgIndex] := $80; inc(Context.MsgIndex);
  end;
  while Context.MsgIndex < 56 do begin
    Context.MsgBlock[Context.MsgIndex] := 0; inc(Context.MsgIndex);
  end;
  Context.MsgBlock[56] := (Context.Len_Hi shr 24) and $FF;
  Context.MsgBlock[57] := (Context.Len_Hi shr 16) and $FF;
  Context.MsgBlock[58] := (Context.Len_Hi shr  8) and $FF;
  Context.MsgBlock[59] := (Context.Len_Hi       ) and $FF;
  Context.MsgBlock[60] := (Context.Len_Lo shr 24) and $FF;
  Context.MsgBlock[61] := (Context.Len_Lo shr 16) and $FF;
  Context.MsgBlock[62] := (Context.Len_Lo shr  8) and $FF;
  Context.MsgBlock[63] := (Context.Len_Lo       ) and $FF;
  }
end;

procedure SHA1Result(var Context: TSHA1Context; var Digest: TSHA1Digest);
var i: Integer;
begin
  if not Context.Computed then begin
    SHA1PadMessage(Context);
    // FillChar(Context.MsgBlock, SizeOf(Context.MsgBlock), 0); // wipe data from memory
    // Context.Len_Lo := 0; Context.Len_Hi := 0;
    Context.Computed := True;
  end;
  for i := 0 to 19 do
    Digest[i] := (Context.H[i shr 2] shr (8 * (3 - (i and 3)))) and $FF;
end;

procedure SHA1Continue(var Context: TSHA1Context);
begin
  Assert(Context.Computed);
  Context.Computed := False;
end;

function  SHA1Test: String;
// test procedure
const
  testarray: Array [0..3] of String = (
    'abc',
    'abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq',
    'a',
    '0123456701234567012345670123456701234567012345670123456701234567'
  );
  repeatcount: Array [0..3] of Integer = (1,1,1000000,10);
  resultarray: Array [0..3] of String = (
    'A9 99 3E 36 47 06 81 6A BA 3E 25 71 78 50 C2 6C 9C D0 D8 9D',
    '84 98 3E 44 1C 3B D2 6E BA AE 4A A1 F9 51 29 E5 E5 46 70 F1',
    '34 AA 97 3C D4 C4 DA A4 F6 1E EB 2B DB AD 27 31 65 34 01 6F',
    'DE A3 56 A2 CD DD 90 C7 A7 EC ED C5 EB B5 63 93 4F 46 04 52'
  );

var
  sha: TSHA1Context;
  i,j: Integer;
  Digest: TSHA1Digest;
  sRes:   String;
begin                                   
  sRes := '';
  for j := 0 to 3 do begin
    sRes := sRes + Format('Test %d: %d, "%s"'#13,[j+1,repeatcount[j],testarray[j]]);
    SHA1Reset(sha);
    for i := 0 to repeatcount[j]-1 do
      SHA1Input(sha, @(testarray[j][1]), Length(testarray[j]));
    SHA1Result(sha, Digest);
    sRes := sRes + #9;
    for i := 0 to 19 do sRes := sRes + IntToHex(Digest[i],2) + ' ';
    sRes := sRes + #13'Should match:'#13;
    sRes := sRes + #9 + resultarray[j] + #13;
  end;
  Result := sRes;
end;

function  SHA1DigestToLoHexString(var Digest: TSHA1Digest): String;
var i: Integer;
begin
  Result := '';
  for i := 0 to 19 do Result := Result + IntToHex(Digest[i],2);
  Result := LowerCase(Result);
end;

procedure Util_CalcSHA1Digest(pData: Pointer; DataLen: Integer; var Digest: TSHA1Digest);
var sha: TSHA1Context;
begin
  SHA1Reset(sha);
  SHA1Input(sha, pData, DataLen);
  SHA1Result(sha, Digest);
end;

procedure Util_RollingHashPad(pOut: Pointer; PadLen: Cardinal; sha: TSHA1Context);
const DIGLEN = SizeOf(TSHA1Digest);
var
  pWorkBuf,pW,pO: PChar;
  WorkBufLen,Len: Cardinal;
  dig:            TSHA1Digest;
begin
  if PadLen = 0 then exit;
  WorkBufLen := 0; Len := PadLen;
  while Len > 0 do begin
    inc(WorkBufLen, DIGLEN);
    if WorkBufLen > Len then Len := 0 else dec(Len, WorkBufLen);
  end;
  gMemMgr.GetMem(Pointer(pWorkBuf), WorkBufLen);
  try
    WorkBufLen := 0; pW := pWorkBuf; pO := pOut;
    while PadLen > 0 do begin
      SHA1Result(sha, dig);
      SHA1Continue(sha); 
      SHA1Input(sha, @dig, DIGLEN);
      Move(dig, pW^, DIGLEN); inc(pW, DIGLEN); inc(WorkBufLen, DIGLEN);
      if PadLen < WorkBufLen then Len := PadLen else Len := WorkBufLen;
      Move(pWorkBuf^, pO^, Len); inc(pO, Len); dec(PadLen, Len);
    end;
  finally
    gMemMgr.FreeMem(Pointer(pWorkBuf));
  end;
end;

end.
