//PROFILE-NO
unit IncompleteFile;

// *****************************************************************************
// * Copyright 2003-2006 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, QueueFile, BaseFormUnit;

type
  TSegmentControls = record
    LbSegment: TLabel;
    LbDB:      TLabel;
    LbCB:      TLabel;
    LbNB:      TLabel;
    LbStatus:  TLabel;
  end;

  TFrmIncompleteFile = class(TBaseForm)
    PanelStats: TPanel;
    LbTemplSeg: TLabel;
    LbDataBlocks: TLabel;
    LbCheckBlocks: TLabel;
    LbNeededBlocks: TLabel;
    LbStatus: TLabel;
    LbTemplDB: TLabel;
    LbTemplCB: TLabel;
    LbTemplNB: TLabel;
    LbTemplStatus: TLabel;
    PanelControl: TPanel;
    Label1: TLabel;
    EdSavename: TEdit;
    BtBrowse: TButton;
    BtSave: TButton;
    BtCancel: TButton;
    LbFirstBlock: TLabel;
    LbLastBlock: TLabel;
    LbFirstBlockInfo: TLabel;
    LbLastBlockInfo: TLabel;
    Label2: TLabel;
    RbMissingSkip: TRadioButton;
    RbMissingZero: TRadioButton;
    RbMissingCopy: TRadioButton;
    SaveDlg: TSaveDialog;
    procedure FormCreate(Sender: TObject);
    procedure BtBrowseClick(Sender: TObject);
    procedure BtSaveClick(Sender: TObject);
  private
    FGetQFile:    TGetQueueFile;
    FSegControls: Array of TSegmentControls;
    FFirstLast:   Array [0..1] of Integer;
    procedure Init;
    procedure SetAnchors;
  public
  end;

procedure SaveIncompleteFile(pG: TGetQueueFile);
  
implementation

uses FreenetUtils, WaitForThread, Misc, Settings, Main;

{$R *.dfm}

procedure SaveIncompleteFile(pG: TGetQueueFile);
// assumes it is save to use the PrepFile !
// ticker should be locked, and no threads working on the file
var frm: TFrmIncompleteFile;
begin
  if pG.PrepBasename = '' then raise Exception.Create('Need at least some blocks to save incomplete file');

  // not if the file is active! - otherwise we can get a race condition in DecodeThread!
  if pG.NumActiveThreads <> 0 then raise Exception.Create('There are active threads for this file. Abort the download first!');

  frm := TFrmIncompleteFile.Create(Application);
  try
    frm.FGetQFile := pG;
    frm.Init;
    frm.ShowModal;
  finally
    frm.Free;
  end;
end;

{ TFrmIncompleteFile }

procedure TFrmIncompleteFile.FormCreate(Sender: TObject);
var
  i:    Integer;
  ctrl: TControl;
  s:    String;
begin
  SetAnchors;
  // init hints - replace # by #13
  for i := 0 to ComponentCount-1 do
    if Components[i] is TControl then begin
      ctrl := Components[i] as TControl;
      s := ctrl.Hint;
      while Pos('#',s) <> 0 do s[Pos('#',s)] := #13;
      ctrl.Hint := s;
    end;
end;

procedure TFrmIncompleteFile.Init;

  function CreateLabelFromTemplate(Template: TLabel; TopOffset: Integer): TLabel;
  begin
    Result := TLabel.Create(Self);
    Result.Parent    := Template.Parent;
    Result.Alignment := Template.Alignment;
    Result.AutoSize  := Template.AutoSize;
    Result.Caption   := Template.Caption;
    Result.Left      := Template.Left;
    Result.Top       := Template.Top + TopOffset;
    Result.Width     := Template.Width;
    Result.Visible   := True;
  end;

var
  PrepFile:  TPreparedFile;
  pG:        TGetQueueFile;
  iSeg,iBlk: Integer;
  dY,dH,i:   Integer;
  NumDB,NumCB,NumTO,NumNB: Integer;
  s:         String;
  col:       TColor;
  Lb:        TLabel;
begin
  pG := FGetQFile;

  // Savename
  EdSavename.Text := ExtractFilePath(pG.SavFilename) + '_INCOMPLETE_' + ExtractFileName(pG.SavFilename);

  // Stats
  PrepFile := TPreparedFile.CreateDownload(pG.PrepBasename);
  try
    PrepFile.ReadHeader;

    // create segment controls
    if PrepFile.NumSegments <= 0 then raise Exception.CreateFmt('WTF!? NumSegments = %d', [PrepFile.NumSegments]);
    SetLength(FSegControls, PrepFile.NumSegments);
    dY := LbTemplSeg.Top - LbDataBlocks.Top;
    dH := Integer(PrepFile.NumSegments - 1) * dY;
    Height := Height + dH;
    for iSeg := 0 to PrepFile.NumSegments-1 do begin
      FSegControls[iSeg].LbSegment := CreateLabelFromTemplate(LbTemplSeg,    iSeg*dY);
      FSegControls[iSeg].LbDB      := CreateLabelFromTemplate(LbTemplDB,     iSeg*dY);
      FSegControls[iSeg].LbCB      := CreateLabelFromTemplate(LbTemplCB,     iSeg*dY);
      FSegControls[iSeg].LbNB      := CreateLabelFromTemplate(LbTemplNB,     iSeg*dY);
      FSegControls[iSeg].LbStatus  := CreateLabelFromTemplate(LbTemplStatus, iSeg*dY);
    end;

    // init segment info
    for iSeg := 0 to PrepFile.NumSegments-1 do begin
      NumDB := PrepFile.DataBlocksCompletedInSeg[iSeg];
      NumCB := PrepFile.CheckBlocksCompletedInSeg[iSeg];
      NumNB := PrepFile.BlocksRequiredInSeg[iSeg];
      NumTO := NumDB + NumCB;
      if      NumDB >= NumNB then begin s := 'Complete';   col := clGreen;  end
      else if NumTO >= NumNB then begin s := 'Decodeable'; col := clTeal;   end
      else if NumTO >  0     then begin s := 'Partial';    col := clOlive;  end
      else                        begin s := 'N/A';        col := clMaroon; end
      ;

      FSegControls[iSeg].LbSegment.Caption   := 'Segment ' + IntToStr(iSeg + 1);
      FSegControls[iSeg].LbDB.Caption        := IntToStr(NumDB);
      FSegControls[iSeg].LbCB.Caption        := IntToStr(NumCB);
      FSegControls[iSeg].LbNB.Caption        := IntToStr(NumNB);
      FSegControls[iSeg].LbStatus.Caption    := s;
      FSegControls[iSeg].LbStatus.Font.Color := col;

      if iSeg = 0 then begin
        iBlk := 0;
        if      PrepFile.BlockDone[iBlk] then FFirstLast[0] := 2
        else if NumTO >= NumNB           then FFirstLast[0] := 1
        else                                  FFirstLast[0] := 0;
      end;
      if iSeg = Integer(PrepFile.NumSegments)-1 then begin
        iBlk := PrepFile.LastDataBlock;
        if      PrepFile.BlockDone[iBlk] then FFirstLast[1] := 2
        else if NumTO >= NumNB           then FFirstLast[1] := 1
        else                                  FFirstLast[1] := 0;
      end;
    end;

    // Info about first/last block
    LbFirstBlock.Top     := LbFirstBlock.Top     + dH;
    LbFirstBlockInfo.Top := LbFirstBlockInfo.Top + dH;
    LbLastBlock.Top      := LbLastBlock.Top      + dH;
    LbLastBlockInfo.Top  := LbLastBlockInfo.Top  + dH;
    for i := 0 to 1 do begin
      if i = 0 then Lb := LbFirstBlockInfo else Lb := LbLastBlockInfo;
      case FFirstLast[i] of
        2:   begin s := 'Available';  col := clGreen;  end;
        1:   begin s := 'Decodeable'; col := clTeal;   end;
        else begin s := 'N/A';        col := clMaroon; end;
      end;
      Lb.Caption    := s;
      Lb.Font.Color := col;
    end;

  finally
    PrepFile.Free;
  end;
end;

procedure TFrmIncompleteFile.SetAnchors;
begin
  // Set anchors for controls that stick to the right or bottom of their parent.
  // Normally this would be done at design time, but D7 then saves .DesignSize,
  // breaking compatibility with earlier Delphi versions. So do it at runtime.
end;

procedure TFrmIncompleteFile.BtBrowseClick(Sender: TObject);
begin
  SaveDlg.FileName := EdSavename.Text;
  if not ExecuteSaveDialogSafely(SaveDlg) then exit;
  EdSavename.Text := SaveDlg.FileName;
end;

procedure TFrmIncompleteFile.BtSaveClick(Sender: TObject);
var
  pG: TGetQueueFile;
  PrepFile: TPreparedFile;
  s,sName: String;
  oldcur: TCursor;
  WaitForm:  TFrmWaitForThread;
  DecThread: TFECDecodeThread;
  ok:        Boolean;
  ps:        TPartialSave;
begin
  pG := FGetQFile;

  ok := False;
  sName := EdSavename.Text;
  if sName = '' then raise Exception.Create('No filename');
  if FileExists(sName) then begin
    if MessageDlg('File already exists. Overwrite?', mtWarning, [mbYes,mbAbort], 0) <> mrYes then exit;
  end;

  oldcur := Screen.Cursor;
  WaitForm := nil; DecThread := nil;
  PrepFile := TPreparedFile.CreateDownload(pG.PrepBasename);
  try
    Screen.Cursor := crHourglass;
    PrepFile.ReadHeader;

    // check if we have (or at least can decode) the first/last data block
    if (FFirstLast[0] = 0) and (FFirstLast[1] = 0) then
      s := 'Neither the first nor the last data block is available.'
    else if FFirstLast[0] = 0 then
      s := 'The first data block is not available.'
    else if FFirstLast[1] = 0 then
      s := 'The last data block is not available.'
    else
      s := '';
    if s <> '' then begin
      s := s
         + #13 + 'Depending on the filetype this may render the saved file useless.'
         + #13 + 'Save anyway?';
      if MessageDlg(s, mtWarning, [mbYes,mbAbort], 0) <> mrYes then exit;
    end;

    if      RbMissingZero.Checked then ps := psZeroMissing
    else if RbMissingCopy.Checked then ps := psCopyLastGood
    else                               ps := psSkipMissing;
    
    // fully decode any segments we have enough blocks for
    DecThread := TFECDecodeThread.Create(
                   MakeThreadID(ExtractFileName(sName), 'Decode'),
                   GlobalSettings.FCPAddr, GlobalSettings.FCPPort,
                   PrepFile.Basename,
                   sName,
                   0,
                   ps,
                   FrmMain.StatusCallback,
                   pG, pG.UniqueID,
                   True
                 );
    WaitForm := TFrmWaitForThread.Create(Self);
    WaitForm.LaunchAndWait( DecThread );

    if DecThread.DecodeResult then begin
      MessageDlg('Incomplete file saved as ' + sName, mtInformation, [mbOk], 0);
      ok := True;
    end else
      MessageDlg('Incomplete file not saved.', mtError, [mbOk], 0);
    // PrepFile.SaveOutputFile(sName, nil, True);
  finally
    PrepFile.Free;
    Screen.Cursor := oldcur;
    WaitForm.Free;
    DecThread.Free;
  end;
  if ok then ModalResult := mrOk;
end;

end.
