unit FuqidMemoryManager;

// *****************************************************************************
// * Copyright 2003-2006 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses Windows,SysUtils,Classes,SyncObjs;

const
  FUQIDMEMORYMANAGER_SAFETYSIZE = 8 * 128; // must be dividable by 8 !
type
  TFuqidMemoryManagerBlock = record
    UserAddress: Pointer;
    UserSize:    Integer;
    RealAddress: Pointer;
  end;
  PFuqidMemoryManagerBlock = ^TFuqidMemoryManagerBlock;

  TFuqidMemoryManagerCallback = procedure (Sender: TObject; AMessage: String) of object;
  TFuqidMemoryManager = class
  private
    FSafety:    Pointer;
    FAllocList: TList;
    FCritSect:  TCriticalSection;
    FOnError:   TFuqidMemoryManagerCallback;
    procedure AddSafetyBoundary(pMem: PFuqidMemoryManagerBlock);
    procedure VerifySafetyBoundary(pMem: PFuqidMemoryManagerBlock);
    function  FindMemBlock(pUserAddress: Pointer): Integer;
    procedure ReportError(pMem: PFuqidMemoryManagerBlock; AMessage: String);
  public
    constructor Create;
    destructor Destroy; override;
    procedure GetMem(var P: Pointer; Size: Integer);
    procedure FreeMem(var P: Pointer);
    procedure ReallocMem(var P: Pointer; Size: Integer);
    property  OnError: TFuqidMemoryManagerCallback read FOnError write FOnError;
  end;

var gMemMgr: TFuqidMemoryManager = nil;

implementation

{ TFuqidMemoryManager }

constructor TFuqidMemoryManager.Create;
const
  SAFETYDATA: packed Array [0..1] of DWord = ($DEADBEEF,$ABADCAFE);
var
  p: PChar;
  i: Integer;
begin
  inherited;
  FCritSect := TCriticalSection.Create;
  FAllocList := TList.Create;
  Assert(FUQIDMEMORYMANAGER_SAFETYSIZE mod SizeOf(SAFETYDATA) = 0);
  System.GetMem(FSafety, FUQIDMEMORYMANAGER_SAFETYSIZE);
  p := FSafety;
  for i := 0 to (FUQIDMEMORYMANAGER_SAFETYSIZE div SizeOf(SAFETYDATA)) - 1 do begin
    Move(SAFETYDATA, p^, SizeOf(SAFETYDATA));
    inc(p, SizeOf(SAFETYDATA));
  end;
end;

destructor TFuqidMemoryManager.Destroy;
var i: Integer;
begin
  for i := 0 to FAllocList.Count-1 do
    ReportError(FAllocList.Items[i], 'Unfreed memory');
  FAllocList.Free;
  System.FreeMem(FSafety);
  FCritSect.Free;
  inherited;
end;

procedure TFuqidMemoryManager.AddSafetyBoundary(pMem: PFuqidMemoryManagerBlock);
var p: Pointer;
begin
  p := pMem^.RealAddress;
  Move(FSafety^, p^, FUQIDMEMORYMANAGER_SAFETYSIZE);
  p := PChar(pMem^.UserAddress) + pMem^.UserSize;
  Move(FSafety^, p^, FUQIDMEMORYMANAGER_SAFETYSIZE);
end;

procedure TFuqidMemoryManager.VerifySafetyBoundary(pMem: PFuqidMemoryManagerBlock);
var p: Pointer;
begin
  p := pMem^.RealAddress;
  if not CompareMem(p, FSafety, FUQIDMEMORYMANAGER_SAFETYSIZE) then
    ReportError(pMem, 'Buffer underrun detected');
  p := PChar(pMem^.UserAddress) + pMem^.UserSize;
  if not CompareMem(p, FSafety, FUQIDMEMORYMANAGER_SAFETYSIZE) then
    ReportError(pMem, 'Buffer overrun detected');
end;

function TFuqidMemoryManager.FindMemBlock(pUserAddress: Pointer): Integer;
var i: Integer;
begin
  for i := 0 to FAllocList.Count-1 do begin
    if PFuqidMemoryManagerBlock(FAllocList.Items[i])^.UserAddress = pUserAddress then begin
      Result := i; exit;
    end;
  end;
  Result := -1;
  ReportError(nil, Format('Memory block allocated at 0x%p not found',[pUserAddress]));
end;


procedure TFuqidMemoryManager.GetMem(var P: Pointer; Size: Integer);
var pMem: PFuqidMemoryManagerBlock;
begin
  FCritSect.Enter;
  try
    New(pMem);
    try
      pMem^.UserSize := Size;
      System.GetMem(pMem^.RealAddress, pMem^.UserSize + 2*FUQIDMEMORYMANAGER_SAFETYSIZE);
      pMem^.UserAddress := PChar(pMem^.RealAddress) + FUQIDMEMORYMANAGER_SAFETYSIZE;
      AddSafetyBoundary(pMem);
      P := pMem^.UserAddress;
      FAllocList.Add(pMem);
      //ReportError(pMem, 'GETMEM');
    except
      Dispose(pMem);
      raise;
    end;
  finally
    FCritSect.Leave;
  end;
end;

procedure TFuqidMemoryManager.FreeMem(var P: Pointer);
var
  pMem: PFuqidMemoryManagerBlock;
  idx:  Integer;
begin
  FCritSect.Enter;
  try
    idx := FindMemBlock(P);
    if idx < 0 then begin
      System.FreeMem(P);
    end else begin
      pMem := FAllocList.Items[idx];
      VerifySafetyBoundary(pMem);
      System.FreeMem(pMem^.RealAddress);
      FAllocList.Delete(idx);
      //ReportError(pMem, 'FREEMEM');
      Dispose(pMem);
    end;
  finally
    FCritSect.Leave;
  end;
end;

procedure TFuqidMemoryManager.ReallocMem(var P: Pointer; Size: Integer);
var
  pMem: PFuqidMemoryManagerBlock;
  idx:  Integer;
  pNew: Pointer;
begin
  FCritSect.Enter;
  try
    if P = nil then begin
      if Size <> 0 then Self.GetMem(P, Size);
    end else begin
      idx := FindMemBlock(P);
      if idx < 0 then begin
        System.ReallocMem(P,Size)
      end else if Size = 0 then begin
        Self.FreeMem(P);
        P := nil;
      end else begin
        pMem := FAllocList.Items[idx];
        VerifySafetyBoundary(pMem);
        pNew := pMem^.RealAddress;
        System.ReallocMem(pNew, Size + 2*FUQIDMEMORYMANAGER_SAFETYSIZE);
        pMem^.RealAddress := pNew;
        pMem^.UserAddress := PChar(pNew) + FUQIDMEMORYMANAGER_SAFETYSIZE;
        pMem^.UserSize    := Size;
        AddSafetyBoundary(pMem);
        P := pMem^.UserAddress;
      end;
    end;
  finally
    FCritSect.Leave;
  end;
end;

procedure TFuqidMemoryManager.ReportError(pMem: PFuqidMemoryManagerBlock; AMessage: String);
begin
  if not Assigned(FOnError) then exit;
  if Assigned(pMem) then
    AMessage := AMessage + Format(' [0x%p %d bytes]',[pMem^.UserAddress, pMem^.UserSize]);
  FOnError(Self, AMessage);
end;


end.
