//PROFILE-NO
unit FECUtils;

// *****************************************************************************
// * Copyright 2003-2006 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

// partially based on freenet's wrapper for onionnetwork's java fec codec

{$INCLUDE CompilerOpts.pas}

interface

uses SysUtils, Classes;

const
  ALGONAME_ONIONFEC_A_1_2 = 'OnionFEC_a_1_2';

type
  EFECError = class(Exception);

  TPointerArray = Array of Pointer;
  TIntegerArray = Array of Integer;

  TFECSegParams = record
    n,k:         Integer;
    SegLength:   Integer;
    BlockSize:   Integer;
    StripeWidth: Integer;
  end;

  TFECSegmentHeader = record
    AlgoName:         String;
    FileLength:       Int64;
    Offset:           Int64;
    BlocksRequired:   Integer;
    BlockCount:       Integer;
    BlockSize:        Integer;
    CheckBlockCount:  Integer;
    CheckBlockSize:   Integer;
    DataBlockOffset:  Integer;
    CheckBlockOffset: Integer;
    Segments:         Integer;
    SegmentNum:       Integer;
  end;

  TFECBase = class
  private
    FRedNum, FRedDen: Integer;
    FAlgoName:        String;

    FNumSegments: Integer;
    FFileSize:    Integer;
    FSegParams:   Array [Boolean] of TFECSegParams; // index: is last segment?
    procedure GetSegParams(FileSize: Int64);
  public
    constructor Create(AAlgoName: String; ARedNum, ARedDen: Integer);
    procedure   Init(FileSize: Int64);

    function  GetSegmentHeader(Seg: Integer): TFECSegmentHeader; // Seg: 0-based

    property  NumSegments: Integer read FNumSegments;
  end;

  TFECEncoder = class(TFECBase)
  public
    procedure Encode(SegNum: Integer; DataStream: TStream; CheckBlocks: TList; Requested: TList = nil; OutputStream: TStream = nil; OutputOffsets: TList = nil);
  end;

  TFECDecoder = class(TFECBase)
  public
    procedure Decode(SegNum: Integer; DataStream: TStream; AvailableBlocks: TList; AllBlockOffsets: TList);
  end;

function  IsNativeFECPossible(AlgoName: String): Boolean;
function  CreateFECEncoderFromName(AlgoName: String): TFECEncoder;
function  CreateFECDecoderFromName(AlgoName: String): TFECDecoder;

implementation

uses Windows, FreenetUtils, FEC, FuqidMemoryManager;

{ Util functions }

function  IsNativeFECPossible(AlgoName: String): Boolean;
begin
  Result := (AlgoName = ALGONAME_ONIONFEC_A_1_2);
end;

function  CreateFECEncoderFromName(AlgoName: String): TFECEncoder;
begin
  if AlgoName = ALGONAME_ONIONFEC_A_1_2 then
    Result := TFECEncoder.Create(AlgoName,1,2)
  else
    Result := nil;
end;

function  CreateFECDecoderFromName(AlgoName: String): TFECDecoder;
begin
  if AlgoName = ALGONAME_ONIONFEC_A_1_2 then
    Result := TFECDecoder.Create(AlgoName,1,2)
  else
    Result := nil;
end;

{ TFECBase }

const
  SEGLEN_MAX = 128 * 1024 * 1024;
  SEGLEN_MIN = 6 * 128 * 1024;

constructor TFECBase.Create(AAlgoName: String; ARedNum, ARedDen: Integer);
begin
  inherited Create;
  Assert(AAlgoName = ALGONAME_ONIONFEC_A_1_2, 'Unsupported FEC algorithm');
  FAlgoName := AAlgoName;
  FRedNum   := ARedNum;
  FRedDen   := ARedDen;
end;

procedure TFECBase.GetSegParams(FileSize: Int64);
var
  bLastSeg: Boolean;
  S:        TFECSegParams;
  cnt:      Cardinal;
  len:      Int64;
begin
  FNumSegments := (FileSize + SEGLEN_MAX - 1) div SEGLEN_MAX;
  FFileSize    := FileSize;
  for bLastSeg := False to True do begin
    if bLastSeg and (FNumSegments > 1) then
      len := FileSize mod FSegParams[False].SegLength
    else
      len := FileSize;

    if      len > SEGLEN_MAX then S.SegLength := SEGLEN_MAX
    else if len < SEGLEN_MIN then S.SegLength := SEGLEN_MIN
    else                          S.SegLength := len;

    if      S.SegLength < (   1024*1024) then S.BlockSize :=  128*1024
    else if S.SegLength < (32*1024*1024) then S.BlockSize :=  256*1024
    else if S.SegLength < (64*1024*1024) then S.BlockSize :=  512*1024
    else                                      S.BlockSize := 1024*1024;

    if S.SegLength < SEGLEN_MAX then
      S.k := (S.SegLength + S.BlockSize - 1) div S.BlockSize
    else
      S.k := SEGLEN_MAX div S.BlockSize;

    S.n := S.k + (S.k * FRedNum div FRedDen);
    if S.n = S.k then inc(S.n);

    if S.SegLength < (16*1024*1024) then
      S.StripeWidth := -1
    else begin
      cnt := 1;
      while (Cardinal(S.SegLength) div cnt) > (16*1024*1024) do cnt := cnt shl 1;
      S.StripeWidth := Cardinal(S.BlockSize) div cnt;
    end;

    if S.k > 128 then raise EFECError.Create('k > 128');
    if (2 * FRedNum) > FRedDen then raise EFECError.Create('Max. 50% redundancy');

    FSegParams[bLastSeg] := S;
  end;
end;

procedure TFECBase.Init(FileSize: Int64);
begin
  GetSegParams(FileSize);
end;

function TFECBase.GetSegmentHeader(Seg: Integer): TFECSegmentHeader;
var bLast: Boolean;
begin
  Assert((Seg >= 0) and (Seg < FNumSegments), 'Invalid segment');

  bLast := (Seg = FNumSegments-1);
  Result.AlgoName          := FAlgoName;
  Result.FileLength        := FFileSize;
  Result.Offset            := Seg * FSegParams[False].SegLength;
  Result.BlocksRequired    := FSegParams[bLast].k;
  Result.BlockCount        := Result.BlocksRequired;
  Result.BlockSize         := FSegParams[bLast].BlockSize;
  Result.CheckBlockCount   := FSegParams[bLast].n - Result.BlocksRequired;
  Result.CheckBlockSize    := FSegParams[bLast].BlockSize;
  Result.DataBlockOffset   := Seg * FSegParams[False].k;
  Result.CheckBlockOffset  := Seg * (FSegParams[False].n - FSegParams[False].k);
  Result.Segments          := FNumSegments;
  Result.SegmentNum        := Seg + 1; // 1-based in header
end;

type
  TPointerPointer = ^Pointer;
  TIntegerPointer = ^Integer;

{ TFECEncoder }

procedure TFECEncoder.Encode(SegNum: Integer; DataStream: TStream; CheckBlocks,Requested: TList; OutputStream: TStream; OutputOffsets: TList);
// DataStream: must already be positioned to begin of segment offset
// CheckBlocks: list of pointers; Requested: list of integers
// CheckBlocks must be initialized to either nil or allocated memory
// if nil, mem is allocated here and the caller is responsible for freeing it
// CheckBlocks can also be nil -> will be allocated, caller has to free mem & list
// Requested can be nil or empty; in that case all checkblocks are generated
// the indices in requested are full block indices, i.e. k is the first checkblock
// CheckBlocks-Array is sorted like Request list;
//   i.e. if Requested = [9,6,5,8] then CheckBlocks[0] = block#9, and so on.
//   if no request list was given, it's initialized to [k..n-1]
//
// if OutputStream and OutputOffsets are given:
//   CheckBlocks must be nil
//   Results are directly saved to OutputStream
var
  bCreatedReq:   Boolean;
  bCreatedCheck: Boolean;
  bLastSeg:    Boolean;
  i,idx,NumCheck,NumData:  Integer;
  AllocList:   TList;
  p,pDat:       Pointer;
  pSrc,pDst,pStripedDst,pp,pp2: TPointerPointer;
  pIdx,pI:      TIntegerPointer;
  Code:         Pointer;
  StripeWidth:  Integer;
  NumStripes:   Integer;
  StreamStart,StreamSize, Offset, LenLeft: Int64;
  Len:          Integer;
  iStripe:      Integer;
  bFreeAlloced: Boolean;
begin
  Assert((SegNum >= 0) and (SegNum < FNumSegments), 'Invalid segment');
  Assert((OutputStream = nil) = (OutputOffsets = nil), 'Invalid parameters');
  Assert((OutputStream = nil) or (CheckBlocks = nil), 'Invalid parameters');

  bLastSeg := (SegNum = FNumSegments-1);
  NumData  := FSegParams[bLastSeg].k;
  NumCheck := FSegParams[bLastSeg].n - FSegParams[bLastSeg].k;

  AllocList := nil; pSrc := nil; pDst := nil; pStripedDst := nil; pIdx := nil; Code := nil;
  bCreatedReq   := (Requested = nil);
  bCreatedCheck := (CheckBlocks = nil);
  bFreeAlloced  := False;
  try
    try
      if bCreatedReq   then Requested   := TList.Create;

      if Requested.Count = 0 then
        for i := 0 to NumCheck-1 do
          Requested.Add(Pointer(FSegParams[bLastSeg].k + i));

      if (Requested.Count = 0) or (Requested.Count > NumCheck) then
        raise EFECError.Create('You asked for ridiculous check block indices');
      for i := 0 to Requested.Count-1 do
        if (Integer(Requested.Items[i]) <  FSegParams[bLastSeg].k)
        or (Integer(Requested.Items[i]) >= FSegParams[bLastSeg].n) then
          raise EFECError.Create('You asked for ridiculous check block indices');

      if bCreatedCheck then begin
        CheckBlocks := TList.Create;
        for i := 0 to Requested.Count-1 do CheckBlocks.Add(nil);
      end;

      if (CheckBlocks = nil) or (CheckBlocks.Count <> Requested.Count) then
        raise EFECError.Create('Wrong number of entries in the checkblocks list');

      if (OutputOffsets <> nil) and (OutputOffsets.Count <> Requested.Count) then
        raise EFECError.Create('Wrong number of entries in the output offsets list');

      if FSegParams[bLastSeg].StripeWidth = -1 then begin
        StripeWidth := FSegParams[bLastSeg].BlockSize;
        NumStripes := 1;
      end else begin
        StripeWidth := FSegParams[bLastSeg].StripeWidth;
        NumStripes := FSegParams[bLastSeg].BlockSize div StripeWidth;
      end;

      // alloc unallocated checkblocks
      // if we do direct streaming just alloc StripeWidth for each block and free mem on exit;
      if OutputStream = nil then
        Len := FSegParams[bLastSeg].BlockSize
      else begin
        Len := StripeWidth; bFreeAlloced := True;
      end;
      for i := 0 to Requested.Count-1 do begin
        if CheckBlocks.Items[i] = nil then begin
          if AllocList = nil then AllocList := TList.Create;
          gMemMgr.GetMem(p, Len);
          CheckBlocks.Items[i] := p;
          AllocList.Add(Pointer(i));
        end;
      end;

      // set up pointer arrays
      gMemMgr.GetMem(Pointer(pSrc), NumData * SizeOf(Pointer));
      FillChar(pSrc^, NumData * SizeOf(Pointer), 0);
      pp := pSrc;
      for i := 0 to NumData-1 do begin
        gMemMgr.GetMem(pDat, StripeWidth);
        pp^ := pDat; inc(pp);
      end;

      gMemMgr.GetMem(Pointer(pDst), Requested.Count * SizeOf(Pointer));
      gMemMgr.GetMem(Pointer(pIdx), Requested.Count * SizeOf(Integer));
      pp := pDst; pI := pIdx;
      for i := 0 to Requested.Count-1 do begin
        pp^ := CheckBlocks.Items[i]; inc(pp);
        pI^ := Integer(Requested.Items[i]); inc(pI);
      end;

      if NumStripes > 1 then gMemMgr.GetMem(Pointer(pStripedDst), Requested.Count * SizeOf(Pointer));

      Code := FECCode_New(FSegParams[bLastSeg].k, FSegParams[bLastSeg].n);

      StreamStart := DataStream.Position;
      StreamSize  := DataStream.Size;
      if NumStripes = 1 then begin
        pp := pSrc; LenLeft := StreamSize - StreamStart;
        for i := 0 to NumData-1 do begin
          pDat := pp^;
          Len := StripeWidth;
          if Len > LenLeft then begin
            Len := LenLeft; FillChar(pDat^, StripeWidth, 0);
          end;
          if Len > 0 then DataStream.ReadBuffer(pDat^, Len);
          inc(pp); dec(LenLeft, Len);
        end;
        FECCode_Encode(Code, pSrc, pDst, pIdx, Requested.Count, StripeWidth);
        if OutputStream <> nil then begin
          pp := pDst;
          for i := 0 to Requested.Count-1 do begin
            pDat := pp^;
            OutputStream.Seek( DWord(OutputOffsets.Items[i]), soFromBeginning );
            OutputStream.WriteBuffer( pDat^, StripeWidth );
            inc(pp);
          end;
        end;
      end else begin
        for iStripe := 0 to NumStripes-1 do begin
          Offset := StreamStart + iStripe*StripeWidth;
          if OutputStream = nil then begin
            pp := pDst; pp2 := pStripedDst;
            for i := 0 to Requested.Count-1 do begin
              pDat := pp^;
              pp2^ := PChar(pDat) + iStripe*StripeWidth;
              inc(pp); inc(pp2);
            end;
          end else
            Move(pDst^, pStripedDst^, Requested.Count * SizeOf(Pointer));
          pp := pSrc;
          for i := 0 to NumData-1 do begin
            pDat := pp^;
            if Offset >= StreamSize then
              FillChar(pDat^, StripeWidth, 0)
            else begin
              Len := StripeWidth;
              if Offset + Len > StreamSize then begin
                FillChar(pDat^, StripeWidth, 0);
                Len := StreamSize - Offset;
              end;
              DataStream.Seek(Offset, soFromBeginning);
              DataStream.ReadBuffer(pDat^, Len);
            end;
            inc(pp); inc(Offset, FSegParams[bLastSeg].BlockSize);
          end;
          FECCode_Encode(Code, pSrc, pStripedDst, pIdx, Requested.Count, StripeWidth);
          if OutputStream <> nil then begin
            pp := pStripedDst;
            for i := 0 to Requested.Count-1 do begin
              pDat := pp^;
              OutputStream.Seek( DWord(OutputOffsets.Items[i]) + DWord(iStripe*StripeWidth), soFromBeginning );
              OutputStream.WriteBuffer( pDat^, StripeWidth );
              inc(pp);
            end;
          end;
        end;
      end;

    except
      bFreeAlloced := True;
      raise;
    end;
  finally
    if bFreeAlloced then begin
      if AllocList <> nil then begin
        while AllocList.Count > 0 do begin
          p := CheckBlocks.Items[Integer(AllocList.Items[0])];
          CheckBlocks.Items[Integer(AllocList.Items[0])] := nil;
          gMemMgr.FreeMem(p); AllocList.Delete(0);
        end;
      end;
      if bCreatedCheck then CheckBlocks.Free;
    end;
    AllocList.Free;
    if pSrc <> nil then begin
      pp := pSrc;
      for i := 0 to NumData-1 do begin
        pDat := pp^; if pDat <> nil then gMemMgr.FreeMem(pDat);
        inc(pp);
      end;
      gMemMgr.FreeMem(Pointer(pSrc));
    end;
    if pStripedDst <> nil then gMemMgr.FreeMem(Pointer(pStripedDst));
    if pDst <> nil then gMemMgr.FreeMem(Pointer(pDst));
    if pIdx <> nil then gMemMgr.FreeMem(Pointer(pIdx));
    FECCode_Free(Code);
    if bCreatedReq then Requested.Free;
  end;
end;

{ TFECDecoder }

procedure TFECDecoder.Decode(SegNum: Integer; DataStream: TStream; AvailableBlocks, AllBlockOffsets: TList);
// DataStream:      in/out; area for missing data blocks must already be prewritten.
// AvailableBlocks: block numbers of exactly k available data and check blocks (0..n-1)
// AllBlockOffsets: offset in DataStream for all blocks; ordered by block number
// note: AllBlockOffsets[i] isn't used if (i >= k) and block[i] is not available
var
  bLastSeg:    Boolean;
  pSrc,pp:     TPointerPointer;
  pIdx,pI:     TIntegerPointer;
  pDat:        Pointer;
  i,BlockNum:  Integer;
  ok:          Boolean;
  TmpList:     TList;
  StripeWidth: Integer;
  NumStripes:  Integer;
  iStripe:     Integer;
  StreamSize, Offset:  Int64;
  Len:         Integer;
  Code:        Pointer;
begin
  Assert((SegNum >= 0) and (SegNum < FNumSegments), 'Invalid segment');

  bLastSeg := (SegNum = FNumSegments-1);
  Assert(AllBlockOffsets.Count = FSegParams[bLastSeg].n, 'Invalid parameters');
  Assert(AvailableBlocks.Count = FSegParams[bLastSeg].k, 'Invalid parameters');

  pSrc := nil; pIdx := nil; TmpList := nil; Code := nil;
  try
    // some sanity checks
    TmpList := TList.Create;
    ok := False;
    for i := 0 to FSegParams[bLastSeg].n-1 do TmpList.Add(nil);
    for i := 0 to AvailableBlocks.Count-1 do begin
      BlockNum := Integer(AvailableBlocks.Items[i]);
      if (BlockNum < 0) or (BlockNum >= FSegParams[bLastSeg].n) then
        raise EFECError.CreateFmt('Invalid block number (%d)',[BlockNum]);
      if BlockNum >= FSegParams[bLastSeg].k then ok := True;
      if TmpList.Items[BlockNum] <> nil then
        raise EFECError.CreateFmt('Block number specified twice (%d)',[BlockNum]);
      TmpList.Items[BlockNum] := Pointer(1);
    end;
    if not ok then raise EFECError.Create('You already have all data blocks');

    // TmpList: 1 for each available block, 0 for unavailable

    if FSegParams[bLastSeg].StripeWidth = -1 then begin
      StripeWidth := FSegParams[bLastSeg].BlockSize;
      NumStripes := 1;
    end else begin
      StripeWidth := FSegParams[bLastSeg].StripeWidth;
      NumStripes := FSegParams[bLastSeg].BlockSize div StripeWidth;
    end;

    // set up decoding arrays
    gMemMgr.GetMem(Pointer(pSrc), FSegParams[bLastSeg].k * SizeOf(Pointer));
    FillChar(pSrc^, FSegParams[bLastSeg].k * SizeOf(Pointer), 0);
    pp := pSrc;
    for i := 0 to FSegParams[bLastSeg].k-1 do begin
      gMemMgr.GetMem(pDat, StripeWidth);
      pp^ := pDat; inc(pp);
    end;
    gMemMgr.GetMem(Pointer(pIdx), FSegParams[bLastSeg].k * SizeOf(Integer));

    Code := FECCode_New(FSegParams[bLastSeg].k, FSegParams[bLastSeg].n);

    StreamSize := DataStream.Size;
    for iStripe := 0 to NumStripes-1 do begin
      // load data
      pp := pSrc; pI := pIdx;
      for i := 0 to AvailableBlocks.Count-1 do begin
        pDat := pp^;
        BlockNum := Integer(AvailableBlocks.Items[i]);
        Offset   := DWord(AllBlockOffsets.Items[BlockNum]) + DWord(iStripe*StripeWidth);
        if Offset > StreamSize then
          FillChar(pDat^, StripeWidth, 0)
        else begin
          Len := StripeWidth;
          if (Offset + Len) > StreamSize then begin
            FillChar(pDat^, StripeWidth, 0);
            Len := StreamSize - Offset;
          end;
          DataStream.Seek(Offset, soFromBeginning);
          DataStream.ReadBuffer(pDat^, Len);
        end;
        pI^ := BlockNum;
        inc(pp); inc(pI);
      end;
      // decode
      if not FECCode_Decode(Code, pSrc, pIdx, StripeWidth) then
        raise EFECError.Create('Decoding failed');
      // save results
      pp := pSrc; pI := pIdx;
      for i := 0 to FSegParams[bLastSeg].k-1 do begin
        BlockNum := pI^;
        Assert(BlockNum < FSegParams[bLastSeg].k, 'BlockNum >= k after decoding');
        if TmpList.Items[BlockNum] = nil then begin
          pDat := pp^;
          Offset := DWord(AllBlockOffsets.Items[BlockNum]) + DWord(iStripe*StripeWidth);
          if Offset + StripeWidth > StreamSize then
            raise EFECError.Create('Invalid block offset - would write past end of file');
          DataStream.Seek(Offset, soFromBeginning);
          DataStream.WriteBuffer(pDat^, StripeWidth);
        end;
        inc(pp); inc(pI);
      end;
    end;


  finally
    if pSrc <> nil then begin
      pp := pSrc;
      for i := 0 to FSegParams[bLastSeg].k-1 do begin
        pDat := pp^;
        if pDat <> nil then gMemMgr.FreeMem(pDat);
        inc(pp);
      end;
      gMemMgr.FreeMem(Pointer(pSrc));
    end;
    if pIdx <> nil then gMemMgr.FreeMem(Pointer(pIdx));
    TmpList.Free;
    FECCode_Free(Code);
  end;

end;

end.
