//PROFILE-NO
unit ClipboardFormats;

// *****************************************************************************
// * Copyright 2003-2006 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses SysUtils,Classes,ComCtrls,FreenetUtils;

const
  TT_ON  = '***TT_ON***';
  TT_OFF = '***TT_OFF***';

  CLIPFORMATSQUICKHELP =
    'Format strings consist of a sequence of plain text and keywords.'#13#10+
    'Text has to be placed between doublequotes ("), keywords between percent characters (%).'#13#10+
    'To get the actual characters " % and \ in text, prefix them with \ (like: \" \% \\).'#13#10 +
    '(See below for a list of all allowed escape sequences).'#13#10+
    #13#10+
    #13#10+
    'Keywords:'#13#10+
    'General format: % ["leadtext"] [!] <token> ["trailtext"] %'#13#10+
    #13#10+
    '<token> usually expands to a part of the key. (See below for a list of valid tokens)'#13#10+
    'If it is prefixed by a ! the raw key format is used, otherwise the key is URL-encoded.'#13#10+
    'For example: raw: CHK@.../my file #1.txt ; url-encoded: CHK@.../my+file+%231.txt'#13#10+
    #13#10+
    'Leadtext and trailtext are plain text strings.'#13#10+
    'If the token evaluates to something other than an empty string,'#13#10+
    'these texts are placed before/after the token result.'#13#10+
    'Otherwise they are not used.'#13#10+
    #13#10+
    'Some tokens need parameters. These follow the token, enclosed in parantheses.'#13#10+
    #13#10+
    #13#10+
    'Valid tokens:'#13#10 +
    TT_ON +
    ' fulluri      complete uri (example: freenet:SSK@gdahsgh,sfhghs/blah//index.html)'#13#10+
    ' uri          same as %uri% but without the freenet: prefix'#13#10+
    ' key          for CHK-keys: key without document name (other keytypes: like %uri%)'#13#10+
    ' comment      optional comment'#13#10+
    ' keytype      CHK,SSK,KSK (without @)'#13#10+
    ' routingkey   routing key part'#13#10+
    ' cryptokey    crypto key part'#13#10+
    ' metainfo     metainfo part'#13#10+
    ' docname      document name'#13#10+
    ' metastring   metastring'#13#10+
    ' size[("format"[,"dec.places"[,"punctuation?"[,"separators"]]])]  filesize. formats: b,k,M,G'#13#10+
    TT_OFF +
    #13#10+
    'Interactive tokens:'#13#10+
    TT_ON +
    'askstring("title"[,"default"])%   ask the user for a string'#13#10+
    'askyesno("title"[,"default"])%    ask the user for a yes/no decision (default can be Y or N).'#13#10+
    '                                  If yes is chosen this expression evaluates to the leadtext,'#13#10+
    '                                  else to the trailtext.'#13#10+
    TT_OFF +
    'If no default is specified, and multiple keys are formatted then the '+
    'value entered for the previous key is used as default answer for the next key.'#13#10+
    #13#10+
    'Special tokens for formatting multiple keys at once:'#13#10+
    TT_ON +
    '%nomulti%            if this token appears, attempts to format multiple keys are refused'#13#10+
    '%separator("text")%  defines the separator between formatted keys (default: newline)'#13#10+
    TT_OFF +
    #13#10+
    #13#10+
    'Allowed escape sequences in text:'#13#10+
    TT_ON +
    ' \n      carriage return + linefeed (= newline)'#13#10+
    ' \r      carriage return'#13#10+
    ' \t      tab'#13#10+
    ' \\      the backslash character (\)'#13#10+
    ' \%      the percent charcter (%)'#13#10+
    ' \"      the quotes character (")  (only needed inside keywords, see below)'#13#10+
    ' \0x##   where ##  are two hex digits: the character code with hex representation ##'#13#10+
    ' \###    where ### are three octal digits: the character code with octal representation ###'#13#10+
    TT_OFF;


// clipboard format strings:
// % is used for keywords, use \% to get the percent character
// \ is used for special characters; use \\ to get a plain \
//   \n      carriage return + linefeed (= newline)
//   \r      carriage return
//   \t      tab
//   \\      the backslash character (\)
//   \%      the percent charcter (%)
//   \"      the quotes character (")  (only needed inside keywords, see below)
//   \0x##   where ##  are two hex digits: the character code with hex representation ##
//   \###    where ### are three octal digits: the character code with octal representation ###
//
// all plain text must be quoted; spaces between keywords/text are allowed
// for better readability
//
// keywords
//
// general format: % ["leadtext"] [!] token ["trailtext"] %
//
// %keytype%      CHK,SSK,KSK (without @)
// %routingkey%   routing key part
// %cryptokey%    crypto key part
// %metainfo%     metainfo part
// %docname%      document name
// %metastring%   metastring
// %comment%      comment
//
// %fulluri%      complete uri (example: freenet:SSK@gdahsgh,sfhghs/blah//index.html)
// %uri%          same as %uri% but without the freenet: prefix
// %key%          for CHK-keys: key without document name
//                for other keytypes: like %uri%
//
// %size[("format","dec.places","punctuation?")]% filesize. formats: b,k,M,G
//
// %askstring("title"[,"default"])%   ask the user for a string
// %askyesno("title"[,"default"])%    ask the user for a yes/no decision (default can be Y or N)
//                                    note: if user chooses yes this expression evaluates to
//                                      the leadtext, else to the trailtext
// if no default is specified, and multiple keys are formatted then the
// last entered value is used as default.
//
// %text("text")%       only used internally
//
// special keywords for formatting multiple keys at once:
// %nomulti%            if this keyword appears, attempts to format multiple keys are refused
// %separator("text")%  defines the separator between formatted keys (default: linefeed)
//
// special characters in keywords:
//  keywords return the respective part URL-encoded.
//  if you want the raw value, use ! as first letter, like %!uri%
// if you want some text only when the keyword does not evaluate to an
//  empty string, use "text" after the beginning or before the ending %.
//  example: %"\nDocument name: "docname%
//    if no document name is present in the key this evaluates to nothing
//    else to a newline, followed by the text "Document name: " (without quotes)
//    followed by the document name of the key
//  if you need the quotes character inside this conditional text, use \"

type
  TKeywordType = (
    kwKeytype, kwRoutingkey, kwCryptokey, kwMetainfo, kwDocname, kwMetastring,
    kwFullURI, kwURI, kwKey, kwAskString, kwAskYesNo, kwNoMulti, kwSeparator, kwText,
    kwSize, kwComment
  );

  EKeywordError = class(Exception);

  TKeyword = class
  private
    FType:       TKeywordType;
    FLeadText:   String;
    FTrailText:  String;
    FRawFormat:  Boolean;
    FParams:     Array of String;
    FSubKeyword: TKeyword;
    FLastAnswer: record
                   AskString: String;
                   AskYesNo:  Boolean;
                 end;
  public
    destructor Destroy; override;
    procedure Init(var Expression: String);
    function  Execute(uri: TFreenetURI; FileSize: Int64; Comment: String): String;
  end;

  TClipboardFormat = class
  private
    FKeywords:    TList;
    FNoMulti:     Boolean;
    FSeparator:   String;
    FDescription: String;
    procedure FreeKeywords;
  public
    constructor Create;
    destructor  Destroy; override;
    procedure Init(ADescription,Expression: String);
    function  FormatKeys(URIList: TList; FileSizeList: TList; CommentList: TStringList): String;
  end;

procedure QuickHelpToRichEd(RichEd: TRichEdit);
function  EscapeString(const S: String): String;


implementation

uses Windows,Dialogs;

{ Util funcs }

procedure QuickHelpToRichEd(RichEd: TRichEdit);
var
  iPos:   Integer;
  s,sRem: String;
begin
  RichEd.Clear;
  sRem := CLIPFORMATSQUICKHELP;
  while sRem <> '' do begin
    iPos := Pos(TT_ON, sRem); if iPos = 0 then iPos := Length(sRem) + 1;
    s := Copy(sRem,1,iPos-1); Delete(sRem,1,iPos + Length(TT_ON) - 1);
    RichEd.SelAttributes.Name := 'MS Sans Serif';
    RichEd.SelText := s;
    RichEd.SelStart := RichEd.SelStart + Length(s);

    if sRem = '' then break;

    iPos := Pos(TT_OFF, sRem); if iPos = 0 then iPos := Length(sRem) + 1;
    s := Copy(sRem,1,iPos-1); Delete(sRem,1,iPos + Length(TT_OFF) - 1);
    RichEd.SelAttributes.Name := 'Courier New';
    RichEd.SelText := s;
    RichEd.SelStart := RichEd.SelStart + Length(s);
  end;
end;

function  EscapeString(const S: String): String;
var i: Integer;
begin
  Result := '';
  i := 1;
  while i <= Length(S) do begin
    if S[i] = #13 then begin
      if (i < Length(S)) and (S[i+1] = #10) then begin
        inc(i); Result := Result + '\n';
      end else
        Result := Result + '\r';
    end else if S[i] = #9 then
      Result := Result + '\t'
    else if (S[i] < #32) or (S[i] > #128) then
      Result := Result + '\0x' + IntToHex(Ord(S[i]),2)
    else if S[i] in ['%','"','\'] then
      Result := Result + '\' + S[i]
    else
      Result := Result + S[i];
    inc(i);
  end;
end;

function  UnescapeString(const S: String): String;
const HEXDIGS = '0123456789ABCDEF';
var
  i,j,base,digs,val,digval: Integer;
  sub: String;
begin
  Result := '';
  i := 1;
  while i <= Length(S) do begin
    if S[i] = '\' then begin
      inc(i); if i > Length(S) then raise EKeywordError.Create('Invalid escape sequence');
      case UpCase(S[i]) of
        'N': Result := Result + #13#10;
        'R': Result := Result + #13;
        'T': Result := Result + #9;
        '0'..'9':
             begin
               if UpperCase(Copy(s,1,2)) = '0X' then begin
                 inc(i,2); base := 16; digs := 2;
               end else begin
                 base := 8; digs := 3;
               end;
               sub := UpperCase(Copy(s,i,digs)); inc(i,digs-1);
               if Length(sub) < digs then raise EKeywordError.Create('Invalid escape sequence');
               val := 0;
               for j := 1 to digs do begin
                 digval := Pos(sub[j], HEXDIGS) - 1;
                 if (digval < 0) or (digval >= base) then raise EKeywordError.Create('Invalid escape sequence');
                 val := val * base + digval;
               end;
               Result := Result + Chr(val);
             end;
        else Result := Result + S[i];
      end;
    end else
      Result := Result + S[i];
    inc(i);
  end;
end;

function  GetQuotedStringEndPos(const S: String; iStartPos: Integer): Integer;
// returns the position of the end quote of a quoted string (starting from iStartPos)
var i: Integer;
begin
  if Copy(S,iStartPos,1) <> '"' then raise EKeywordError.Create('Quoted string expected');
  i := iStartPos+1;
  while i <= Length(S) do begin
    if S[i] = '\' then
      inc(i)
    else if S[i] = '"' then begin
      Result := i; exit;
    end;
    inc(i);
  end;
  raise EKeywordError.Create('Unterminated string');
end;

function  RemoveQuotedString(var S: String): String;
// remove quoted string from S; return the removed string as result (unquoted)
var i: Integer;
begin
  i := GetQuotedStringEndPos(S,1);
  Result := Copy(S,2,i-2); Delete(S,1,i);
end;

{ TKeyword }

procedure TKeyword.Init(var Expression: String);
// on return Expression contains the rest of the string
const
  KwTab: Array [TKeywordType] of record Token:String; MinParms,MaxParms: Integer; end = (
           (Token:'Keytype';     MinParms: 0; MaxParms: 0;),
           (Token:'Routingkey';  MinParms: 0; MaxParms: 0;),
           (Token:'Cryptokey';   MinParms: 0; MaxParms: 0;),
           (Token:'Metainfo';    MinParms: 0; MaxParms: 0;),
           (Token:'Docname';     MinParms: 0; MaxParms: 0;),
           (Token:'Metastring';  MinParms: 0; MaxParms: 0;),
           (Token:'FullURI';     MinParms: 0; MaxParms: 0;),
           (Token:'URI';         MinParms: 0; MaxParms: 0;),
           (Token:'Key';         MinParms: 0; MaxParms: 0;),
           (Token:'AskString';   MinParms: 1; MaxParms: 2;),
           (Token:'AskYesNo';    MinParms: 1; MaxParms: 2;),
           (Token:'NoMulti';     MinParms: 0; MaxParms: 0;),
           (Token:'Separator';   MinParms: 1; MaxParms: 1;),
           (Token:'Text';        MinParms: 1; MaxParms: 1;),
           (Token:'Size';        MinParms: 0; MaxParms: 4;),
           (Token:'Comment';     MinParms: 0; MaxParms: 0;)
         );

  function  ExtractKeywordFromExpression: String;
  var
    i:   Integer;
    S:   String;
    ok:  Boolean;
  begin
    // find end of keyword
    ok := False; S := '';
    i := 2;
    while i <= Length(Expression) do begin
      if Expression[i] = '%' then begin
        ok := True;
        S := Trim(Copy(Expression,2,i-2));
        Delete(Expression,1,i);
        break;
      end else if Expression[i] = '"' then
        i := GetQuotedStringEndPos(Expression, i) + 1
      else
        inc(i);
    end;
    if not ok then
      raise EKeywordError.Create('Invalid keyword (ending % missing)');
    Result := Trim(S);
  end;

var
  i:   Integer;
  S:   String;
  ok:  Boolean;
  kwt: TKeywordType;
begin
  if Copy(Expression,1,1) <> '%' then
    raise EKeywordError.Create('Invalid keyword (% missing)');

  S := ExtractKeywordFromExpression;

  // allow ! before or after leadtext
  FRawFormat := False;
  if Copy(S,1,1) = '!' then begin
    FRawFormat := True; S := Trim(Copy(S,2,Length(S)-1));
  end;

  FLeadText := '';  // leadtext
  if Copy(S,1,1) = '"' then begin
    FLeadText := UnescapeString(RemoveQuotedString(S));
    S := Trim(S);
  end;

  // ! directly before keyword?
  if Copy(S,1,1) = '!' then begin
    FRawFormat := True; S := Trim(Copy(S,2,Length(S)-1));
  end;

  // the keyword itself (token)
  i := 0; ok := False;
  for kwt := Low(KwTab) to High(KwTab) do begin
    if  (Length(KwTab[kwt].Token) > i)
    and (CompareText(Copy(s,1,Length(KwTab[kwt].Token)),KwTab[kwt].Token) = 0) then begin
      i := Length(KwTab[kwt].Token); FType := kwt; ok := True;
    end;
  end;
  if not ok then raise EKeywordError.Create('Invalid keyword');
  S := Trim(Copy(S,i+1,Length(S)));

  // params
  SetLength(FParams, 0);
  if Copy(S,1,1) = '(' then begin
    S := Trim(Copy(S,2,Length(S)));
    while True do begin
      if Copy(S,1,1) = ')' then begin
        S := Trim(Copy(S,2,Length(S)));
        break;
      end;
      if Length(FParams) > 0 then begin
        if Copy(S,1,1) <> ',' then raise EKeywordError.Create('Invalid parameters (comma expected)');
        S := Trim(Copy(S,2,Length(S)));
      end;
      if Copy(S,1,1) = '"' then begin
        SetLength(FParams, Length(FParams)+1);
        FParams[High(FParams)] := UnescapeString(RemoveQuotedString(S));
        S := Trim(S);
      end else if (S = '') and (FType = kwAskString) and (Length(FParams) = 1) then begin
        // param may be a keyword itself - (valid only as 2nd arg to %askstring%)
        Insert('%',Expression,1); // we removed the start % mistakenly, so put it back
        Assert(FSubKeyword = nil, 'Subkeyword already assigned');
        FSubKeyword := TKeyword.Create;
        FSubKeyword.Init(Expression);
        SetLength(FParams, Length(FParams)+1);
        FParams[High(FParams)] := '';
        Insert('%',Expression,1);
        S := ExtractKeywordFromExpression;
      end else
        raise EKeywordError.Create('Invalid parameters (quotes missing)');
    end;
  end;

  if Length(FParams) < KwTab[FType].MinParms then raise EKeywordError.Create('Too few parameters');
  if Length(FParams) > KwTab[FType].MaxParms then raise EKeywordError.Create('Too many parameters');
  if (FType = kwAskYesNo) and (Length(FParams) >= 2) then begin
    if (UpperCase(FParams[1]) <> 'Y') and (UpperCase(FParams[1]) <> 'N') then
      raise EKeywordError.Create('Invalid default value for askyesno (valid: "Y" or "N")');
  end;
  if (FType = kwSize) and (Length(FParams) >= 1) then begin
    if (Length(FParams[0]) <> 1) or not (UpCase(FParams[0][1]) in ['B','K','M','G']) then
      raise EKeywordError.Create('Invalid format for size (valid: "b","k","M","G")');
  end;
  if (FType = kwSize) and (Length(FParams) >= 2) then begin
    i := StrToIntDef(FParams[1], -1);
    if i < 0 then raise EKeywordError.Create('Invalid number of decimal places for size');
  end;
  if (FType = kwSize) and (Length(FParams) >= 3) then begin
    if (UpperCase(FParams[2]) <> 'Y') and (UpperCase(FParams[2]) <> 'N') then
      raise EKeywordError.Create('Invalid punctuation value for size (valid: "Y" or "N")');
  end;
  if (FType = kwSize) and (Length(FParams) >= 4) then begin
    if Length(FParams[3]) <> 2then
      raise EKeywordError.Create('Invalid separator values for size (must be exactly 2 characters)');
  end;

  FTrailText := '';  // trailtext
  if Copy(S,1,1) = '"' then begin
    FTrailText := UnescapeString(RemoveQuotedString(S));
    S := Trim(S);
  end;

  if S <> '' then raise EKeywordError.Create('Invalid keyword (text after expected end of keyword)');

  FLastAnswer.AskString := '';
  FLastAnswer.AskYesNo  := True;
end;

function TKeyword.Execute(uri: TFreenetURI; FileSize: Int64; Comment: String): String;
var
  b: Boolean;
  s: String;
  i: Integer;
  bNoLeadTrail: Boolean;
  sFmt: String;
  val:  Extended;
  DecSep,ThoSep: Char;
  OldDec,OldTho: Char;
begin
  bNoLeadTrail := False;
  b := uri.DontURLEncode;
  uri.DontURLEncode := FRawFormat;
  case FType of
    kwKeytype:    Result := uri.KeyType;
    kwRoutingkey: Result := uri.RoutingKeyB64;
    kwCryptokey:  Result := uri.CryptoKeyB64;
    kwMetainfo:   Result := uri.MetaInfo;
    kwDocname:    Result := uri.DocumentName;
    kwMetastring: Result := uri.MetaString;
    kwFullURI:    Result := uri.GetURI;
    kwURI:        Result := uri.GetURI(False);
    kwKey:        Result := uri.GetURI(False,False,False);
    kwAskString:  begin
                    // second parameter may be a subkeyword
                    if Assigned(FSubKeyword) then
                      s := FSubKeyword.Execute(uri, FileSize, Comment)
                    else if Length(FParams) >= 2 then
                      s := FParams[1]
                    else
                      s := FLastAnswer.AskString;
                    if not InputQuery('Formatting key', FParams[0], s) then raise EAbort.Create('Aborted');
                    Result := s; FLastAnswer.AskString := s;
                  end;
    kwAskYesNo:   begin
                    bNoLeadTrail := True;
                    if Length(FParams) >= 2 then begin
                      if UpperCase(FParams[1]) = 'Y' then i := MB_DEFBUTTON1 else i := MB_DEFBUTTON2;
                    end else begin
                      if FLastAnswer.AskYesNo        then i := MB_DEFBUTTON1 else i := MB_DEFBUTTON2;
                    end;
                    s := FParams[0];
                    i := MessageBox(0, PChar(s), 'Formatting key', MB_YESNOCANCEL or MB_ICONQUESTION or i or MB_TASKMODAL or MB_SETFOREGROUND);
                    if i = IDYES then Result := FLeadText
                    else if i = IDNO then Result := FTrailText
                    else raise EAbort.Create('Aborted');
                  end;
    kwNoMulti:    Result := '';
    kwSeparator:  Result := '';
    kwText:       Result := FParams[0];
    kwSize:       begin
                    if FileSize = 0 then
                      Result := ''
                    else begin
                      val := FileSize;
                      if Length(FParams) > 0 then begin
                        if      UpperCase(FParams[0]) =  'K' then val := FileSize / 1024
                        else if UpperCase(FParams[0]) =  'M' then val := FileSize / (1024*1024)
                        else if UpperCase(FParams[0]) =  'G' then val := FileSize / (1024*1024*1024)
                        else if UpperCase(FParams[0]) <> 'B' then raise EKeywordError.Create('Invalid format');
                      end;
                      if Length(FParams) > 1 then begin
                        i := StrToIntDef(FParams[1],-1);
                        if i < 0 then raise EKeywordError.Create('Invalid decimal places');
                      end else
                        i := 0;
                      sFmt := '%.' + IntToStr(i);
                      if Length(FParams) > 2 then begin
                        if UpperCase(FParams[2]) = 'Y' then sFmt := sFmt + 'n' else sFmt := sFmt + 'f';
                      end else
                        sFmt := sFmt + 'f';
                      DecSep := DecimalSeparator;
                      ThoSep := ThousandSeparator;
                      if Length(FParams) > 3 then begin
                        if Length(FParams[3]) > 0 then DecSep := FParams[3][1];
                        if Length(FParams[3]) > 1 then ThoSep := FParams[3][2];
                      end;
                      OldDec := DecimalSeparator;
                      OldTho := ThousandSeparator;
                      try
                        DecimalSeparator  := DecSep;
                        ThousandSeparator := ThoSep;
                        Result := Format(sFmt,[val]);
                      finally
                        DecimalSeparator  := OldDec;
                        ThousandSeparator := OldTho;
                      end;
                    end;
                  end;
    kwComment:    Result := Comment;
    else          Result := '';
  end;
  if (Result <> '') and not bNoLeadTrail then begin
    Insert(FLeadText, Result, 1);
    Result := Result + FTrailText;
  end;
  uri.DontURLEncode := b;
end;


destructor TKeyword.Destroy;
begin
  FSubKeyword.Free;
  inherited;
end;

{ TClipboardFormat }

constructor TClipboardFormat.Create;
begin
  inherited;
  FKeywords := TList.Create;
end;

destructor TClipboardFormat.Destroy;
begin
  FreeKeywords;
  FKeywords.Free;
  inherited;
end;

procedure TClipboardFormat.FreeKeywords;
begin
  while FKeywords.Count > 0 do begin
    TKeyword(FKeywords.Items[0]).Free;
    FKeywords.Delete(0);
  end;
end;

procedure TClipboardFormat.Init(ADescription,Expression: String);

  function NewKeyWord(var Exp: String): TKeyword;
  begin
    Result := TKeyword.Create;
    FKeywords.Add(Result);
    Result.Init(Exp);
  end;

var
  s:  String;
  i:  Integer;
  kw: TKeyword;
begin
  FDescription := ADescription;
  FNoMulti     := False;
  FSeparator   := #13#10;

  FreeKeywords;

  Expression := Trim(Expression);
  if Expression = '' then raise EKeywordError.Create('Expression is empty');
  repeat
    i := Length(Expression);
    case Expression[1] of
      '%': NewKeyWord(Expression);
      '"': begin
             s := RemoveQuotedString(Expression);
             if s <> '' then begin
               Insert('%text("'+s+'")%', Expression, 1);
               NewKeyWord(Expression);
             end;
           end;
      ' ': Expression := Trim(Expression);
      else raise EKeywordError.Create('Invalid expression (keyword or quoted text expected)');
    end;
    if Length(Expression) >= i then raise EKeywordError.Create('Internal error');
  until Expression = '';

  // check for nomulti and separator keywords
  i := 0;
  while i < FKeywords.Count do begin
    kw := FKeywords.Items[i];
    if kw.FType in [kwNoMulti, kwSeparator] then begin
      if kw.FType = kwNoMulti then
        FNoMulti := True
      else
        FSeparator := kw.FParams[0];
      kw.Free;
      FKeywords.Delete(i);
    end else
      inc(i);
  end;
end;

function TClipboardFormat.FormatKeys(URIList,FileSizeList: TList; CommentList: TStringList): String;
var
  i,j:  Integer;
  uri:  TFreenetURI;
  kw:   TKeyword;
  size: Cardinal;
  Comment: String;
begin
  Result := '';
  if URIList.Count = 0 then exit;
  if FNoMulti and (URIList.Count > 1) then
    raise EKeywordError.Create('This clipboard format does not support multiple keys');
  for i := 0 to URIList.Count-1 do begin
    uri := URIList.Items[i];
    if FileSizeList <> nil then size := Cardinal(FileSizeList.Items[i]) else size := 0;
    if CommentList <> nil then Comment := CommentList.Strings[i] else Comment := '';
    if i > 0 then Result := Result + FSeparator;
    for j := 0 to FKeywords.Count-1 do begin
      kw := FKeywords.Items[j];
      Result := Result + kw.Execute(uri,size,Comment);
    end;
  end;
end;


end.
