//PROFILE-NO
unit Twofish;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

// ported from Freenet's Twofish_Algorithm.java

{$INCLUDE CompilerOpts.pas}

interface

uses Windows,SysUtils,CryptUtils;

{-DEFINE TWOFISH_DEBUG}

type
  TTwofish = class(TCipher)
  private
    FpSessionKey: Pointer;
    FKeySize:     Integer; // in bits!
  protected
    function    GetBlockSize:  Integer; override;
    function    GetKeySize:    Integer; override;
    function    GetCipherName: String;  override;
  public
    constructor Create(AKeySize: Integer = 128);
    destructor  Destroy; override;
    procedure   Initialize(pKey: PByte; KeyLen: Integer); override;
    procedure   Encipher(pIn,pOut: PByte); override;
    procedure   Decipher(pIn,pOut: PByte); override;
  end;

function self_test(keysize: Integer): Boolean;

implementation

{$IFDEF TWOFISH_DEBUG}
uses Clipbrd,Dialogs,Classes;

var DebugLog: TStringList = nil;

procedure debug_clear; begin DebugLog.Clear; end;
procedure debug_log(const msg: String); begin DebugLog.Add(msg); end;
procedure debug_show;
begin
  Clipboard.AsText := DebugLog.Text;
  ShowMessage('Debug log in clipboard');
end;

function DataToStr(pData: PByte; DataLen: Integer): String;
const HEX_DIGITS = '0123456789ABCDEF';
var i: Integer;
begin
  Result := '';
  for i := 0 to DataLen-1 do begin
    Result := Result + HEX_DIGITS[1+((pData^ shr 4) and $0F)] + HEX_DIGITS[1+(pData^ and $0F)];
    inc(pData);
  end;
end;
{$ENDIF}

const
  BLOCK_SIZE    = 16;
  ROUNDS        = 16;
  MAX_ROUNDS    = 16;
  INPUT_WHITEN  = 0;
  OUTPUT_WHITEN = INPUT_WHITEN  + BLOCK_SIZE div 4;
  ROUND_SUBKEYS = OUTPUT_WHITEN + BLOCK_SIZE div 4;
  TOTAL_SUBKEYS = ROUND_SUBKEYS + 2*MAX_ROUNDS;
  SK_STEP       = $02020202;
  SK_BUMP       = $01010101;
  SK_ROTL       = 9;

  P: Array [0..1,0..255] of Byte = (
      (  // p0
         $A9, $67, $B3, $E8, $04, $FD, $A3, $76, $9A, $92, $80, $78, $E4, $DD, $D1, $38,
         $0D, $C6, $35, $98, $18, $F7, $EC, $6C, $43, $75, $37, $26, $FA, $13, $94, $48,
         $F2, $D0, $8B, $30, $84, $54, $DF, $23, $19, $5B, $3D, $59, $F3, $AE, $A2, $82,
         $63, $01, $83, $2E, $D9, $51, $9B, $7C, $A6, $EB, $A5, $BE, $16, $0C, $E3, $61,
         $C0, $8C, $3A, $F5, $73, $2C, $25, $0B, $BB, $4E, $89, $6B, $53, $6A, $B4, $F1,
         $E1, $E6, $BD, $45, $E2, $F4, $B6, $66, $CC, $95, $03, $56, $D4, $1C, $1E, $D7,
         $FB, $C3, $8E, $B5, $E9, $CF, $BF, $BA, $EA, $77, $39, $AF, $33, $C9, $62, $71,
         $81, $79, $09, $AD, $24, $CD, $F9, $D8, $E5, $C5, $B9, $4D, $44, $08, $86, $E7,
         $A1, $1D, $AA, $ED, $06, $70, $B2, $D2, $41, $7B, $A0, $11, $31, $C2, $27, $90,
         $20, $F6, $60, $FF, $96, $5C, $B1, $AB, $9E, $9C, $52, $1B, $5F, $93, $0A, $EF,
         $91, $85, $49, $EE, $2D, $4F, $8F, $3B, $47, $87, $6D, $46, $D6, $3E, $69, $64,
         $2A, $CE, $CB, $2F, $FC, $97, $05, $7A, $AC, $7F, $D5, $1A, $4B, $0E, $A7, $5A,
         $28, $14, $3F, $29, $88, $3C, $4C, $02, $B8, $DA, $B0, $17, $55, $1F, $8A, $7D,
         $57, $C7, $8D, $74, $B7, $C4, $9F, $72, $7E, $15, $22, $12, $58, $07, $99, $34,
         $6E, $50, $DE, $68, $65, $BC, $DB, $F8, $C8, $A8, $2B, $40, $DC, $FE, $32, $A4,
         $CA, $10, $21, $F0, $D3, $5D, $0F, $00, $6F, $9D, $36, $42, $4A, $5E, $C1, $E0
      ),
      (  // p1
         $75, $F3, $C6, $F4, $DB, $7B, $FB, $C8, $4A, $D3, $E6, $6B, $45, $7D, $E8, $4B,
         $D6, $32, $D8, $FD, $37, $71, $F1, $E1, $30, $0F, $F8, $1B, $87, $FA, $06, $3F,
         $5E, $BA, $AE, $5B, $8A, $00, $BC, $9D, $6D, $C1, $B1, $0E, $80, $5D, $D2, $D5,
         $A0, $84, $07, $14, $B5, $90, $2C, $A3, $B2, $73, $4C, $54, $92, $74, $36, $51,
         $38, $B0, $BD, $5A, $FC, $60, $62, $96, $6C, $42, $F7, $10, $7C, $28, $27, $8C,
         $13, $95, $9C, $C7, $24, $46, $3B, $70, $CA, $E3, $85, $CB, $11, $D0, $93, $B8,
         $A6, $83, $20, $FF, $9F, $77, $C3, $CC, $03, $6F, $08, $BF, $40, $E7, $2B, $E2,
         $79, $0C, $AA, $82, $41, $3A, $EA, $B9, $E4, $9A, $A4, $97, $7E, $DA, $7A, $17,
         $66, $94, $A1, $1D, $3D, $F0, $DE, $B3, $0B, $72, $A7, $1C, $EF, $D1, $53, $3E,
         $8F, $33, $26, $5F, $EC, $76, $2A, $49, $81, $88, $EE, $21, $C4, $1A, $EB, $D9,
         $C5, $39, $99, $CD, $AD, $31, $8B, $01, $18, $23, $DD, $1F, $4E, $2D, $F9, $48,
         $4F, $F2, $65, $8E, $78, $5C, $58, $19, $8D, $E5, $98, $57, $67, $7F, $05, $64,
         $AF, $63, $B6, $FE, $F5, $B7, $3C, $A5, $CE, $E9, $68, $44, $E0, $4D, $43, $69,
         $29, $2E, $AC, $15, $59, $A8, $0A, $9E, $6E, $47, $DF, $34, $35, $6A, $CF, $DC,
         $22, $C9, $C0, $9B, $89, $D4, $ED, $AB, $12, $A2, $0D, $52, $BB, $02, $2F, $A9,
         $D7, $61, $1E, $B4, $50, $04, $F6, $C2, $16, $25, $86, $56, $55, $09, $BE, $91
      )
   );

  P_00 = 1;
  P_01 = 0;
  P_02 = 0;
  P_03 = P_01 xor 1;
  P_04 = 1;

  P_10 = 0;
  P_11 = 0;
  P_12 = 1;
  P_13 = P_11 xor 1;
  P_14 = 0;

  P_20 = 1;
  P_21 = 1;
  P_22 = 0;
  P_23 = P_21 xor 1;
  P_24 = 0;

  P_30 = 0;
  P_31 = 1;
  P_32 = 1;
  P_33 = P_31 xor 1;
  P_34 = 1;

  GF256_FDBK   = $169;
  GF256_FDBK_2 = $169 div 2;
  GF256_FDBK_4 = $169 div 4;

  RS_GF_FDBK = $14D;

var
  MDS: Array [0..3,0..255] of Cardinal; // inited in initialization

function RS_rem(x: Cardinal): Cardinal;
var
  b: Byte;
  g2,g3: Cardinal;
begin
  b := (x shr 24) and $FF;
  g2 := b shl 1;
  if b and $80 <> 0 then g2 := g2 xor RS_GF_FDBK;
  g2 := g2 and $FF;
  g3 := b shr 1;
  if b and $01 <> 0 then g3 := g3 xor (RS_GF_FDBK shr 1);
  g3 := g3 xor g2;
  Result := (x shl 8) xor (g3 shl 24) xor (g2 shl 16) xor (g3 shl 8) xor b;
end;

function RS_MDS_Encode(k0,k1: Cardinal): Cardinal;
var r,i: Cardinal;
begin
  r := k1;
  for i := 0 to 3 do r := RS_rem(r);
  r := r xor k0;
  for i := 0 to 3 do r := RS_rem(r);
  Result := r;
end;

function b_0(x: Cardinal): Byte; begin Result := x and $FF; end;
function b_1(x: Cardinal): Byte; begin Result := (x shr 8) and $FF; end;
function b_2(x: Cardinal): Byte; begin Result := (x shr 16) and $FF; end;
function b_3(x: Cardinal): Byte; begin Result := (x shr 24) and $FF; end;

type
  T4CardinalArray = Array [0..3] of Cardinal;
  TSBox           = Array [0..1023] of Cardinal;
  TSessionKey = record
    SBox:    TSBox;
    SubKeys: Array [0..ROUND_SUBKEYS + 2*ROUNDS - 1] of Cardinal;
  end;
  PSessionKey = ^TSessionKey;


function F32(k64Cnt: Integer; x: Cardinal; k32: T4CardinalArray): Cardinal;
var
  b0,b1,b2,b3,AndVal: Byte;
  k0,k1,k2,k3:        Cardinal;
begin
  b0 := b_0(x); b1 := b_1(x); b2 := b_2(x); b3 := b_3(x);
  k0 := k32[0]; k1 := k32[1]; k2 := k32[2]; k3 := k32[3];
  AndVal := k64Cnt and 3;
  if AndVal = 1 then begin
    Result :=  MDS[0, P[P_01,b0] xor b_0(k0)]
           xor MDS[1, P[P_11,b1] xor b_1(k0)]
           xor MDS[2, P[P_21,b2] xor b_2(k0)]
           xor MDS[3, P[P_31,b3] xor b_3(k0)];
  end else begin
    if AndVal = 0 then begin
      b0 := P[P_04,b0] xor b_0(k3);
      b1 := P[P_14,b1] xor b_1(k3);
      b2 := P[P_24,b2] xor b_2(k3);
      b3 := P[P_34,b3] xor b_3(k3);
    end;
    if AndVal in [0,3] then begin
      b0 := P[P_03,b0] xor b_0(k2);
      b1 := P[P_13,b1] xor b_1(k2);
      b2 := P[P_23,b2] xor b_2(k2);
      b3 := P[P_33,b3] xor b_3(k2);
    end;
    Result :=  MDS[0, P[P_01, P[P_02,b0] xor b_0(k1)] xor b_0(k0)]
           xor MDS[1, P[P_11, P[P_12,b1] xor b_1(k1)] xor b_1(k0)]
           xor MDS[2, P[P_21, P[P_22,b2] xor b_2(k1)] xor b_2(k0)]
           xor MDS[3, P[P_31, P[P_32,b3] xor b_3(k1)] xor b_3(k0)];
  end;
end;

function Fe32_0(var SBox: TSBox; x: Cardinal): Cardinal;
begin
  Result :=  SBox[        (x shl  1) and $1FE     ]
         xor SBox[       ((x shr  7) and $1FE) + 1]
         xor SBox[$200 + ((x shr 15) and $1FE)    ]
         xor SBox[$200 + ((x shr 23) and $1FE) + 1];
end;

function Fe32_3(var SBox: TSBox; x: Cardinal): Cardinal;
begin
  Result :=  SBox[        (x shr 23) and $1FE     ]
         xor SBox[       ((x shl  1) and $1FE) + 1]
         xor SBox[$200 + ((x shr  7) and $1FE)    ]
         xor SBox[$200 + ((x shr 15) and $1FE) + 1];
end;

function MakeKey(pData: PByte; DataLen: Integer): TSessionKey;
const
  SubkeyCnt = ROUND_SUBKEYS + 2*ROUNDS;
var
  k64Cnt,i,j,Offset: Integer;
  k32e,k32o,SBoxKey: T4CardinalArray;
  q,A,B: Cardinal;
  b0,b1,b2,b3,AndVal: Byte;
  k0,k1,k2,k3: Cardinal;
  {$IFDEF TWOFISH_DEBUG}
  s: String;
  {$ENDIF}
begin
  {$IFDEF TWOFISH_DEBUG} debug_log('MakeKey( ' + DataToStr(pData,DataLen) + ' )'); {$ENDIF}
  if not (DataLen in [8,16,24,32]) then raise Exception.Create('Invalid key length');
  k64Cnt := DataLen div 8;
  i := 0; j := k64Cnt-1; Offset := 0;
  while (i < 4) and (Offset < DataLen) do begin
    k32e[i] :=  PByte(PChar(pData)+Offset  )^
            or (PByte(PChar(pData)+Offset+1)^ shl  8)
            or (PByte(PChar(pData)+Offset+2)^ shl 16)
            or (PByte(PChar(pData)+Offset+3)^ shl 24);
    inc(Offset,4);
    k32o[i] :=  PByte(PChar(pData)+Offset  )^
            or (PByte(PChar(pData)+Offset+1)^ shl  8)
            or (PByte(PChar(pData)+Offset+2)^ shl 16)
            or (PByte(PChar(pData)+Offset+3)^ shl 24);
    inc(Offset,4);
    SBoxKey[j] := RS_MDS_Encode(k32e[i], k32o[i]);
    inc(i); dec(j);
  end;
  i := 0; q := 0;
  while i < (SubkeyCnt div 2) do begin
    A := F32(k64Cnt, q,           k32e);
    B := F32(k64Cnt, q + SK_BUMP, k32o);
    B := (B shl 8) or (B shr 24);
    inc(A,B);
    Result.SubKeys[2*i  ] := A;
    inc(A,B);
    Result.SubKeys[2*i+1] := (A shl SK_ROTL) or (A shr (32-SK_ROTL));
    inc(i); inc(q, SK_STEP);
  end;
  k0 := SBoxKey[0]; k1 := SBoxKey[1]; k2 := SBoxKey[2]; k3 := SBoxKey[3];
  for i := 0 to 255 do begin
    b0 := i; b1 := i; b2 := i; b3 := i;
    AndVal := k64Cnt and 3;
    if AndVal = 1 then begin
      Result.SBox[     2*i  ] := MDS[0, P[P_01,b0] xor b_0(k0)];
      Result.SBox[     2*i+1] := MDS[1, P[P_11,b1] xor b_1(k0)];
      Result.SBox[$200+2*i  ] := MDS[2, P[P_21,b2] xor b_2(k0)];
      Result.SBox[$200+2*i+1] := MDS[3, P[P_31,b3] xor b_3(k0)];
    end else begin
      if AndVal = 0 then begin
        b0 := P[P_04,b0] xor b_0(k3);
        b1 := P[P_14,b1] xor b_1(k3);
        b2 := P[P_24,b2] xor b_2(k3);
        b3 := P[P_34,b3] xor b_3(k3);
      end;
      if AndVal in [0,3] then begin
        b0 := P[P_03,b0] xor b_0(k2);
        b1 := P[P_13,b1] xor b_1(k2);
        b2 := P[P_23,b2] xor b_2(k2);
        b3 := P[P_33,b3] xor b_3(k2);
      end;
      Result.SBox[     2*i  ] := MDS[0, P[P_01, P[P_02,b0] xor b_0(k1)] xor b_0(k0)];
      Result.SBox[     2*i+1] := MDS[1, P[P_11, P[P_12,b1] xor b_1(k1)] xor b_1(k0)];
      Result.SBox[$200+2*i  ] := MDS[2, P[P_21, P[P_22,b2] xor b_2(k1)] xor b_2(k0)];
      Result.SBox[$200+2*i+1] := MDS[3, P[P_31, P[P_32,b3] xor b_3(k1)] xor b_3(k0)];
    end;
  end;
  {$IFDEF TWOFISH_DEBUG}
  debug_log('S-box[]:');
  for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(Result.SBox[    i*4+j],8)+', '; debug_log(s); end; debug_log('');
  for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(Result.SBox[256+i*4+j],8)+', '; debug_log(s); end; debug_log('');
  for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(Result.SBox[512+i*4+j],8)+', '; debug_log(s); end; debug_log('');
  for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(Result.SBox[768+i*4+j],8)+', '; debug_log(s); end; debug_log('');
  debug_log('User (odd, even) keys  --> S-Box keys:');
  for i := 0 to k64Cnt-1 do debug_log('0x' + IntToHex(k32o[i],8) + '  0x' + IntToHex(k32e[i],8) + ' --> 0x' + IntToHex(SBoxKey[k64Cnt-1-i],8));
  debug_log('');
  debug_log('Round keys:');
  i := 0;
  while i < ROUND_SUBKEYS + 2*ROUNDS do begin
    debug_log('0x' + IntToHex(Result.SubKeys[i],8) + '  0x' + IntToHex(Result.SubKeys[i+1],8));
    inc(i,2);
  end;
  debug_log('');
  {$ENDIF}
end;

procedure BlockEncrypt(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey);
var
  x0,x1,x2,x3,t0,t1,k,R: Cardinal;
begin
  {$IFDEF TWOFISH_DEBUG}
    debug_log(Format('BlockEncrypt(%p,%p,%d,%p)',[pIn,pOut,InOffset,@SessionKey]));
    debug_log('PT=' + DataToStr(PByte(PChar(pIn)+InOffset), BLOCK_SIZE));
  {$ENDIF}
  x0 :=  PByte(PChar(pIn) + InOffset     )^
     or (PByte(PChar(pIn) + InOffset +  1)^ shl  8)
     or (PByte(PChar(pIn) + InOffset +  2)^ shl 16)
     or (PByte(PChar(pIn) + InOffset +  3)^ shl 24);
  x1 :=  PByte(PChar(pIn) + InOffset +  4)^
     or (PByte(PChar(pIn) + InOffset +  5)^ shl  8)
     or (PByte(PChar(pIn) + InOffset +  6)^ shl 16)
     or (PByte(PChar(pIn) + InOffset +  7)^ shl 24);
  x2 :=  PByte(PChar(pIn) + InOffset +  8)^
     or (PByte(PChar(pIn) + InOffset +  9)^ shl  8)
     or (PByte(PChar(pIn) + InOffset + 10)^ shl 16)
     or (PByte(PChar(pIn) + InOffset + 11)^ shl 24);
  x3 :=  PByte(PChar(pIn) + InOffset + 12)^
     or (PByte(PChar(pIn) + InOffset + 13)^ shl  8)
     or (PByte(PChar(pIn) + InOffset + 14)^ shl 16)
     or (PByte(PChar(pIn) + InOffset + 15)^ shl 24);

  x0 := x0 xor SessionKey.SubKeys[INPUT_WHITEN    ];
  x1 := x1 xor SessionKey.SubKeys[INPUT_WHITEN + 1];
  x2 := x2 xor SessionKey.SubKeys[INPUT_WHITEN + 2];
  x3 := x3 xor SessionKey.SubKeys[INPUT_WHITEN + 3];
  {$IFDEF TWOFISH_DEBUG} debug_log('PTw='+IntToHex(x0,8)+IntToHex(x1,8)+IntToHex(x2,8)+IntToHex(x3,8)); {$ENDIF}

  k := ROUND_SUBKEYS;
  R := 0;
  while R < ROUNDS do begin
    t0 := Fe32_0(SessionKey.SBox, x0);
    t1 := Fe32_3(SessionKey.SBox, x1);
    x2 := x2 xor (t0 +   t1 + SessionKey.SubKeys[k]); inc(k);
    x2 := (x2 shr 1) or (x2 shl 31);
    x3 := (x3 shl 1) or (x3 shr 31);
    x3 := x3 xor (t0 + 2*t1 + SessionKey.SubKeys[k]); inc(k);
    {$IFDEF TWOFISH_DEBUG} debug_log('CT'+IntToStr(R)+'='+IntToHex(x0,8)+IntToHex(x1,8)+IntToHex(x2,8)+IntToHex(x3,8)); {$ENDIF}

    t0 := Fe32_0(SessionKey.SBox, x2);
    t1 := Fe32_3(SessionKey.SBox, x3);
    x0 := x0 xor (t0 +   t1 + SessionKey.SubKeys[k]); inc(k);
    x0 := (x0 shr 1) or (x0 shl 31);
    x1 := (x1 shl 1) or (x1 shr 31);
    x1 := x1 xor (t0 + 2*t1 + SessionKey.SubKeys[k]); inc(k);
    {$IFDEF TWOFISH_DEBUG} debug_log('CT'+IntToStr(R+1)+'='+IntToHex(x0,8)+IntToHex(x1,8)+IntToHex(x2,8)+IntToHex(x3,8)); {$ENDIF}

    inc(R,2);
  end;

  x2 := x2 xor SessionKey.SubKeys[OUTPUT_WHITEN    ];
  x3 := x3 xor SessionKey.SubKeys[OUTPUT_WHITEN + 1];
  x0 := x0 xor SessionKey.SubKeys[OUTPUT_WHITEN + 2];
  x1 := x1 xor SessionKey.SubKeys[OUTPUT_WHITEN + 3];
  {$IFDEF TWOFISH_DEBUG} debug_log('CTw='+IntToHex(x0,8)+IntToHex(x1,8)+IntToHex(x2,8)+IntToHex(x3,8)); {$ENDIF}

  PByte(PChar(pOut)     )^ :=  x2         and $FF;
  PByte(PChar(pOut) +  1)^ := (x2 shr  8) and $FF;
  PByte(PChar(pOut) +  2)^ := (x2 shr 16) and $FF;
  PByte(PChar(pOut) +  3)^ := (x2 shr 24) and $FF;
  PByte(PChar(pOut) +  4)^ :=  x3         and $FF;
  PByte(PChar(pOut) +  5)^ := (x3 shr  8) and $FF;
  PByte(PChar(pOut) +  6)^ := (x3 shr 16) and $FF;
  PByte(PChar(pOut) +  7)^ := (x3 shr 24) and $FF;
  PByte(PChar(pOut) +  8)^ :=  x0         and $FF;
  PByte(PChar(pOut) +  9)^ := (x0 shr  8) and $FF;
  PByte(PChar(pOut) + 10)^ := (x0 shr 16) and $FF;
  PByte(PChar(pOut) + 11)^ := (x0 shr 24) and $FF;
  PByte(PChar(pOut) + 12)^ :=  x1         and $FF;
  PByte(PChar(pOut) + 13)^ := (x1 shr  8) and $FF;
  PByte(PChar(pOut) + 14)^ := (x1 shr 16) and $FF;
  PByte(PChar(pOut) + 15)^ := (x1 shr 24) and $FF;
  {$IFDEF TWOFISH_DEBUG} debug_log('CT='+DataToStr(pOut,BLOCK_SIZE)); debug_log(''); {$ENDIF}
end;

procedure BlockDecrypt(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey);
var
  x0,x1,x2,x3,t0,t1,k,R: Cardinal;
begin
  {$IFDEF TWOFISH_DEBUG}
    debug_log(Format('BlockDecrypt(%p,%p,%d,%p)',[pIn,pOut,InOffset,@SessionKey]));
    debug_log('CT=' + DataToStr(PByte(PChar(pIn)+InOffset), BLOCK_SIZE));
  {$ENDIF}
  x2 :=  PByte(PChar(pIn) + InOffset     )^
     or (PByte(PChar(pIn) + InOffset +  1)^ shl  8)
     or (PByte(PChar(pIn) + InOffset +  2)^ shl 16)
     or (PByte(PChar(pIn) + InOffset +  3)^ shl 24);
  x3 :=  PByte(PChar(pIn) + InOffset +  4)^
     or (PByte(PChar(pIn) + InOffset +  5)^ shl  8)
     or (PByte(PChar(pIn) + InOffset +  6)^ shl 16)
     or (PByte(PChar(pIn) + InOffset +  7)^ shl 24);
  x0 :=  PByte(PChar(pIn) + InOffset +  8)^
     or (PByte(PChar(pIn) + InOffset +  9)^ shl  8)
     or (PByte(PChar(pIn) + InOffset + 10)^ shl 16)
     or (PByte(PChar(pIn) + InOffset + 11)^ shl 24);
  x1 :=  PByte(PChar(pIn) + InOffset + 12)^
     or (PByte(PChar(pIn) + InOffset + 13)^ shl  8)
     or (PByte(PChar(pIn) + InOffset + 14)^ shl 16)
     or (PByte(PChar(pIn) + InOffset + 15)^ shl 24);

  x2 := x2 xor SessionKey.SubKeys[OUTPUT_WHITEN    ];
  x3 := x3 xor SessionKey.SubKeys[OUTPUT_WHITEN + 1];
  x0 := x0 xor SessionKey.SubKeys[OUTPUT_WHITEN + 2];
  x1 := x1 xor SessionKey.SubKeys[OUTPUT_WHITEN + 3];
  {$IFDEF TWOFISH_DEBUG} debug_log('CTw='+IntToHex(x2,8)+IntToHex(x3,8)+IntToHex(x0,8)+IntToHex(x1,8)); {$ENDIF}

  k := ROUND_SUBKEYS + 2*ROUNDS - 1;
  R := 0;
  while R < ROUNDS do begin
    t0 := Fe32_0(SessionKey.SBox, x2);
    t1 := Fe32_3(SessionKey.SBox, x3);
    x1 := x1 xor (t0 + 2*t1 + SessionKey.SubKeys[k]); dec(k);
    x1 := (x1 shr 1) or (x1 shl 31);
    x0 := (x0 shl 1) or (x0 shr 31);
    x0 := x0 xor (t0 +   t1 + SessionKey.SubKeys[k]); dec(k);
    {$IFDEF TWOFISH_DEBUG} debug_log('PT'+IntToStr(ROUNDS-R)+'='+IntToHex(x2,8)+IntToHex(x3,8)+IntToHex(x0,8)+IntToHex(x1,8)); {$ENDIF}

    t0 := Fe32_0(SessionKey.SBox, x0);
    t1 := Fe32_3(SessionKey.SBox, x1);
    x3 := x3 xor (t0 + 2*t1 + SessionKey.SubKeys[k]); dec(k);
    x3 := (x3 shr 1) or (x3 shl 31);
    x2 := (x2 shl 1) or (x2 shr 31);
    x2 := x2 xor (t0 +   t1 + SessionKey.SubKeys[k]); dec(k);
    {$IFDEF TWOFISH_DEBUG} debug_log('PT'+IntToStr(ROUNDS-R-1)+'='+IntToHex(x2,8)+IntToHex(x3,8)+IntToHex(x0,8)+IntToHex(x1,8)); {$ENDIF}

    inc(R,2);
  end;

  x0 := x0 xor SessionKey.SubKeys[INPUT_WHITEN    ];
  x1 := x1 xor SessionKey.SubKeys[INPUT_WHITEN + 1];
  x2 := x2 xor SessionKey.SubKeys[INPUT_WHITEN + 2];
  x3 := x3 xor SessionKey.SubKeys[INPUT_WHITEN + 3];
  {$IFDEF TWOFISH_DEBUG} debug_log('PTw='+IntToHex(x2,8)+IntToHex(x3,8)+IntToHex(x0,8)+IntToHex(x1,8)); {$ENDIF}

  PByte(PChar(pOut)     )^ :=  x0         and $FF;
  PByte(PChar(pOut) +  1)^ := (x0 shr  8) and $FF;
  PByte(PChar(pOut) +  2)^ := (x0 shr 16) and $FF;
  PByte(PChar(pOut) +  3)^ := (x0 shr 24) and $FF;
  PByte(PChar(pOut) +  4)^ :=  x1         and $FF;
  PByte(PChar(pOut) +  5)^ := (x1 shr  8) and $FF;
  PByte(PChar(pOut) +  6)^ := (x1 shr 16) and $FF;
  PByte(PChar(pOut) +  7)^ := (x1 shr 24) and $FF;
  PByte(PChar(pOut) +  8)^ :=  x2         and $FF;
  PByte(PChar(pOut) +  9)^ := (x2 shr  8) and $FF;
  PByte(PChar(pOut) + 10)^ := (x2 shr 16) and $FF;
  PByte(PChar(pOut) + 11)^ := (x2 shr 24) and $FF;
  PByte(PChar(pOut) + 12)^ :=  x3         and $FF;
  PByte(PChar(pOut) + 13)^ := (x3 shr  8) and $FF;
  PByte(PChar(pOut) + 14)^ := (x3 shr 16) and $FF;
  PByte(PChar(pOut) + 15)^ := (x3 shr 24) and $FF;
  {$IFDEF TWOFISH_DEBUG} debug_log('PT='+DataToStr(pOut,BLOCK_SIZE)); debug_log(''); {$ENDIF}
end;

function self_test(keysize: Integer): Boolean;
var
  kb,pt,ct,cpt: PByte;
  i:     Integer;
  key:   TSessionKey;
  {$IFDEF TWOFISH_DEBUG}
  j: Integer;
  s: String;
  {$ENDIF}
begin
  {$IFDEF TWOFISH_DEBUG}
  debug_clear;
  debug_log('Static data');
  debug_log(''); debug_log('MDS[0][]:'); for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(MDS[0,i*4+j],8)+', '; debug_log(s); end;
  debug_log(''); debug_log('MDS[1][]:'); for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(MDS[1,i*4+j],8)+', '; debug_log(s); end;
  debug_log(''); debug_log('MDS[2][]:'); for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(MDS[2,i*4+j],8)+', '; debug_log(s); end;
  debug_log(''); debug_log('MDS[3][]:'); for i := 0 to 63 do begin s := ''; for j := 0 to 3 do s := s + '0x' + IntToHex(MDS[3,i*4+j],8)+', '; debug_log(s); end;
  debug_log(PChar('self_test(' + IntToStr(keysize) + ')'));
  {$ENDIF}

  kb := nil; pt := nil; ct := nil; cpt := nil;
  try
    GetMem(kb, keysize);
    GetMem(pt, BLOCK_SIZE);
    GetMem(ct, BLOCK_SIZE);
    GetMem(cpt, BLOCK_SIZE);
    for i := 0 to keysize-1 do
      PByte(PChar(kb)+i)^ := i and $FF;
    for i := 0 to BLOCK_SIZE-1 do
      PByte(PChar(pt)+i)^ := i and $FF;
    {$IFDEF TWOFISH_DEBUG}
    debug_log('KEYSIZE=' + IntToStr(8*keysize));
    debug_log('KEY=' + DataToStr(kb,keysize));
    {$ENDIF}
    key := MakeKey(kb, keysize);
    {$IFDEF TWOFISH_DEBUG} debug_log('PT =' + DataToStr(pt,BLOCK_SIZE)); {$ENDIF}
    BlockEncrypt(pt, ct, 0, key);
    {$IFDEF TWOFISH_DEBUG} debug_log('CT =' + DataToStr(ct,BLOCK_SIZE)); {$ENDIF}
    BlockDecrypt(ct, cpt, 0, key);
    {$IFDEF TWOFISH_DEBUG} debug_log('CPT=' + DataToStr(cpt,BLOCK_SIZE)); {$ENDIF}
    Result := CompareMem(pt, cpt, BLOCK_SIZE);
    {$IFDEF TWOFISH_DEBUG}
    if Result then
      debug_log('Self-test OK')
    else
      debug_log('Self-test FAILED');
    {$ENDIF}
  finally
    if kb <> nil then FreeMem(kb);
    if pt <> nil then FreeMem(pt);
    if ct <> nil then FreeMem(ct);
    if cpt <> nil then FreeMem(cpt);
    {$IFDEF TWOFISH_DEBUG} debug_show; {$ENDIF}
  end;
end;

procedure InitMDS;
  function LFSR1(x: Byte): Byte;
  begin
    Result := (x shr 1);
    if (x and 1) <> 0 then Result := Result xor GF256_FDBK_2;
  end;
  function LFSR2(x: Byte): Byte;
  begin
    Result := (x shr 2);
    if (x and 2) <> 0 then Result := Result xor GF256_FDBK_2;
    if (x and 1) <> 0 then Result := Result xor GF256_FDBK_4;
  end;
  function Mx_1(x: Byte): Byte; begin Result := x; end;
  function Mx_X(x: Byte): Byte; begin Result := x xor LFSR2(x); end;
  function Mx_Y(x: Byte): Byte; begin Result := x xor LFSR1(x) xor LFSR2(x); end;
var
  m1,mX,mY: Array [0..1] of Byte;
  i,j:      Byte;
begin
  for i := 0 to 255 do begin
    j := P[0,i];
    m1[0] := j;
    mX[0] := Mx_X(j);
    mY[0] := Mx_Y(j);

    j := P[1,i];
    m1[1] := j;
    mX[1] := Mx_X(j);
    mY[1] := Mx_Y(j);

    MDS[0,i] :=  m1[P_00]
             or (mX[P_00] shl  8)
             or (mY[P_00] shl 16)
             or (mY[P_00] shl 24);
    MDS[1,i] :=  mY[P_10]
             or (mY[P_10] shl  8)
             or (mX[P_10] shl 16)
             or (m1[P_10] shl 24);
    MDS[2,i] :=  mX[P_20]
             or (mY[P_20] shl  8)
             or (m1[P_20] shl 16)
             or (mY[P_20] shl 24);
    MDS[3,i] :=  mX[P_30]
             or (m1[P_30] shl  8)
             or (mY[P_30] shl 16)
             or (mX[P_30] shl 24);
  end;
end;

{ TTwofish }

constructor TTwofish.Create(AKeySize: Integer);
begin
  inherited Create;
  FKeySize := AKeySize;
  if (AKeySize <> 64) and (AKeySize <> 128) and (AKeySize <> 192) and (AKeySize <> 256) then
    raise Exception.Create('Invalid key size');
  New(PSessionKey(FpSessionKey));
  FillChar(FpSessionKey^, SizeOf(TSessionKey), 0);
end;

destructor TTwofish.Destroy;
begin
  if FpSessionKey <> nil then Dispose(PSessionKey(FpSessionKey));
  inherited;
end;

procedure TTwofish.Initialize(pKey: PByte; KeyLen: Integer);
var KeyData: packed Array [0..255] of Byte; // max.key len
begin
  FillChar(KeyData, SizeOf(KeyData), 0);
  if KeyLen >= (FKeySize shr 3) then
    Move(pKey^, KeyData, KeyLen)
  else
    Move(pKey^, KeyData, (FKeySize shr 3));

  PSessionKey(FpSessionKey)^ := Twofish.MakeKey(@KeyData, (FKeySize shr 3));
end;

procedure TTwofish.Encipher(pIn, pOut: PByte);
begin
  Twofish.BlockEncrypt(pIn, pOut, 0, PSessionKey(FpSessionKey)^);
end;

procedure TTwofish.Decipher(pIn, pOut: PByte);
begin
  Twofish.BlockDecrypt(pIn, pOut, 0, PSessionKey(FpSessionKey)^);
end;


function TTwofish.GetBlockSize: Integer;
begin
  Result := Twofish.BLOCK_SIZE shl 3;
end;

function TTwofish.GetKeySize: Integer;
begin
  Result := FKeySize;
end;

function TTwofish.GetCipherName: String;
begin
  Result := CIPHERNAME_TWOFISH;
end;

initialization
{$IFDEF TWOFISH_DEBUG}
  DebugLog := TStringList.Create;
{$ENDIF}
  InitMDS;

finalization;
{$IFDEF TWOFISH_DEBUG}
  DebugLog.Free;
{$ENDIF}

end.

