//PROFILE-NO
unit ScanFrost;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, CheckLst, Mask, Menus, ComCtrls, ImgList;

type
  TFrmScanFrost = class(TForm)
    Panel1: TPanel;
    LbFrostFolder: TLabel;
    BtFrostFolder: TButton;
    Label1: TLabel;
    PanelBoards: TPanel;
    Label2: TLabel;
    ClbBoards: TCheckListBox;
    OpenDlgFrostFolder: TOpenDialog;
    Panel2: TPanel;
    Label3: TLabel;
    EdMinDate: TEdit;
    Label5: TLabel;
    EdMaxDate: TEdit;
    BtOk: TButton;
    PopupMenu1: TPopupMenu;
    MiCheckAll: TMenuItem;
    MiUncheckAll: TMenuItem;
    TvBoards: TTreeView;
    ImageList1: TImageList;
    RbFlatView: TRadioButton;
    RbTreeView: TRadioButton;
    Label4: TLabel;
    RbSearchKeys: TRadioButton;
    RbSearchMessages: TRadioButton;
    Label6: TLabel;
    EdSearchText: TEdit;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    CbSearchSubject: TCheckBox;
    CbSearchPoster: TCheckBox;
    CbSearchBody: TCheckBox;
    CbSearchFiles: TCheckBox;
    BtClose: TButton;
    Label10: TLabel;
    procedure BtFrostFolderClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure BtOkClick(Sender: TObject);
    procedure MiCheckAllClick(Sender: TObject);
    procedure RbViewClick(Sender: TObject);
    procedure TvBoardsClick(Sender: TObject);
    procedure BtCloseClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    FFileList:   String;
    FComments:   String;
    FDoInitOnShow: Boolean;
    procedure UpdateBoardList;
    procedure UpdateTreeView;
    procedure SetAnchors;
    procedure GetSelectedBoards(List: TStringList; UpperCaseNames: Boolean = False);
  public
    procedure DiscardFilesAndComments;
    property Files:    String read FFileList;
    property Comments: String read FComments;
  end;

  TFrostScanSettings = record
    FrostDir: String;
    TreeView: Boolean;
    MinDat,MaxDat: TDateTime;
    SearchKeys:    Boolean;
    SearchText:    String;
    SearchSubject: Boolean;
    SearchPoster:  Boolean;
    SearchBody:    Boolean;
    SearchFiles:   Boolean;
    SelBoards:     String;
  end;

var
  FrmScanFrost: TFrmScanFrost = nil;
  FrostScanSettings: TFrostScanSettings = (
    FrostDir:      '';
    TreeView:      True;
    MinDat:        0;
    MaxDat:        0;
    SearchKeys:    True;
    SearchText:    '';
    SearchSubject: True;
    SearchPoster:  False;
    SearchBody:    True;
    SearchFiles:   True;
    SelBoards:     '';
  );

implementation

// avoid FileCtrl is platform specific warning
{$IFDEF DELPHI_6_OR_HIGHER} {$WARN UNIT_PLATFORM OFF} {$ENDIF}
uses FileCtrl, Misc;

{$R *.dfm}

function IsFolder(tn: TTreeNode): Boolean;
begin
  Result := Boolean(tn.Data);
end;

procedure SetNodeImgIndex(tn: TTreeNode; img: Integer; ForceUpdate: Boolean = True; AffectChilds: Boolean = True);
var s: String;
begin
  tn.ImageIndex := img; tn.SelectedIndex := img;
  if ForceUpdate then begin
    s := tn.Text; tn.Text := s+' '; tn.Text := s; // force a repaint
  end;
  if AffectChilds and IsFolder(tn) then begin
    tn := tn.GetFirstChild;
    while tn <> nil do begin
      SetNodeImgIndex(tn, img);
      tn := tn.GetNextSibling;
    end;
  end;
end;


procedure TFrmScanFrost.FormCreate(Sender: TObject);
begin
  SetAnchors;

  // init from saved settings
  LbFrostFolder.Caption := FrostScanSettings.FrostDir;
  if FrostScanSettings.TreeView then RbTreeView.Checked := True else RbFlatView.Checked := True;
  if FrostScanSettings.MinDat = 0 then EdMinDate.Text := '' else EdMinDate.Text := FormatDateTime('yyyy-mm-dd', FrostScanSettings.MinDat);
  if FrostScanSettings.MaxDat = 0 then EdMaxDate.Text := '' else EdMaxDate.Text := FormatDateTime('yyyy-mm-dd', FrostScanSettings.MaxDat);
  if FrostScanSettings.SearchKeys then RbSearchKeys.Checked := True else RbSearchMessages.Checked := True;
  EdSearchText.Text := FrostScanSettings.SearchText;
  CbSearchSubject.Checked := FrostScanSettings.SearchSubject;
  CbSearchPoster.Checked  := FrostScanSettings.SearchPoster;
  CbSearchBody.Checked    := FrostScanSettings.SearchBody;
  CbSearchFiles.Checked   := FrostScanSettings.SearchFiles;

  RbViewClick(nil);
  FDoInitOnShow := (FrostScanSettings.FrostDir <> '') and DirectoryExists(FrostScanSettings.FrostDir);
end;

procedure TFrmScanFrost.FormShow(Sender: TObject);
begin
  if FDoInitOnShow then begin
    FDoInitOnShow := False;
    BtFrostFolderClick(nil);
  end;
end;

procedure TFrmScanFrost.BtFrostFolderClick(Sender: TObject);
var s: String;
begin
  if Sender <> nil then begin
    if not OpenDlgFrostFolder.Execute then exit;
    s := ExtractFilePath(OpenDlgFrostFolder.Filename);
    if not DirectoryExists(s + 'keypool') then begin
      MessageDlg('Folder ' + s + 'keypool does not exists.', mtError, [mbOk], 0);
      exit;
    end;
    FrostScanSettings.FrostDir := s;
    LbFrostFolder.Caption := FrostScanSettings.FrostDir;
  end;
  UpdateBoardList;
  UpdateTreeView;
end;

procedure TFrmScanFrost.UpdateBoardList;
var
  SearchRec: TSearchRec;
  sl:        TStringList;
  i:         Integer;
begin
  ClbBoards.Items.Clear;
  if 0 = FindFirst(FrostScanSettings.FrostDir + 'keypool\*.*', faDirectory, SearchRec) then begin
    sl := nil;
    ClbBoards.Items.BeginUpdate;
    try
      sl := TStringList.Create;
      sl.Text := FrostScanSettings.SelBoards;
      repeat
        if (SearchRec.Name = '.') or (SearchRec.Name = '..') then continue;
        i := ClbBoards.Items.Add(SearchRec.Name);
        if sl.IndexOf(AnsiUpperCase(SearchRec.Name)) >= 0 then ClbBoards.Checked[i] := True;
      until 0 <> FindNext(SearchRec);
    finally
      ClbBoards.Items.EndUpdate;
      FindClose(SearchRec);
      sl.Free;
    end;
  end;
end;


procedure TFrmScanFrost.BtOkClick(Sender: TObject);

  function  MyStrToDate(Str: String): TDateTime;
  begin
    Str := Trim(Str);
    if Str = '' then
      Result := 0
    else begin
      try
        if Length(Str) <> 10 then raise EAbort.Create('');
        if (Str[5] <> '-') or (Str[8] <> '-') then raise EAbort.Create('');
        Result := EncodeDate(StrToInt(Copy(Str,1,4)),StrToInt(Copy(Str,6,2)),StrToInt(Copy(Str,9,2)));
      except
        raise Exception.Create('Invalid date "' + Str + '" (use format YYYY-MM-DD)!');
      end;
    end;
  end;

  function ExtractIndexFromFrostXMLName(AFileName: String): Integer;
  var i,n: Integer;
  begin
    Result := -1;
    if CompareText(Copy(AFileName, Length(AFileName)-3, 4), '.xml') <> 0 then exit;
    n := 0;
    for i := Length(AFileName)-4 downto 1 do begin
      case AFileName[i] of
        '-':      begin Result := StrToIntDef(Copy(AFileName,i+1,n), -1); exit; end;
        '0'..'9': ;
        else      exit;
      end;
      inc(n);
    end;
  end;

var
  dtMin,dtMax,dt: TDateTime;
  SearchRec,SRec2: TSearchRec;
  slFiles:     TStringList;
  slComments:  TStringList;
  iBoard,i,idx: Integer;
  s,sBoardDir: String;
  y,m,d:       Word;
  slBoards:    TStringList;
begin
  dtMin := MyStrToDate(EdMinDate.Text);
  dtMax := MyStrToDate(EdMaxDate.Text);

  slFiles := nil; slComments := nil; slBoards := nil;
  dt := 0; // stupid compiler warning
  try
    Screen.Cursor := crHourglass;
    slFiles    := TStringList.Create;
    slComments := TStringList.Create;
    slBoards   := TStringList.Create;

    GetSelectedBoards(slBoards);
    for iBoard := 0 to slBoards.Count-1 do begin
      sBoardDir := FrostScanSettings.FrostDir + 'keypool\' + slBoards.Strings[iBoard] + '\';
      if 0 = FindFirst(sBoardDir + '*.*', faDirectory, SearchRec) then begin
        try
          repeat
            if (SearchRec.Name = '.') or (SearchRec.Name = '..') then continue;

            // parse dir name (frost uses year.month.day)
            s := SearchRec.Name;
            i := Pos('.', s); if i = 0 then continue;
            y := StrToIntDef(Copy(s,1,i-1), 0); Delete(s,1,i);
            i := Pos('.', s); if i = 0 then continue;
            m := StrToIntDef(Copy(s,1,i-1), 0); Delete(s,1,i);
            d := StrToIntDef(s, 0);
            if (y = 0) or (m < 1) or (m > 12) or (d < 1) or (d > 31) then continue;
            try
              dt := EncodeDate(y,m,d);
              if (dtMin <> 0) and (dt < dtMin) then continue;
              if (dtMax <> 0) and (dt > dtMax) then continue;
            except
              continue;
            end;

            // ok, get the xml files for that day
            if 0 = FindFirst(sBoardDir + SearchRec.Name + '\*.xml', 0, SRec2) then begin
              try
                repeat
                  slFiles.Add(sBoardDir + SearchRec.Name + '\' + SRec2.Name);

                  // add comment: BoardName YYYY-MM-DD (idx)
                  idx := ExtractIndexFromFrostXMLName(SRec2.Name);
                  if idx >= 0 then s := ' (' + IntToStr(idx) + ')' else s := '';
                  slComments.Add(
                    slBoards.Strings[iBoard] + ' ' +
                    FormatDateTime('yyyy-mm-dd', dt) + ' ' +
                    s
                  );
                until 0 <> FindNext(SRec2);
              finally
                FindClose(SRec2);
              end;
            end;

          until 0 <> FindNext(SearchRec);
        finally
          FindClose(SearchRec);
        end;
      end;
    end;

    FFileList := slFiles.Text;
    FComments := slComments.Text;

    for iBoard := 0 to slBoards.Count-1 do begin
      slBoards.Strings[iBoard] := ExtractFileName(slBoards.Strings[iBoard]);
    end;
    FrostScanSettings.SelBoards := AnsiUpperCase(slBoards.Text);

  finally
    Screen.Cursor := crDefault;
    slFiles.Free;
    slComments.Free;
    slBoards.Free;
  end;

  FrostScanSettings.MinDat         := dtMin;
  FrostScanSettings.MaxDat         := dtMax;
  FrostScanSettings.SearchKeys     := RbSearchKeys.Checked;
  FrostScanSettings.SearchText     := EdSearchText.Text;
  FrostScanSettings.SearchSubject  := CbSearchSubject.Checked;
  FrostScanSettings.SearchPoster   := CbSearchPoster.Checked;
  FrostScanSettings.SearchBody     := CbSearchBody.Checked;
  FrostScanSettings.SearchFiles    := CbSearchFiles.Checked;

  ModalResult := mrOk;
end;

procedure TFrmScanFrost.MiCheckAllClick(Sender: TObject);
var
  bCheck: Boolean;
  i:      Integer;
begin
  bCheck := (Sender = MiCheckAll);
  ClbBoards.Items.BeginUpdate;
  try
    for i := 0 to ClbBoards.Items.Count-1 do
      ClbBoards.Checked[i] := bCheck;
  finally
    ClbBoards.Items.EndUpdate;
  end;
end;

procedure TFrmScanFrost.GetSelectedBoards(List: TStringList; UpperCaseNames: Boolean);
var
  i:  Integer;
  tn: TTreeNode;
begin
  List.Clear;
  if FrostScanSettings.TreeView then begin
    tn := TvBoards.Items.GetFirstNode;
    while tn <> nil do begin
      if (tn.ImageIndex = 1) and (not IsFolder(tn)) then begin
        if UpperCaseNames then List.Add(AnsiUpperCase(tn.Text)) else List.Add(tn.Text);
      end;
      tn := tn.GetNext;
    end;
  end else begin
    for i := 0 to ClbBoards.Items.Count-1 do
      if ClbBoards.Checked[i] then begin
        if UpperCaseNames then List.Add(AnsiUpperCase(ClbBoards.Items[i])) else List.Add(ClbBoards.Items[i]);
      end;
  end;
end;

procedure TFrmScanFrost.UpdateTreeview;
var slSel: TStringList;

  function AddTreeNode(ParentTreeNode: TTreeNode; Text: String; IsFolder: Boolean): TTreeNode;
  var s: String;
  begin
    if IsFolder then s := '[' + Text + ']' else s := Text;
    Result := TvBoards.Items.AddChild(ParentTreeNode, s);
    if (not IsFolder) and (slSel.IndexOf(AnsiUpperCase(Text)) >= 0) then
      Result.ImageIndex := 1
    else
      Result.ImageIndex := 0;
    Result.SelectedIndex := Result.ImageIndex;
    Result.Data := Pointer(IsFolder);
  end;

  procedure AddBoards(ParentTreeNode: TTreeNode; Board: TFrostBoard);
  var tn: TTreeNode;
  begin
    while Board <> nil do begin
      tn := AddTreeNode(ParentTreeNode, Board.BoardName, Board.IsFolder);
      if Board.Child <> nil then AddBoards(tn, Board.Child);
      Board := Board.Next;
    end;
  end;
var
  Root: TFrostBoard;
  sl:   TStringList;
  i:    Integer;
  tn:   TTreeNode;
begin
  if FrostScanSettings.FrostDir = '' then exit;

  sl := nil; slSel := nil; Root := nil;
  try
    TvBoards.Items.Clear;

    slSel := TStringList.Create;
    slSel.Text := FrostScanSettings.SelBoards;

    // add active boards
    Root := GetFrostBoardListFromXML(GetFrostXMLAsString(FrostScanSettings.FrostDir+'config\boards.xml'));
    if Root <> nil then AddBoards(nil,Root.Child);

    // add boards that have been removed from Frost
    sl := TStringList.Create;
    tn := TvBoards.Items.GetFirstNode;
    while tn <> nil do begin
      sl.Add(AnsiUpperCase(tn.Text));
      tn := tn.GetNext;
    end;
    tn := nil;
    for i := 0 to ClbBoards.Items.Count-1 do begin
      if sl.IndexOf( AnsiUpperCase(ClbBoards.Items[i]) ) < 0 then begin
        if tn = nil then tn := AddTreeNode(nil, 'Removed Frost boards', True);
        AddTreeNode(tn, ClbBoards.Items[i], False);
      end;
    end;

    TvBoards.FullExpand;
    tn := TvBoards.Items.GetFirstNode;
    if tn <> nil then tn.MakeVisible;
    
  finally
    Root.Free;
    sl.Free;
    slSel.Free;
  end;
end;

procedure TFrmScanFrost.RbViewClick(Sender: TObject);
var
  bCopySelection: Boolean;
  sl:             TStringList;
  tn:             TTreeNode;
  bSel:           Boolean;
  i:              Integer;
begin
  sl := nil;
  try
    bCopySelection := (Sender <> nil) and (TvBoards.Visible <> RbTreeView.Checked);
    if bCopySelection then begin
      sl := TStringList.Create;
      GetSelectedBoards(sl, True); // get boards from old view, in upper case
    end;
    
    FrostScanSettings.TreeView := RbTreeView.Checked;

    if bCopySelection then begin
      if FrostScanSettings.TreeView then begin
        tn := TvBoards.Items.GetFirstNode;
        while tn <> nil do begin
          if IsFolder(tn) then
            bSel := False
          else
            bSel := (sl.IndexOf(AnsiUpperCase(tn.Text)) >= 0);
          if bSel then SetNodeImgIndex(tn, 1, False, False)
                  else SetNodeImgIndex(tn, 0, False, False);
          tn := tn.GetNext;
        end;
        TvBoards.Repaint;
      end else begin
        for i := 0 to ClbBoards.Items.Count-1 do
          ClbBoards.Checked[i] := (sl.IndexOf(AnsiUpperCase(ClbBoards.Items[i])) >= 0);
      end;
    end;

    ClbBoards.Visible := not FrostScanSettings.TreeView;
    TvBoards.Visible  := FrostScanSettings.TreeView;
  finally
    sl.Free;
  end;
end;

procedure TFrmScanFrost.TvBoardsClick(Sender: TObject);
var
  tn: TTreeNode;
  p:  TPoint;
begin
  p := TvBoards.ScreenToClient(Mouse.CursorPos);
  if htOnIcon in TvBoards.GetHitTestInfoAt(p.X,p.Y) then begin
    tn := TvBoards.GetNodeAt(p.X,p.Y);
    if tn <> nil then SetNodeImgIndex(tn, 1 - tn.ImageIndex);
  end;
end;

procedure TFrmScanFrost.BtCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TFrmScanFrost.DiscardFilesAndComments;
begin
  FFileList := '';
  FComments := '';
end;

procedure TFrmScanFrost.SetAnchors;
begin
  // Set anchors for controls that stick to the right or bottom of their parent.
  // Normally this would be done at design time, but D7 then saves .DesignSize,
  // breaking compatibility with earlier Delphi versions. So do it at runtime.
  with LbFrostFolder          do Anchors := [akLeft,akTop,akRight];
  with BtFrostFolder          do Anchors := [akTop,akRight];
  with ClbBoards              do Anchors := [akLeft,akTop,akRight,akBottom];
  with TvBoards               do Anchors := [akLeft,akTop,akRight,akBottom];
end;

end.
