//PROFILE-NO
unit Rijndael;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

// ported from Freenet's Rijndael_Algorithm.java

{$INCLUDE CompilerOpts.pas}

interface

uses Windows,SysUtils,CryptUtils;

{-DEFINE RIJNDAEL_DEBUG}

type
  TRijndael = class(TCipher)
  private
    FpSessionKey: Pointer;
    FKeySize:     Integer; // in bits!
    FBlockSize:   Integer; // in bits!
  protected
    function    GetBlockSize:  Integer; override;
    function    GetKeySize:    Integer; override;
    function    GetCipherName: String;  override;
  public
    constructor Create(AKeySize: Integer = 128; ABlockSize: Integer = 128);
    destructor  Destroy; override;
    procedure   Initialize(pKey: PByte; KeyLen: Integer); override;
    procedure   Encipher(pIn,pOut: PByte); override;
    procedure   Decipher(pIn,pOut: PByte); override;
  end;

function self_test(keysize: Integer): Boolean; overload;
function self_test(keysize,blocksize: Integer): Boolean; overload;

implementation

{$IFDEF RIJNDAEL_DEBUG}
uses Clipbrd,Dialogs,Classes;

var DebugLog: TStringList = nil;

procedure debug_clear; begin DebugLog.Clear; end;
procedure debug_log(const msg: String); begin DebugLog.Add(msg); end;
procedure debug_show(Descript: String = 'Debug log');
begin
  Clipboard.AsText := DebugLog.Text;
  ShowMessage(Descript + ' copied to clipboard');
end;

function DataToStr(pData: PByte; DataLen: Integer): String;
const HEX_DIGITS = '0123456789ABCDEF';
var i: Integer;
begin
  Result := '';
  for i := 0 to DataLen-1 do begin
    Result := Result + HEX_DIGITS[1+((pData^ shr 4) and $0F)] + HEX_DIGITS[1+(pData^ and $0F)];
    inc(pData);
  end;
end;
{$ENDIF}

const
  BLOCK_SIZE = 16;

var
  alog,log:    Array [0..255] of Cardinal;
  S,Si:        Array [0..255] of Byte;
  T1,T2,T3,T4,
  T5,T6,T7,T8,
  U1,U2,U3,U4: Array [0..255] of Cardinal;
  rcon:        Array [0..29] of Byte;

const
  shifts: Array [0..2,0..3,0..1] of Cardinal = (
            ( (0, 0), (1, 3), (2, 2), (3, 1) ),
            ( (0, 0), (1, 5), (2, 4), (3, 3) ),
            ( (0, 0), (1, 7), (3, 5), (4, 4) )
          );

  MAX_ROUNDS     = 14; // max. number of rounds (for session key)
  MAX_BLOCK_SIZE = 32; // max. block size in bytes (for session key)
  MAX_BC         = MAX_BLOCK_SIZE div 4;

type
  TSessionKey = record
    Ke,Kd:  Array [0..MAX_ROUNDS {not -1!}, 0..MAX_BC-1] of Cardinal;
    ROUNDS,BC,BlockSize: Cardinal;
  end;
  PSessionKey = ^TSessionKey;


function MakeKey(pData: PByte; DataLen: Integer; BlockSize: Integer): TSessionKey;
const MAX_KC = 32 div 4; // max. keylength / 4
var
  ROUNDS,BC,ROUND_KEY_COUNT,KC: Cardinal;
  i,j,t,tt,r,rconpointer:       Cardinal;
  tk:  Array [0..MAX_KC-1] of Cardinal;
begin
  {$IFDEF RIJNDAEL_DEBUG} debug_log('MakeKey( ' + DataToStr(pData,DataLen) + ', ' + IntToStr(BlockSize) + ' )'); {$ENDIF}
  if pData = nil then raise Exception.Create('Key missing');
  if not (DataLen in [16,24,32]) then raise Exception.Create('Invalid key length');
  if not (BlockSize in [16,24,32]) then raise Exception.Create('Invalid block size');

  if DataLen = 16 then
    ROUNDS := 6 + BlockSize div 4
  else if (DataLen = 24) and (BlockSize <> 32) then
    ROUNDS := 12
  else
    ROUNDS := 14;
  Assert(ROUNDS in [10,12,14]);

  FillChar(Result, SizeOf(Result), 0);
  BC := BlockSize div 4;
  ROUND_KEY_COUNT := (ROUNDS + 1) * BC;
  KC := DataLen div 4;
  FillChar(tk, SizeOf(tk), 0);
  Result.ROUNDS := ROUNDS;
  Result.BC     := BC;
  Result.BlockSize := BlockSize;

  j := 0;
  for i := 0 to KC-1 do begin
    tk[i] := (PByte(PChar(pData) + j    )^ shl 24)
          or (PByte(PChar(pData) + j + 1)^ shl 16)
          or (PByte(PChar(pData) + j + 2)^ shl  8)
          or  PByte(PChar(pData) + j + 3)^;
    inc(j,4);
  end;
  t := 0; j := 0;
  while (j < KC) and (t < ROUND_KEY_COUNT) do begin
    Result.Ke[t div BC, t mod BC] := tk[j];
    Result.Kd[ROUNDS - t div BC, t mod BC] := tk[j];
    inc(j); inc(t);
  end;
  rconpointer := 0;
  while t < ROUND_KEY_COUNT do begin
    tt := tk[KC - 1];
    tk[0] :=  tk[0]
          xor ((S[(tt shr 16) and $FF] and $FF) shl 24)
          xor ((S[(tt shr  8) and $FF] and $FF) shl 16)
          xor ((S[ tt         and $FF] and $FF) shl  8)
          xor ((S[(tt shr 24) and $FF] and $FF)       )
          xor ((rcon[rconpointer]      and $FF) shl 24);
    inc(rconpointer);
    if KC <> 8 then begin
      j := 0;
      for i := 1 to KC-1 do begin
        tk[i] := tk[i] xor tk[j];
        inc(j);
      end;
    end else begin
      j := 0;
      for i := 1 to (KC div 2)-1 do begin
        tk[i] := tk[i] xor tk[j];
        inc(j);
      end;
      tt := tk[KC div 2 - 1];
      tk[KC div 2] :=  tk[KC div 2]
                   xor ((S[ tt         and $FF] and $FF)       )
                   xor ((S[(tt shr  8) and $FF] and $FF) shl  8)
                   xor ((S[(tt shr 16) and $FF] and $FF) shl 16)
                   xor ((S[(tt shr 24) and $FF] and $FF) shl 24);
      j := KC div 2;
      for i := j+1 to KC-1 do begin
        tk[i] := tk[i] xor tk[j];
        inc(j);
      end;
    end;
    j := 0;
    while (j < KC) and (t < ROUND_KEY_COUNT) do begin
      Result.Ke[t div BC, t mod BC] := tk[j];
      Result.Kd[ROUNDS - t div BC, t mod BC] := tk[j];
      inc(j); inc(t);
    end;
  end;

  for r := 1 to ROUNDS-1 do begin
    for j := 0 to BC-1 do begin
      tt := Result.Kd[r,j];
      Result.Kd[r,j] :=  U1[(tt shr 24) and $FF]
                     xor U2[(tt shr 16) and $FF]
                     xor U3[(tt shr  8) and $FF]
                     xor U4[ tt         and $FF];
    end;
  end;

end;

procedure BlockEncrypt_DefBlockSize(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey);
// default blocksize
var
  t_0,t_1,t_2,t_3,a0,a1,a2,a3,r,tt: Cardinal;
begin
  t_0 := (    (PByte(PChar(pIn) + InOffset     )^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  1)^ shl 16)
           or (PByte(PChar(pIn) + InOffset +  2)^ shl  8)
           or (PByte(PChar(pIn) + InOffset +  3)^       )
         ) xor SessionKey.Ke[0,0];
  t_1 := (    (PByte(PChar(pIn) + InOffset +  4)^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  5)^ shl 16)
           or (PByte(PChar(pIn) + InOffset +  6)^ shl  8)
           or (PByte(PChar(pIn) + InOffset +  7)^       )
         ) xor SessionKey.Ke[0,1];
  t_2 := (    (PByte(PChar(pIn) + InOffset +  8)^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  9)^ shl 16)
           or (PByte(PChar(pIn) + InOffset + 10)^ shl  8)
           or (PByte(PChar(pIn) + InOffset + 11)^       )
         ) xor SessionKey.Ke[0,2];
  t_3 := (    (PByte(PChar(pIn) + InOffset + 12)^ shl 24)
           or (PByte(PChar(pIn) + InOffset + 13)^ shl 16)
           or (PByte(PChar(pIn) + InOffset + 14)^ shl  8)
           or (PByte(PChar(pIn) + InOffset + 15)^       )
         ) xor SessionKey.Ke[0,3];
  for r := 1 to SessionKey.ROUNDS-1 do begin
    a0 := (     T1[(t_0 shr 24) and $FF]
            xor T2[(t_1 shr 16) and $FF]
            xor T3[(t_2 shr  8) and $FF]
            xor T4[ t_3         and $FF]
          ) xor SessionKey.Ke[r,0];
    a1 := (     T1[(t_1 shr 24) and $FF]
            xor T2[(t_2 shr 16) and $FF]
            xor T3[(t_3 shr  8) and $FF]
            xor T4[ t_0         and $FF]
          ) xor SessionKey.Ke[r,1];
    a2 := (     T1[(t_2 shr 24) and $FF]
            xor T2[(t_3 shr 16) and $FF]
            xor T3[(t_0 shr  8) and $FF]
            xor T4[ t_1         and $FF]
          ) xor SessionKey.Ke[r,2];
    a3 := (     T1[(t_3 shr 24) and $FF]
            xor T2[(t_0 shr 16) and $FF]
            xor T3[(t_1 shr  8) and $FF]
            xor T4[ t_2         and $FF]
          ) xor SessionKey.Ke[r,3];
    t_0 := a0;
    t_1 := a1;
    t_2 := a2;
    t_3 := a3;
  end;
  tt := SessionKey.Ke[SessionKey.ROUNDS,0];
  PByte(PChar(pOut)     )^ := (S[(t_0 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  1)^ := (S[(t_1 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) +  2)^ := (S[(t_2 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) +  3)^ := (S[ t_3         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Ke[SessionKey.ROUNDS,1];
  PByte(PChar(pOut) +  4)^ := (S[(t_1 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  5)^ := (S[(t_2 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) +  6)^ := (S[(t_3 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) +  7)^ := (S[ t_0         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Ke[SessionKey.ROUNDS,2];
  PByte(PChar(pOut) +  8)^ := (S[(t_2 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  9)^ := (S[(t_3 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) + 10)^ := (S[(t_0 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) + 11)^ := (S[ t_1         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Ke[SessionKey.ROUNDS,3];
  PByte(PChar(pOut) + 12)^ := (S[(t_3 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) + 13)^ := (S[(t_0 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) + 14)^ := (S[(t_1 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) + 15)^ := (S[ t_2         and $FF] xor  tt        ) and $FF;
  {$IFDEF RIJNDAEL_DEBUG} debug_log('CT='+DataToStr(pOut,BLOCK_SIZE)); {$ENDIF}
end;

procedure BlockEncrypt(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey);
var
  SC: Cardinal;
  i,j,r,tt,s_1,s_2,s_3: Cardinal;
  a,t: Array [0..MAX_BC-1] of Cardinal;
begin
  if SessionKey.BlockSize = BLOCK_SIZE then begin
    BlockEncrypt_DefBlockSize(pIn, pOut, InOffset, SessionKey);
    exit;
  end;

  if      SessionKey.BC = 4 then SC := 0
  else if SessionKey.BC = 6 then SC := 1
  else                           SC := 2;
  s_1 := shifts[SC,1,0];
  s_2 := shifts[SC,2,0];
  s_3 := shifts[SC,3,0];

  for i := 0 to SessionKey.BC-1 do begin
    t[i] := (    (PByte(PChar(pIn) + InOffset     )^ shl 24)
              or (PByte(PChar(pIn) + InOffset +  1)^ shl 16)
              or (PByte(PChar(pIn) + InOffset +  2)^ shl  8)
              or (PByte(PChar(pIn) + InOffset +  3)^       )
            ) xor SessionKey.Ke[0,i];
    inc(InOffset,4);
  end;
  for r := 1 to SessionKey.ROUNDS-1 do begin
    for i := 0 to SessionKey.BC-1 do begin
      a[i] := (     T1[(t[ i                         ] shr 24) and $FF]
                xor T2[(t[(i + s_1) mod SessionKey.BC] shr 16) and $FF]
                xor T3[(t[(i + s_2) mod SessionKey.BC] shr  8) and $FF]
                xor T4[ t[(i + s_3) mod SessionKey.BC]         and $FF]
              ) xor SessionKey.Ke[r,i];
    end;
    t := a;
  end;
  j := 0;
  for i := 0 to SessionKey.BC-1 do begin
    tt := SessionKey.Ke[SessionKey.ROUNDS,i];
    PByte(PChar(pOut) + j)^ := (S[(t[ i                         ] shr 24) and $FF] xor (tt shr 24)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (S[(t[(i + s_1) mod SessionKey.BC] shr 16) and $FF] xor (tt shr 16)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (S[(t[(i + s_2) mod SessionKey.BC] shr  8) and $FF] xor (tt shr  8)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (S[ t[(i + s_3) mod SessionKey.BC]         and $FF] xor  tt        ) and $FF; inc(j);
  end;
  {$IFDEF RIJNDAEL_DEBUG} debug_log('CT='+DataToStr(pOut,SessionKey.BlockSize)); {$ENDIF}
end;

procedure BlockDecrypt_DefBlockSize(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey);
// default blocksize
var
  t_0,t_1,t_2,t_3,a0,a1,a2,a3,r,tt: Cardinal;
begin
  t_0 := (    (PByte(PChar(pIn) + InOffset +  0)^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  1)^ shl 16)
           or (PByte(PChar(pIn) + InOffset +  2)^ shl  8)
           or (PByte(PChar(pIn) + InOffset +  3)^       )
         ) xor SessionKey.Kd[0,0];
  t_1 := (    (PByte(PChar(pIn) + InOffset +  4)^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  5)^ shl 16)
           or (PByte(PChar(pIn) + InOffset +  6)^ shl  8)
           or (PByte(PChar(pIn) + InOffset +  7)^       )
         ) xor SessionKey.Kd[0,1];
  t_2 := (    (PByte(PChar(pIn) + InOffset +  8)^ shl 24)
           or (PByte(PChar(pIn) + InOffset +  9)^ shl 16)
           or (PByte(PChar(pIn) + InOffset + 10)^ shl  8)
           or (PByte(PChar(pIn) + InOffset + 11)^       )
         ) xor SessionKey.Kd[0,2];
  t_3 := (    (PByte(PChar(pIn) + InOffset + 12)^ shl 24)
           or (PByte(PChar(pIn) + InOffset + 13)^ shl 16)
           or (PByte(PChar(pIn) + InOffset + 14)^ shl  8)
           or (PByte(PChar(pIn) + InOffset + 15)^       )
         ) xor SessionKey.Kd[0,3];
  for r := 1 to SessionKey.ROUNDS-1 do begin
    a0 := (     T5[(t_0 shr 24) and $FF]
            xor T6[(t_3 shr 16) and $FF]
            xor T7[(t_2 shr  8) and $FF]
            xor T8[ t_1         and $FF]
          ) xor SessionKey.Kd[r,0];
    a1 := (     T5[(t_1 shr 24) and $FF]
            xor T6[(t_0 shr 16) and $FF]
            xor T7[(t_3 shr  8) and $FF]
            xor T8[ t_2         and $FF]
          ) xor SessionKey.Kd[r,1];
    a2 := (     T5[(t_2 shr 24) and $FF]
            xor T6[(t_1 shr 16) and $FF]
            xor T7[(t_0 shr  8) and $FF]
            xor T8[ t_3         and $FF]
          ) xor SessionKey.Kd[r,2];
    a3 := (     T5[(t_3 shr 24) and $FF]
            xor T6[(t_2 shr 16) and $FF]
            xor T7[(t_1 shr  8) and $FF]
            xor T8[ t_0         and $FF]
          ) xor SessionKey.Kd[r,3];
    t_0 := a0;
    t_1 := a1;
    t_2 := a2;
    t_3 := a3;
  end;
  tt := SessionKey.Kd[SessionKey.ROUNDS,0];
  PByte(PChar(pOut)     )^ := (Si[(t_0 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  1)^ := (Si[(t_3 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) +  2)^ := (Si[(t_2 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) +  3)^ := (Si[ t_1         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Kd[SessionKey.ROUNDS,1];
  PByte(PChar(pOut) +  4)^ := (Si[(t_1 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  5)^ := (Si[(t_0 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) +  6)^ := (Si[(t_3 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) +  7)^ := (Si[ t_2         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Kd[SessionKey.ROUNDS,2];
  PByte(PChar(pOut) +  8)^ := (Si[(t_2 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) +  9)^ := (Si[(t_1 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) + 10)^ := (Si[(t_0 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) + 11)^ := (Si[ t_3         and $FF] xor  tt        ) and $FF;
  tt := SessionKey.Kd[SessionKey.ROUNDS,3];
  PByte(PChar(pOut) + 12)^ := (Si[(t_3 shr 24) and $FF] xor (tt shr 24)) and $FF;
  PByte(PChar(pOut) + 13)^ := (Si[(t_2 shr 16) and $FF] xor (tt shr 16)) and $FF;
  PByte(PChar(pOut) + 14)^ := (Si[(t_1 shr  8) and $FF] xor (tt shr  8)) and $FF;
  PByte(PChar(pOut) + 15)^ := (Si[ t_0         and $FF] xor  tt        ) and $FF;
  {$IFDEF RIJNDAEL_DEBUG} debug_log('PT='+DataToStr(pOut,BLOCK_SIZE)); {$ENDIF}
end;

procedure BlockDecrypt(pIn,pOut: PByte; InOffset: Integer; var SessionKey: TSessionKey); overload;
var
  SC: Cardinal;
  i,j,r,tt,s_1,s_2,s_3: Cardinal;
  a,t: Array [0..MAX_BC-1] of Cardinal;
begin
  if SessionKey.BlockSize = BLOCK_SIZE then begin
    BlockDecrypt_DefBlockSize(pIn, pOut, InOffset, SessionKey);
    exit;
  end;

  if      SessionKey.BC = 4 then SC := 0
  else if SessionKey.BC = 6 then SC := 1
  else                           SC := 2;
  s_1 := shifts[SC,1,1];
  s_2 := shifts[SC,2,1];
  s_3 := shifts[SC,3,1];

  for i := 0 to SessionKey.BC-1 do begin
    t[i] := (    (PByte(PChar(pIn) + InOffset     )^ shl 24)
              or (PByte(PChar(pIn) + InOffset +  1)^ shl 16)
              or (PByte(PChar(pIn) + InOffset +  2)^ shl  8)
              or (PByte(PChar(pIn) + InOffset +  3)^       )
            ) xor SessionKey.Kd[0,i];
    inc(InOffset,4);
  end;
  for r := 1 to SessionKey.ROUNDS-1 do begin
    for i := 0 to SessionKey.BC-1 do begin
      a[i] := (     T5[(t[ i                         ] shr 24) and $FF]
                xor T6[(t[(i + s_1) mod SessionKey.BC] shr 16) and $FF]
                xor T7[(t[(i + s_2) mod SessionKey.BC] shr  8) and $FF]
                xor T8[ t[(i + s_3) mod SessionKey.BC]         and $FF]
              ) xor SessionKey.Kd[r,i];
    end;
    t := a;
  end;
  j := 0;
  for i := 0 to SessionKey.BC-1 do begin
    tt := SessionKey.Kd[SessionKey.ROUNDS,i];
    PByte(PChar(pOut) + j)^ := (Si[(t[ i                         ] shr 24) and $FF] xor (tt shr 24)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (Si[(t[(i + s_1) mod SessionKey.BC] shr 16) and $FF] xor (tt shr 16)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (Si[(t[(i + s_2) mod SessionKey.BC] shr  8) and $FF] xor (tt shr  8)) and $FF; inc(j);
    PByte(PChar(pOut) + j)^ := (Si[ t[(i + s_3) mod SessionKey.BC]         and $FF] xor  tt        ) and $FF; inc(j);
  end;
  {$IFDEF RIJNDAEL_DEBUG} debug_log('PT='+DataToStr(pOut,SessionKey.BlockSize)); {$ENDIF}
end;

function self_test(keysize: Integer): Boolean; overload;
begin
  Result := self_test(keysize, BLOCK_SIZE);
end;

function self_test(keysize,blocksize: Integer): Boolean; overload;
var
  kb,pt,ct,cpt: packed Array of Byte;
  i:            Cardinal;
  key:          TSessionKey;
{$IFDEF RIJNDAEL_DEBUG}
  j: Cardinal;
  x: String;
{$ENDIF}
begin
  Result := True;
  SetLength(kb, keysize);
  SetLength(pt, blocksize);
  SetLength(ct, blocksize);
  SetLength(cpt,blocksize);
  for i := 0 to keysize-1 do kb[i] := i;
  for i := 0 to blocksize-1 do pt[i] := i;

{$IFDEF RIJNDAEL_DEBUG}
  debug_clear;
  debug_log('Rijndael selftest');
  debug_log('KEYSIZE=' + IntToStr(8*keysize));
  debug_log('BLOCKSIZE=' + IntToStr(8*blocksize));
  debug_log('KEY=' + DataToStr(@(kb[0]),Length(kb)));
  debug_log('');
{$ENDIF}

  key := MakeKey(@(kb[0]), Length(kb), blocksize);
{$IFDEF RIJNDAEL_DEBUG}
  debug_log(''); debug_log('Ke[]:' ); for i := 0 to key.ROUNDS do begin x := ''; for j := 0 to key.BC-1 do x := x + '0x' + IntToHex(key.Ke[i,j], 8)+', '; debug_log(x); end;
  debug_log(''); debug_log('Kd[]:' ); for i := 0 to key.ROUNDS do begin x := ''; for j := 0 to key.BC-1 do x := x + '0x' + IntToHex(key.Kd[i,j], 8)+', '; debug_log(x); end;
{$ENDIF}

  BlockEncrypt(@(pt[0]), @(ct[0]),  0, key);
  BlockDecrypt(@(ct[0]), @(cpt[0]), 0, key);
  if not CompareMem(@(pt[0]), @(cpt[0]), Length(pt)) then begin
    Result := False;
    {$IFDEF RIJNDAEL_DEBUG}
      ShowMessage('Rijndael selftest(' +inttostr(keysize)+','+inttostr(blocksize)+') FAILED!');
    {$ENDIF}
  end;

{$IFDEF RIJNDAEL_DEBUG}
  debug_show('Rijndael selftest(' +inttostr(keysize)+','+inttostr(blocksize)+')');
{$ENDIF}
end;

procedure Init;
type
  T4ByteArray = Array [0..3] of Byte;

  function mul(a,b: Cardinal): Cardinal;
  begin
    if (a <> 0) and (b <> 0) then
      Result := alog[(log[a and $FF] + log[b and $FF]) mod 255]
    else
      Result := 0;
  end;

  function mul4(a: Cardinal; b: T4ByteArray): Cardinal;
  var a0,a1,a2,a3: Byte;
  begin
    if a = 0 then begin Result := 0; exit; end;
    a := log[a and $FF];
    if b[0] <> 0 then a0 := alog[(a + log[b[0] and $FF]) mod 255] and $FF else a0 := 0;
    if b[1] <> 0 then a1 := alog[(a + log[b[1] and $FF]) mod 255] and $FF else a1 := 0;
    if b[2] <> 0 then a2 := alog[(a + log[b[2] and $FF]) mod 255] and $FF else a2 := 0;
    if b[3] <> 0 then a3 := alog[(a + log[b[3] and $FF]) mod 255] and $FF else a3 := 0;
    Result := (a0 shl 24) or (a1 shl 16) or (a2 shl 8) or a3;
  end;

const
  ROOT = $11B;
  A: Array [0..7,0..7] of Byte = (
       (1, 1, 1, 1, 1, 0, 0, 0),
       (0, 1, 1, 1, 1, 1, 0, 0),
       (0, 0, 1, 1, 1, 1, 1, 0),
       (0, 0, 0, 1, 1, 1, 1, 1),
       (1, 0, 0, 0, 1, 1, 1, 1),
       (1, 1, 0, 0, 0, 1, 1, 1),
       (1, 1, 1, 0, 0, 0, 1, 1),
       (1, 1, 1, 1, 0, 0, 0, 1)
     );
  B: Array [0..7] of Byte = (0, 1, 1, 0, 0, 0, 1, 1);
  G: Array [0..3] of T4ByteArray = (
       (2, 1, 1, 3),
       (3, 2, 1, 1),
       (1, 3, 2, 1),
       (1, 1, 3, 2)
     );

var
  i,j,t,ss,r: Cardinal;
  box,cox: Array [0..255,0..7] of Byte;
  AA:      Array [0..3,0..7] of Byte;
  iG:      Array [0..3] of T4ByteArray;
  pivot,tmp: Byte;
{$IFDEF RIJNDAEL_DEBUG}
  x:    String;
  tim0,tim1: Int64;
{$ENDIF}
begin
  {$IFDEF RIJNDAEL_DEBUG} tim0 := GetTickCount; {$ENDIF}
  alog[0] := 1;
  for i := 1 to 255 do begin
    j := (alog[i-1] shl 1) xor alog[i-1];
    if (j and $100) <> 0 then j := j xor ROOT;
    alog[i] := j;
  end;
  for i := 1 to 255 do log[alog[i]] := i;

  FillChar(box, SizeOf(box), 0);
  box[1,7] := 1;
  for i := 2 to 255 do begin
    j := alog[255 - log[i]];
    for t := 0 to 7 do box[i,t] := (j shr (7-t)) and $01;
  end;
  for i := 0 to 255 do begin
    for t := 0 to 7 do begin
      cox[i,t] := B[t];
      for j := 0 to 7 do cox[i,t] := cox[i,t] xor (A[t,j] * box[i,j]);
    end;
  end;

  for i := 0 to 255 do begin
    S[i] := cox[i,0] shl 7;
    for t := 1 to 7 do S[i] := S[i] xor (cox[i,t] shl (7-t));
    Si[S[i] and $FF] := i;
  end;

  for i := 0 to 3 do begin
    for j := 0 to 3 do AA[i,j] := G[i,j];
    AA[i,i+4] := 1;
  end;

  for i := 0 to 3 do begin
    pivot := AA[i,i];
    if pivot = 0 then begin
      t := i+1;
      while (AA[t,i] = 0) and (t < 4) do inc(t);
      if t = 4 then raise Exception.Create('G matrix is not invertible');
      for j := 0 to 7 do begin
        tmp := AA[i,j]; AA[i,j] := AA[t,j]; AA[t,j] := tmp;
      end;
      pivot := AA[i,i];
    end;
    for j := 0 to 7 do begin
      if AA[i,j] <> 0 then
        AA[i,j] := alog[(255 + log[AA[i,j] and $FF] - log[pivot and $FF]) mod 255];
    end;
    for t := 0 to 3 do begin
      if i <> t then begin
        for j := i+1 to 7 do AA[t,j] := AA[t,j] xor mul(AA[i,j], AA[t,i]);
        AA[t,i] := 0;
      end;
    end;
  end;
  for i := 0 to 3 do begin
    for j := 0 to 3 do iG[i,j] := AA[i,j+4];
  end;

  for t := 0 to 255 do begin
    ss := S[t];
    T1[t] := mul4(ss, G[0]);
    T2[t] := mul4(ss, G[1]);
    T3[t] := mul4(ss, G[2]);
    T4[t] := mul4(ss, G[3]);

    ss := Si[t];
    T5[t] := mul4(ss, iG[0]);
    T6[t] := mul4(ss, iG[1]);
    T7[t] := mul4(ss, iG[2]);
    T8[t] := mul4(ss, iG[3]);

    U1[t] := mul4(t, iG[0]);
    U2[t] := mul4(t, iG[1]);
    U3[t] := mul4(t, iG[2]);
    U4[t] := mul4(t, iG[3]);
  end;

  rcon[0] := 1; r := 1;
  for t := 1 to 29 do begin
    r := mul(2,r);
    rcon[t] := r;
  end;

{$IFDEF RIJNDAEL_DEBUG}
  tim1 := GetTickCount; if tim1 < tim0 then inc(tim1, $100000000);
  debug_clear;
  debug_log('Rijndael static data');
  debug_log(''); debug_log('S[]:' );   for i := 0 to 15 do begin x := ''; for j := 0 to 15 do x := x + '0x' + IntToHex(S [i*16+j], 2)+', '; debug_log(x); end;
  debug_log(''); debug_log('Si[]:');   for i := 0 to 15 do begin x := ''; for j := 0 to 15 do x := x + '0x' + IntToHex(Si[i*16+j], 2)+', '; debug_log(x); end;
  debug_log(''); debug_log('iG[]:');   for i := 0 to  3 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(iG[i,j],    2)+', '; debug_log(x); end;
  debug_log(''); debug_log('T1[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T1[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T2[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T2[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T3[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T3[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T4[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T4[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T5[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T5[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T6[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T6[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T7[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T7[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('T8[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(T8[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('U1[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(U1[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('U2[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(U2[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('U3[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(U3[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('U4[]:');   for i := 0 to 63 do begin x := ''; for j := 0 to  3 do x := x + '0x' + IntToHex(U4[i*4+j],  8)+', '; debug_log(x); end;
  debug_log(''); debug_log('rcon[]:'); for i := 0 to  4 do begin x := ''; for j := 0 to  5 do x := x + '0x' + IntToHex(rcon[i*6+j],2)+', '; debug_log(x); end;
  debug_log('');
  debug_log('Initialization time: ' + IntToStr(tim1-tim0) + ' ms');
  debug_show('Rijndael static data');
{$ENDIF}
end;

{ TRijndael }

constructor TRijndael.Create(AKeySize,ABlockSize: Integer);
begin
  inherited Create;
  FKeySize := AKeySize;
  if (AKeySize <> 128) and (AKeySize <> 192) and (AKeySize <> 256) then
    raise Exception.Create('Invalid key size');
  FBlockSize := ABlockSize;
  if (ABlockSize <> 128) and (ABlockSize <> 192) and (ABlockSize <> 256) then
    raise Exception.Create('Invalid block size');

  New(PSessionKey(FpSessionKey));
  FillChar(FpSessionKey^, SizeOf(TSessionKey), 0);
end;

destructor TRijndael.Destroy;
begin
  if FpSessionKey <> nil then Dispose(PSessionKey(FpSessionKey));
  inherited;
end;

procedure TRijndael.Initialize(pKey: PByte; KeyLen: Integer);
var KeyData: packed Array [0..255] of Byte; // max.key len
begin
  FillChar(KeyData, SizeOf(KeyData), 0);
  if KeyLen >= (FKeySize shr 3) then
    Move(pKey^, KeyData, KeyLen)
  else
    Move(pKey^, KeyData, (FKeySize shr 3));

  PSessionKey(FpSessionKey)^ := Rijndael.MakeKey(@KeyData, (FKeySize shr 3), (FBlockSize shr 3));
end;

procedure TRijndael.Encipher(pIn, pOut: PByte);
begin
  Rijndael.BlockEncrypt(pIn, pOut, 0, PSessionKey(FpSessionKey)^);
end;

procedure TRijndael.Decipher(pIn, pOut: PByte);
begin
  Rijndael.BlockDecrypt(pIn, pOut, 0, PSessionKey(FpSessionKey)^);
end;

function TRijndael.GetBlockSize: Integer;
begin
  Result := FBlockSize;
end;

function TRijndael.GetKeySize: Integer;
begin
  Result := FKeySize;
end;

function TRijndael.GetCipherName: String;
begin
  Result := CIPHERNAME_RIJNDAEL;
end;

initialization
{$IFDEF RIJNDAEL_DEBUG}
  DebugLog := TStringList.Create;
{$ENDIF}
  Init;

finalization
{$IFDEF RIJNDAEL_DEBUG}
  DebugLog.Free;
{$ENDIF}

end.
