//PROFILE-NO
unit QueueFile;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses FreenetUtils, ColumnConfig, Misc;

type
  TFileStats = record
    FailCount: Array [TFreenetThreadFailReason] of Integer;
    FailCountTotal: Integer;
    SuccCount: Integer;
    LastSucc:  TDateTime;
    LastFail:  TDateTime;
  end;
  TFileStatType = (fstOverall, fstSession);

  TQueueFile = class (TObject)
  private
    FFreeze: Boolean;
    function GetNumActiveThreads: Integer;
    procedure SetFreeze(const Value: Boolean);
  protected
    //procedure UpdateListItem(item: TListItem = nil);
  public
    Key:       String;
    Progress:  String;
    PrepBasename: String;
    Size:      Int64;
    DateAdded: TDateTime;
    Comment:   String;
    UniqueID:  Int64; // not saved
    LastTick:  Int64; // not saved, used for priority system

    Stats:     Array [TFileStatType] of TFileStats; // stats (overall and session); only overall is saved

    FullRepaint: Boolean; // not saved

    constructor Create;

    procedure AbortAllThreads;
    function  GetFilename(FullPath: Boolean = False): String; virtual; abstract;
    procedure ClearStats(StatType: TFileStatType);
    procedure WriteToIniFile (Section, Appendix: String; Ini: TSafeSaveMemIniFile); virtual;
    procedure ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile); virtual;

    property  NumActiveThreads: Integer read GetNumActiveThreads;
    property  Freeze: Boolean read FFreeze write SetFreeze;
  end;

  // if new states need to be added: add at the end!
  TInsFileStatus = (ifsUnprepared,ifsReady,ifsInserting,ifsDone,ifsError,ifsPreparing,ifsAborting);
  TInsQueueFile = class(TQueueFile)
  private
    FStatus:   TInsFileStatus;
    procedure SetStatus(const Value: TInsFileStatus);
  public
    OrgFilename:  String;
    IsHeal:    Boolean;
    IsDirect:  Boolean;
    InsertStyle: TInsertStyle;
    HeaderInserts: Integer; // not saved

    DisplayColumns: Array [TInsColumn] of String;

    function  GetFilename(FullPath: Boolean = False): String; override;
    procedure WriteToIniFile (Section, Appendix: String; Ini: TSafeSaveMemIniFile); override;
    procedure ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile); override;

    property  Status: TInsFileStatus read FStatus write SetStatus;
  end;

  // if new states need to be added: add at the end!
  TGetFileStatus = (gfsReady,gfsGetting,gfsDecoding,gfsValidating,gfsDone,gfsError,gfsDecoded,gfsAborting);
  TGetQueueFile = class(TQueueFile)
    FStatus:   TGetFileStatus;
  private
    procedure SetStatus(const Value: TGetFileStatus);
  public
    SavFilename:  String;
    Activekey: String;
    CheckSum:  String;
    Retry:     Integer; // not saved
    BlockStat: String;  // not saved
    RestartAt: TDateTime;
    Priority:  Integer;
    PriorityBoost: Boolean;  // not saved, flag to mark file ready for decoding/validating

    Debug_GetAll:    Boolean; // not saved

    DisplayColumns: Array [TGetColumn] of String;

    procedure AbortAllGetThreads;
    function  GetFilename(FullPath: Boolean = False): String; override;
    procedure WriteToIniFile (Section, Appendix: String; Ini: TSafeSaveMemIniFile); override;
    procedure ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile); override;

    property  Status: TGetFileStatus read FStatus write SetStatus;
  end;

implementation

uses SysUtils,Forms,Main;

{ Misc }

procedure FullQualifyPrepareName(var Basename: String);
begin
  if Basename <> '' then
    Basename := ExtractFilePath(Application.ExeName) + 'Data\' + ExtractFilename(Basename);
end;

{ TQueueFile }

var QueueFileIDCounter: Int64 = 0;

constructor TQueueFile.Create;
var fst: TFileStatType;
begin
  inherited;
  inc(QueueFileIDCounter);
  UniqueID := QueueFileIDCounter;
  for fst := Low(TFileStatType) to High(TFileStatType) do ClearStats(fst);
end;

function TQueueFile.GetNumActiveThreads: Integer;
var
  i:   Integer;
  Thr: TFreenetThread;
begin
  Result := 0;
  if not Assigned(FrmMain.ThreadPool) then exit;
  for i := 0 to FrmMain.ThreadPool.Count-1 do begin
    Thr := FrmMain.ThreadPool.Items[i];
    if Thr.UserData = Self then inc(Result);
  end;
end;

procedure TQueueFile.AbortAllThreads;
var
  i: Integer;
  AnyThread: TFreenetThread;
begin
  for i := 0 to FrmMain.ThreadPool.Count-1 do begin
    AnyThread := TFreenetThread(FrmMain.ThreadPool.Items[i]);
    if AnyThread.UserData = Self then begin
      AnyThread.Unnecessary := True;
      AnyThread.Terminate;
    end;
  end;
end;

procedure TQueueFile.ClearStats(StatType: TFileStatType);
var fr: TFreenetThreadFailReason;
begin
  for fr := Low(TFreenetThreadFailReason) to High(TFreenetThreadFailReason) do Stats[StatType].FailCount[fr] := 0;
  Stats[StatType].FailCountTotal := 0;
  Stats[StatType].SuccCount      := 0;
  Stats[StatType].LastSucc       := 0;
  Stats[StatType].LastFail       := 0;
end;

procedure TQueueFile.SetFreeze(const Value: Boolean);
begin
  if FFreeze <> Value then FullRepaint := True; // color update
  FFreeze := Value;
end;

procedure TQueueFile.WriteToIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
var fr: TFreenetThreadFailReason;
begin
  Ini.WriteString  (Section, 'Prep'     + Appendix,  ExtractFilename(Self.PrepBasename));
  Ini.WriteInteger (Section, 'Size'     + Appendix,  Self.Size);
  if Self.DateAdded <> 0 then Ini.WriteDateTime(Section, 'DateTime' + Appendix, Self.DateAdded);
  Ini.WriteBool    (Section, 'Freeze'   + Appendix,  Self.Freeze);
  Ini.WriteString  (Section, 'Key'      + Appendix,  Self.Key);
  Ini.WriteString  (Section, 'Comment'  + Appendix,  Self.Comment);
  // stats
  Ini.WriteInteger (Section, 'StatsSucc' + Appendix, Self.Stats[fstOverall].SuccCount);
  Ini.WriteInteger (Section, 'StatsFail' + Appendix, Self.Stats[fstOverall].FailCountTotal);
  for fr := Low(TFreenetThreadFailReason) to High(TFreenetThreadFailReason) do
    Ini.WriteInteger (Section, 'StatsFail' + IntToStr(Ord(fr)) + Appendix, Self.Stats[fstOverall].FailCount[fr]);
  if Self.Stats[fstOverall].LastSucc <> 0 then Ini.WriteDateTime(Section, 'StatsLastSucc' + Appendix, Self.Stats[fstOverall].LastSucc);
  if Self.Stats[fstOverall].LastFail <> 0 then Ini.WriteDateTime(Section, 'StatsLastFail' + Appendix, Self.Stats[fstOverall].LastFail);
end;

procedure TQueueFile.ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
var fr: TFreenetThreadFailReason;
begin
  Self.PrepBasename := Ini.ReadString  (Section, 'Prep'     + Appendix, '');
  Self.Size         := Ini.ReadInteger (Section, 'Size'     + Appendix, 0);
  Self.DateAdded    := Ini.ReadDateTime(Section, 'DateTime' + Appendix, 0);
  Self.Freeze       := Ini.ReadBool    (Section, 'Freeze'   + Appendix, False);
  Self.Key          := Ini.ReadString  (Section, 'Key'      + Appendix, '');
  Self.Comment      := Ini.ReadString  (Section, 'Comment'  + Appendix, '');
  // stats
  Self.Stats[fstOverall].SuccCount       := Ini.ReadInteger (Section, 'StatsSucc' + Appendix, 0);
  Self.Stats[fstOverall].FailCountTotal  := Ini.ReadInteger (Section, 'StatsFail' + Appendix, 0);
  for fr := Low(TFreenetThreadFailReason) to High(TFreenetThreadFailReason) do
    Self.Stats[fstOverall].FailCount[fr] := Ini.ReadInteger (Section, 'StatsFail' + IntToStr(Ord(fr)) + Appendix, 0);
  Self.Stats[fstOverall].LastSucc        := Ini.ReadDateTime(Section, 'StatsLastSucc' + Appendix, 0);
  Self.Stats[fstOverall].LastFail        := Ini.ReadDateTime(Section, 'StatsLastFail' + Appendix, 0);
  Self.Stats[fstSession] := Self.Stats[fstOverall];
end;

{ TInsQueueFile }

function TInsQueueFile.GetFilename(FullPath: Boolean): String;
begin
  Result := OrgFilename;
  if not FullPath then Result := ExtractFilename(Result);
end;

procedure TInsQueueFile.SetStatus(const Value: TInsFileStatus);
begin
  if FStatus <> Value then begin
    if (FStatus in [ifsDone,ifsError]) or (Value in [ifsDone,ifsError]) then FullRepaint := True; // color change
  end;
  FStatus := Value;
end;

procedure TInsQueueFile.WriteToIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
begin
  inherited;
  Ini.WriteInteger (Section, 'Status'   + Appendix, Ord(Self.Status));
  Ini.WriteString  (Section, 'File'     + Appendix, Self.OrgFilename);
  Ini.WriteBool    (Section, 'IsHeal'   + Appendix, Self.IsHeal);
  Ini.WriteBool    (Section, 'IsDirect' + Appendix, Self.IsDirect);
  Ini.WriteString  (Section, 'Style'    + Appendix, InsertStyleToStr(Self.InsertStyle));
end;

procedure TInsQueueFile.ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
var sStyle: String;
begin
  inherited;
  Self.Status       := TInsFileStatus(Ini.ReadInteger (Section, 'Status'   + Appendix, 0));
  Self.OrgFilename  :=                Ini.ReadString  (Section, 'File'     + Appendix, '');
  Self.IsHeal       :=                Ini.ReadBool    (Section, 'IsHeal'   + Appendix, False);
  Self.IsDirect     :=                Ini.ReadBool    (Section, 'IsDirect' + Appendix, False);
  sStyle            :=                Ini.ReadString  (Section, 'Style'    + Appendix, '');
  if sStyle <> '' then Self.InsertStyle := StrToInsertStyle(sStyle) else Self.InsertStyle := InsertStyle_Fuqid;
  if Self.IsHeal then Self.IsDirect := True;
  Self.Progress     := '';
  FullQualifyPrepareName(Self.PrepBasename);
end;


{ TGetQueueFile }

procedure TGetQueueFile.AbortAllGetThreads;
// mark all download threads for this queue entry as obsolete and terminate them
var
  i: Integer;
  GetThread: TDownloadThread;
begin
  for i := 0 to FrmMain.ThreadPool.Count-1 do begin
    if TFreenetThread(FrmMain.ThreadPool.Items[i]) is TDownloadThread then begin
      GetThread := TFreenetThread(FrmMain.ThreadPool.Items[i]) as TDownloadThread;
      if GetThread.UserData = Self then begin
        GetThread.Unnecessary := True;
        GetThread.Terminate;
      end;
    end;
  end;
end;

function TGetQueueFile.GetFilename(FullPath: Boolean): String;
begin
  Result := SavFilename;
  if not FullPath then Result := ExtractFilename(Result);
end;

procedure TGetQueueFile.SetStatus(const Value: TGetFileStatus);
begin
  if FStatus <> Value then begin
    if (FStatus in [gfsDone,gfsError]) or (Value in [gfsDone,gfsError]) then FullRepaint := True; // color change
  end;
  FStatus := Value;
end;


procedure TGetQueueFile.WriteToIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
begin
  inherited;
  Ini.WriteInteger (Section, 'Status'   + Appendix, Ord(Self.Status));
  Ini.WriteString  (Section, 'File'     + Appendix, Self.SavFilename);
  Ini.WriteString  (Section, 'ActKey'   + Appendix, Self.Activekey);
  Ini.WriteString  (Section, 'ChkSum'   + Appendix, Self.CheckSum);
  if Self.RestartAt <> 0 then Ini.WriteDateTime(Section, 'Restart' + Appendix, Self.RestartAt);
  Ini.WriteInteger (Section, 'Priority' + Appendix, Self.Priority);
end;

procedure TGetQueueFile.ReadFromIniFile(Section, Appendix: String; Ini: TSafeSaveMemIniFile);
begin
  inherited;
  Self.Status       := TGetFileStatus(Ini.ReadInteger (Section, 'Status'   + Appendix, 0));
  Self.SavFilename  :=                Ini.ReadString  (Section, 'File'     + Appendix, '');
  Self.Activekey    :=                Ini.ReadString  (Section, 'ActKey'   + Appendix, '');
  Self.CheckSum     :=                Ini.ReadString  (Section, 'ChkSum'   + Appendix, '');
  Self.RestartAt    :=                Ini.ReadDateTime(Section, 'Restart'  + Appendix, 0);
  Self.Priority     :=                Ini.ReadInteger (Section, 'Priority' + Appendix, 0);
  Self.Retry        := 0;
  Self.BlockStat    := '';
  Self.Progress     := '';
  FullQualifyPrepareName(Self.PrepBasename);
end;


end.
