//PROFILE-NO
unit Misc;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses Windows,SysUtils,Forms,Dialogs,IniFiles,Classes,Menus;

{-DEFINE DEBUG_SSMIF_DISABLE_QUICKREAD}
{-DEFINE DEBUG_SSMIF_DISABLE_QUICKWRITE}

type
  TSafeSaveMemIniFile = class (TMemIniFile)
    FQuickWrite:        Boolean;
    FQuickWriteSection: String;
    FQuickWriteStrings: TStrings;
    FQuickRead:         Boolean;
    FQuickReadSection:  String;
    FQuickReadStrings:  TStrings;
    FQuickReadIndex:    Integer;
    FOldContent:        TStringList;
  private
    function IsDirty: Boolean;
  public
    constructor Create(const FileName: string);
    destructor Destroy; override;
    procedure UpdateFile; override;
    procedure WriteString(const Section, Ident, Value: String); override;
    function  ReadString(const Section, Ident, Default: String): String; override;
    procedure QuickWriteBegin(const NewSection: String = '');
    procedure QuickWriteNewSection(const NewSection: String);
    procedure QuickWriteEnd;
    procedure QuickReadBegin(const NewSection: String = '');
    procedure QuickReadNewSection(const NewSection: String);
    procedure QuickReadEnd;
  end;

  TCachedFile = class;
  ECachedFile = class (Exception);

  TCachedFileList = class
  private
    FList: TList;
  public
    constructor Create;
    destructor  Destroy; override;
    procedure   Add(ACachedFile: TCachedFile);
    procedure   Remove(ACachedFile: TCachedFile);
    function    OpenFile(AFilename: String; MustExist: Boolean): TCachedFile;
    function    FlushAll(CloseOldFiles: Boolean = False): Integer;
    procedure   CloseFile(AFilename: String; FlushIt: Boolean);
    procedure   CloseAll(FlushIt: Boolean);
  end;

  TCachedFile = class
  private
    FFilename:  String;
    FMemStream: TMemoryStream;
    FDirty:     Boolean;
    FCacheList: TCachedFileList;
    FLastAccess: TDateTime;
    function    GetSize: Int64;
  public
    constructor Create(ACacheList: TCachedFileList; AFilename: String; InitFromFile: Boolean);
    destructor  Destroy; override;
    procedure   SetContents(PData: Pointer; Count: Longint);
    procedure   GetContents(PData: Pointer; Count: Longint);
    function    FlushToFile: Boolean;
    property    Filename: String read FFilename;
    property    Size: Int64 read GetSize;
    property    LastAccess: TDateTime read FLastAccess;
  end;

var
  gCachedFiles: TCachedFileList = nil;

type
  TFrostMessageHeader = record
    From:     String;
    Subject:  String;
    DateTime: TDateTime;
  end;
  PFrostMessageHeader = ^TFrostMessageHeader;
  TFrostMessage = record
    Header:   TFrostMessageHeader;
    Board:    String;
    Body:     String;
    FileAttachments: String;
  end;
  PFrostMessage = ^TFrostMessage;
  TFrostBoard = class
  public
    BoardName: String;
    IsFolder:  Boolean;
    Next:      TFrostBoard;
    Parent,Child: TFrostBoard;
    destructor Destroy; override;
  end;

procedure AdjustFormSize(Form: TForm);
procedure DoFileContextMenu(Filename: String; ExVerbs: Boolean = False);
procedure ShowDebugLines(Lines: String; Caption: String = 'Debug');
function  CommandLineOption(const OptionName: String): Boolean;
function  ExecuteSaveDialogSafely(Dlg: TSaveDialog): Boolean;
function  CreateFileStream(const FileName: String; Mode: Word): TFileStream;
function  LastPos(const SubStr, Str: String): Integer;
function  GetFrostXMLAsString(Filename: String): String;
function  GetFrostMessage(Filename: String; out Msg: TFrostMessage): Boolean;
function  GetFrostMessageFromXML(sXML: String; out Msg: TFrostMessage): Boolean;
function  GetFrostBoardListFromXML(sXML: String): TFrostBoard;
function  GetInfoFromFrostFilename(Filename: String; out Board: String; out Date: TDateTime; out Index: Integer): Boolean;

implementation

uses Controls,ComCtrls,StdCtrls,Messages,ShellAPI,ShlObj,ActiveX;

// adjust these when recompiling with different desktop settings
// DeltaH = Form.Height - Form.ClientHeight
// DeltaW = Form.Width  - Form.ClientWidth   in design time mode.

const
  DESIGN_DELTAH = 27;
  DESIGN_DELTAW =  8;

type
  TFileContextMenu = class
  private
    FcmMenu1:      IContextMenu;
    FcmMenu2:      IContextMenu2;
    FcmMenu3:      IContextMenu3;
    FMenuType:     Integer;
    FWindowHandle: HWND;
    procedure WndMethod(var M: TMessage);
  public
    procedure DoFileContextMenu(Filename: String; ExVerbs: Boolean);
  end;

  TAnchorInfo = record
    Control: TControl;
    Anchors: TAnchors;
    OrigXY:  TPoint;
    OrigWH:  TPoint;
  end;
  PAnchorInfo = ^TAnchorInfo;

procedure AdjustFormSize(Form: TForm);
var w,h: Integer;
begin
  w := Form.Width  - DESIGN_DELTAW;
  h := Form.Height - DESIGN_DELTAH;
  if Form.ClientWidth  <> w then Form.ClientWidth  := w;
  if Form.ClientHeight <> h then Form.ClientHeight := h;
end;

procedure DoFileContextMenu(Filename: String; ExVerbs: Boolean);
var Fctm: TFileContextMenu;
begin
  Fctm := TFileContextMenu.Create;
  try
    FCtm.DoFileContextMenu(Filename, ExVerbs);
  finally
    Fctm.Free;
  end;
end;

procedure ShowDebugLines(Lines,Caption: String);
var
  form:   TForm;
  riched: TRichEdit;
begin
  form := nil;
  try
    form := TForm.Create(Application);
    form.Position := poScreenCenter;
    form.Caption  := Caption;
    form.Width    := 600;
    form.Height   := 400;
    riched := TRichEdit.Create(form);
    riched.Parent := form;
    riched.Font.Name := 'Courier New';
    riched.Font.Size := 8;
    riched.Align := alClient;
    riched.ScrollBars := ssBoth;
    riched.WordWrap := False;
    riched.Text := Lines;
    form.ShowModal;
  finally
    form.Free;
  end;
end;

function  CommandLineOption(const OptionName: String): Boolean;
var i: Integer;
begin
  for i := 1 to ParamCount do
    if CompareText(ParamStr(i), OptionName) = 0 then begin
      Result := True; exit;
    end;
  Result := False;
end;

function  ExecuteSaveDialogSafely(Dlg: TSaveDialog): Boolean;
var t,t0: Int64;
begin
  // dirty workaround if the filename is invalid and save dialog doesn't show
  t0 := GetTickCount; Result := Dlg.Execute;
  t := GetTickCount; if t < t0 then t := t + $100000000;
  if (not Result) and ((t-t0) <= 300) then begin
    Dlg.FileName := '';
    Result := Dlg.Execute;
  end;
end;

function  CreateFileStream(const FileName: String; Mode: Word): TFileStream;
// create a file stream
// if it fails for Creating or Writing: retry a couple of times before giving up
// (may be locked by an AV scanner or by Windows Explorer)
var
  iTry: Integer;
begin
  iTry := 0;
  while True do begin
    inc(iTry);
    try
      Result := TFileStream.Create(FileName, Mode);
      exit;
    except
      if iTry > 3 then
        raise
      else
        Sleep(1000);
    end;
  end;
end;

function  LastPos(const SubStr, Str: String): Integer;
var
  i,len,cnt: Integer;
  s:         String;
begin
  Result := 0; cnt := 0;
  s := Str; len := Length(SubStr);
  repeat
    i := Pos(SubStr, s);
    if i > 0 then begin
      Result := cnt + i;
      Delete(s, 1, i + len - 1);
      inc(cnt, i + len - 1);
    end;
  until i = 0;
end;

function  GetFrostXMLAsString(Filename: String): String;
var
  Str:         TFileStream;
  pBuf,pXML:   PChar;
  BufLen:      Int64;
  Len:         Int64;
begin
  Str := nil; pBuf := nil;
  try
    Str := CreateFileStream(Filename, fmOpenRead or fmShareDenyWrite);
    Len := Str.Size; if Len < 4 then raise Exception.Create('Invalid frost XML file'); // can't be valid..
    BufLen := Len + 1;
    GetMem(pBuf, BufLen);
    (pBuf + BufLen - 1)^ := #0; // additional byte for quickly reversing byte order
    Str.ReadBuffer(pBuf^, Len);
    pXML := pBuf;
    // skip BOM, determine byte order
    if      PWord(pBuf)^ = $FFFE then begin inc(pXML,3); dec(Len,2); end // skip BOM, switch byte order
    else if PWord(pBuf)^ = $FEFF then begin inc(pXML,2); dec(Len,2); end // skip BOM
    else if PByte(pBuf)^ = $00   then       inc(pXML,1)                  // switch byte order
    ;
    Result := WideCharLenToString(PWideChar(pXML), Len div 2);
  finally
    Str.Free;
    if Assigned(pBuf)  then FreeMem(pBuf);
  end;
end;

function  GetFrostMessage(Filename: String; out Msg: TFrostMessage): Boolean;
begin
  try
    Result := GetFrostMessageFromXML( GetFrostXMLAsString(Filename), Msg);
  except
    Result := False;
  end;
end;

function GetDataFromToken(Main,Token: String; out Data: String): Boolean;
var
  iPos1,iPos2:   Integer;
  sTok2:         String;
begin
  Result := False;
  // extract Token from Main
  sTok2 := Token; if Pos(' ',sTok2) <> 0 then Delete(sTok2, Pos(' ',sTok2), Length(sTok2));
  iPos1 := Pos('<' + Token + '>', Main);
  iPos2 := LastPos('</' + sTok2 + '>', Main);
  if (iPos1 = 0) or (iPos2 = 0) or (iPos1 > iPos2) then exit;
  Delete(Main, iPos2, Length(Main)); Delete(Main, 1, iPos1 + Length(Token) + 1);
  Token := '<![CDATA['; if CompareText(Copy(Main,1,Length(Token)), Token) = 0 then Delete(Main,1,Length(Token));
  Token := ']]>';       if CompareText(Copy(Main,Length(Main)-Length(Token)+1,Length(Token)), Token) = 0 then Delete(Main,Length(Main)-Length(Token)+1,Length(Token));
  Data := Main;
  Result := True;
end;

procedure RemoveToken(var Main: String; Token: String);
var
  iPos1,iPos2:   Integer;
  sTok2:         String;
begin
  // remove Token from Main
  sTok2 := Token; if Pos(' ',sTok2) <> 0 then Delete(sTok2, Pos(' ',sTok2), Length(sTok2));
  iPos1 := Pos('<' + Token + '>', Main);
  iPos2 := LastPos('</' + sTok2 + '>', Main);
  if (iPos1 = 0) or (iPos2 = 0) or (iPos1 > iPos2) then exit;
  Delete(Main, iPos1, iPos2 - iPos1 + Length(sTok2) + 3);
end;

function  GetFrostMessageFromXML(sXML: String; out Msg: TFrostMessage): Boolean;
var
  s,sMain,sRest:       String;
  iPos1:               Integer;
  sToken,sFName,sKey:  String;
  day,month,year:      Word;
  hour,minut,sec:      Word;
  sl:                  TStringList;
begin
  Result := False;
  sl := nil;
  try
    Msg.Header.From     := '';
    Msg.Header.Subject  := '';
    Msg.Header.DateTime := 0;
    Msg.Board           := '';
    Msg.Body            := '';
    Msg.FileAttachments := '';

    // extract FrostMessage block
    if not GetDataFromToken(sXML, 'FrostMessage', sMain) then exit;

    // extract and remove message body (else could interfere with our primitive XML parsing)
    if not GetDataFromToken(sMain, 'Body', Msg.Body) then exit;
    RemoveToken(sMain, 'Body');

    GetDataFromToken(sMain, 'From',    Msg.Header.From);
    GetDataFromToken(sMain, 'Subject', Msg.Header.Subject);
    GetDataFromToken(sMain, 'Date',    s);
    year  := StrToIntDef(Copy(s,1,Pos('.',s)-1),0); Delete(s,1,Pos('.',s));
    month := StrToIntDef(Copy(s,1,Pos('.',s)-1),0); Delete(s,1,Pos('.',s));
    day   := StrToIntDef(s,0);
    try Msg.Header.DateTime := EncodeDate(year,month,day); except Msg.Header.DateTime := 0; end;
    GetDataFromToken(sMain, 'Time',    s);
    if Pos('GMT',s) <> 0 then Delete(s, Pos('GMT',s), 3);
    s := Trim(s);
    hour  := StrToIntDef(Copy(s,1,Pos(':',s)-1),0); Delete(s,1,Pos(':',s));
    minut := StrToIntDef(Copy(s,1,Pos(':',s)-1),0); Delete(s,1,Pos(':',s));
    sec   := StrToIntDef(s,0);
    try Msg.Header.DateTime := Msg.Header.DateTime + EncodeTime(hour,minut,sec,0); except end;
    GetDataFromToken(sMain, 'Board', Msg.Board);

    sRest := sMain;

    // extract file attachments
    while True do begin
      s := sRest;
      sToken := '<Attachment type="file">';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then break;
      Delete(s, 1, iPos1 + Length(sToken) - 1);
      sToken := '</Attachment>';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then break;
      sRest := Copy(s, iPos1 + Length(sToken), Length(s));
      Delete(s, iPos1, Length(s));

      sToken := '<name><![CDATA[';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then continue;
      Delete(s, 1, iPos1 + Length(sToken) - 1);
      sToken := ']]></name>';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then continue;
      sFName := Copy(s, 1, iPos1 - 1); Delete(s, 1, iPos1 + Length(sToken) - 1);
      sToken := '<key>';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then continue;
      Delete(s, 1, iPos1 + Length(sToken) - 1);
      sToken := '</key>';
      iPos1 := Pos(sToken, s); if iPos1 = 0 then continue;
      sKey := Copy(s, 1, iPos1 - 1); Delete(s, 1, iPos1 + Length(sToken) - 1);
      if Copy(sKey,1,4) <> 'CHK@' then continue; // for now only CHK-keys

      if sl = nil then sl := TStringList.Create;
      sl.Add(sKey + '/' + sFName);
    end;
    if sl <> nil then Msg.FileAttachments := sl.Text;

    Result := True;
  finally
    sl.Free;
  end;
end;

function  GetFrostBoardListFromXML(sXML: String): TFrostBoard;

  function GetBoards(ParentBoard: TFrostBoard; sChildboards: String): Boolean;
  var
    sMain,sTok1,sTok2,s,sChilds: String;
    iPos1,iPos2,cnt: Integer;
    sBoardName: String;
    bFolder:    Boolean;
    LastChild,ThisChild: TFrostBoard;
  begin
    Result := False;
    sMain := sChildboards;
    LastChild := nil;
    while True do begin
      sTok1 := '<FrostBoardTreeEntry';
      iPos1 := Pos(sTok1, sMain); if iPos1 = 0 then break;
      Delete(sMain, 1, iPos1 + Length(sTok1) - 1);
      iPos1 := Pos('>', sMain); if iPos1 = 0 then exit;
      s := Copy(sMain,1,iPos1-1);
      bFolder := Pos('isfolder="true"', s) <> 0;
      Delete(sMain, 1, iPos1);
      iPos1 := Pos('<FrostBoardTreeEntry', sMain);
      iPos2 := Pos('</FrostBoardTreeEntry>', sMain); if iPos2 = 0 then exit;
      if (iPos1 = 0) or (iPos2 < iPos1) then
        s := Copy(sMain,1,iPos2)
      else
        s := Copy(sMain,1,iPos1);
      if not GetDataFromToken(s, 'name', sBoardName) then exit;

      ThisChild := TFrostBoard.Create;
      ThisChild.Parent := ParentBoard;
      if LastChild = nil then ParentBoard.Child := ThisChild else LastChild.Next := ThisChild;
      ThisChild.BoardName := sBoardName;
      ThisChild.IsFolder  := bFolder;
      LastChild := ThisChild;

      if (iPos1 = 0) or (iPos2 < iPos1) then begin
        Delete(sMain,1,iPos2 + Length('</FrostBoardTreeEntry>')-1);
        continue;
      end;

      // has childboards
      Delete(sMain,1,iPos1-1);
      sChilds := ''; cnt := 0;
      repeat
        sTok1 := '<FrostBoardTreeEntry';   iPos1 := Pos(sTok1, sMain);
        sTok2 := '</FrostBoardTreeEntry>'; iPos2 := Pos(sTok2, sMain);
        if (iPos1 <> 0) and (iPos1 < iPos2) then begin
          inc(cnt);
          sChilds := sChilds + Copy(sMain,1,iPos1 + Length(sTok1) - 1);
          Delete(sMain,1,iPos1 + Length(sTok1) - 1);
        end else if (iPos2 <> 0) then begin
          dec(cnt);
          sChilds := sChilds + Copy(sMain,1,iPos2 + Length(sTok2) - 1);
          Delete(sMain,1,iPos2 + Length(sTok2) - 1);
        end else
          raise Exception.Create('WTF? error reading frost boards');
      until cnt < 0;
      if not GetBoards(ThisChild, sChilds) then exit;

    end;
    Result := True;
  end;

var
  ok: Boolean;
  sMain: String;
begin
  Result := nil;
  ok := False;
  try
    // extract FrostBoardTree block
    if not GetDataFromToken(sXML, 'FrostBoardTree', sMain) then exit;

    Result := TFrostBoard.Create;
    Result.IsFolder := True;
    ok := GetBoards(Result, sMain);

  finally
    if not ok then Result.Free;
  end;
end;

function  GetInfoFromFrostFilename(Filename: String; out Board: String; out Date: TDateTime; out Index: Integer): Boolean;
var
  s: String;
  year,month,day: Word;
  i: Integer;
begin
  Result := False;

  try
    Filename := ExtractFileName(Filename);
    s := Copy(Filename,1,Pos('-',Filename)-1); Delete(Filename,1,Pos('-',Filename));
    year  := StrToIntDef(Copy(s,1,Pos('.',s)-1),0); Delete(s,1,Pos('.',s));
    month := StrToIntDef(Copy(s,1,Pos('.',s)-1),0); Delete(s,1,Pos('.',s));
    day   := StrToIntDef(s,0);
    Date  := EncodeDate(year,month,day);

    for i := Length(Filename) downto 1 do begin
      if Filename[i] = '-' then begin
        s := Copy(Filename, i+1, Length(Filename));
        Board := Copy(Filename,1,i-1);
        Index := StrToIntDef(Copy(s,1,Pos('.',s)-1),0);
        Result := True;
        break;
      end;
    end;
  except
  end;
end;

{ TSafeSaveMemIniFile }

constructor TSafeSaveMemIniFile.Create(const FileName: string);
begin
  inherited Create(FileName);
  FOldContent := TStringList.Create;
  GetStrings(FOldContent);
end;

destructor TSafeSaveMemIniFile.Destroy;
begin
  if FQuickWrite then QuickWriteEnd; // frees list
  if FQuickRead  then QuickReadEnd;  // frees list
  FOldContent.Free;
  inherited;
end;

procedure TSafeSaveMemIniFile.UpdateFile;
var
  OldName,BakName,TmpName: String;
begin
  if not IsDirty then exit; // do nothing if there are no changes
  
  // save to a .tmp file first
  OldName := FileName; TmpName := Filename + '.tmp'; BakName := Filename + '.bak';
  Rename(TmpName, False); // add .tmp to filename
  try
    inherited UpdateFile; // save data (to .tmp)
    if FileExists(OldName) then begin
      DeleteFile(BakName);  // delete old bak file if one exists
      if not RenameFile(OldName, BakName) then raise Exception.Create('Failed to backup ini file');
    end;
    if not RenameFile(TmpName, OldName) then raise Exception.Create('Rename failed for saving ini file');
  finally
    Rename(OldName, False); // rename back to old filename
  end;

  GetStrings(FOldContent);
end;

procedure TSafeSaveMemIniFile.QuickWriteBegin(const NewSection: String);
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKWRITE} exit; {$ENDIF}
  Assert(not FQuickWrite);
  FQuickWriteStrings := TStringList.Create;
  GetStrings(FQuickWriteStrings);
  FQuickWrite := True;
  FQuickWriteSection := '';
  if NewSection <> '' then QuickWriteNewSection(NewSection);
end;

procedure TSafeSaveMemIniFile.QuickWriteEnd;
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKWRITE} exit; {$ENDIF}
  Assert(FQuickWrite);
  SetStrings(FQuickWriteStrings);
  FQuickWriteStrings.Free;
  FQuickWrite := False;
end;

procedure TSafeSaveMemIniFile.QuickWriteNewSection(const NewSection: String);
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKWRITE} exit; {$ENDIF}
  Assert(FQuickWrite);
  FQuickWriteStrings.Add('[' + NewSection + ']');
  FQuickWriteSection := NewSection;
end;

procedure TSafeSaveMemIniFile.WriteString(const Section, Ident, Value: String);
begin
  if not FQuickWrite then
    inherited
  else begin
    Assert(Section = FQuickWriteSection);
    FQuickWriteStrings.Add(Ident + '=' + Value);
  end;
end;

procedure TSafeSaveMemIniFile.QuickReadBegin(const NewSection: String);
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKREAD} exit; {$ENDIF}
  Assert(not FQuickRead);
  FQuickReadStrings := TStringList.Create;
  FQuickRead := True;
  FQuickReadSection := '';
  FQuickReadIndex   := 0;
  if NewSection <> '' then QuickReadNewSection(NewSection);
end;

procedure TSafeSaveMemIniFile.QuickReadEnd;
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKREAD} exit; {$ENDIF}
  Assert(FQuickRead);
  FQuickReadStrings.Free;
  FQuickRead := False;
end;

procedure TSafeSaveMemIniFile.QuickReadNewSection(const NewSection: String);
begin
  {$IFDEF DEBUG_SSMIF_DISABLE_QUICKREAD} exit; {$ENDIF}
  Assert(FQuickRead);
  ReadSectionValues(NewSection, FQuickReadStrings);
  FQuickReadSection := NewSection;
  FQuickReadIndex   := 0;
end;

function TSafeSaveMemIniFile.ReadString(const Section, Ident, Default: String): String;
var
  ok: Boolean;
begin
  if not FQuickRead then
    Result := inherited ReadString(Section,Ident,Default)
  else begin
    Assert(Section = FQuickReadSection);
    ok := (FQuickReadIndex >= 0) and (FQuickReadIndex < FQuickReadStrings.Count);
    if ok then ok := AnsiCompareText(FQuickReadStrings.Names[FQuickReadIndex], Ident) = 0;
    if ok then begin
      Result := Copy(FQuickReadStrings.Strings[FQuickReadIndex], Length(Ident)+2, MaxInt);
      inc(FQuickReadIndex);
    end else
      Result := inherited ReadString(Section,Ident,Default)
  end;
end;

function TSafeSaveMemIniFile.IsDirty: Boolean;
var
  List: TStringList;
  i:    Integer;
begin
  Result := False;
  List := TStringList.Create;
  try
    GetStrings(List);
    if List.Count <> FOldContent.Count then
      Result := True
    else begin
      for i := 0 to List.Count-1 do
        if List.Strings[i] <> FOldContent.Strings[i] then begin
          Result := True;
          break;
        end;
    end;
  finally
    List.Free;
  end;
end;

{ TCachedFile }

constructor TCachedFile.Create(ACacheList: TCachedFileList; AFilename: String; InitFromFile: Boolean);
begin
  inherited Create;
  FCacheList := ACacheList;
  FFilename  := AFilename;
  FMemStream := TMemoryStream.Create;
  FDirty     := False;
  FLastAccess:= Now;
  if InitFromFile then FMemStream.LoadFromFile(FFilename);
  if Assigned(FCacheList) then FCacheList.Add(Self);
end;

destructor TCachedFile.Destroy;
begin
  if Assigned(FCacheList) then FCacheList.Remove(Self);
  FMemStream.Free;
  inherited;
end;

procedure TCachedFile.SetContents(PData: Pointer; Count: Integer);
begin
  FLastAccess:= Now;

  // check if content is identical
  if Count = FMemStream.Size then begin
    if CompareMem(PData, FMemStream.Memory, Count) then exit;
  end;

  FMemStream.SetSize(Count);
  FMemStream.Seek(0, soFromBeginning);
  FMemStream.WriteBuffer(PData^, Count);
  FDirty := True;
end;

procedure TCachedFile.GetContents(PData: Pointer; Count: Integer);
begin
  FLastAccess:= Now;

  if Count <> FMemStream.Size then raise ECachedFile.Create('Size mismatch');
  FMemStream.Seek(0, soFromBeginning);
  FMemStream.ReadBuffer(PData^, Count);
end;

function  TCachedFile.FlushToFile: Boolean;
var Str: TFileStream;
begin
  Result := False;
  if not FDirty then exit;

  Str := CreateFileStream(FFilename, fmCreate or fmShareExclusive);
  try
    FMemStream.SaveToStream(Str);
    FDirty := False;
    Result := True;
  finally
    Str.Free;
  end;
end;

function TCachedFile.GetSize: Int64;
begin
  Result := FMemStream.Size;
end;

{ TCachedFileList }

constructor TCachedFileList.Create;
begin
  inherited Create;
  FList := TList.Create;
end;

destructor TCachedFileList.Destroy;
var oldcnt: Integer;
begin
  while FList.Count > 0 do begin
    oldcnt := FList.Count;
    TCachedFile(FList.Items[0]).Free; // that removes it from FList too
    if FList.Count = oldcnt then FList.Delete(0); // just in case something failed
  end;
  FList.Free;
  inherited;
end;

procedure TCachedFileList.Add(ACachedFile: TCachedFile);
begin
  FList.Add(ACachedFile);
end;

procedure TCachedFileList.Remove(ACachedFile: TCachedFile);
var idx: Integer;
begin
  idx := FList.IndexOf(ACachedFile);
  if idx >= 0 then FList.Delete(idx);
end;

function TCachedFileList.OpenFile(AFilename: String; MustExist: Boolean): TCachedFile;
var i: Integer;
begin
  for i := 0 to FList.Count-1 do begin
    Result := FList.Items[i];
    if AnsiCompareText(Result.Filename, AFilename) = 0 then exit;
  end;
  Result := TCachedFile.Create(Self, AFilename, MustExist);
end;

function  TCachedFileList.FlushAll(CloseOldFiles: Boolean = False): Integer;
const OLD_FILE_INTERVAL = 30/(24*60); // 30 minutes
var
  i: Integer;
  CachedFile: TCachedFile;
  dt: TDateTime;
begin
  Result := 0;
  dt := Now;
  i := 0;
  while i < FList.Count do begin
    CachedFile := FList.Items[i];
    if CachedFile.FlushToFile then inc(Result);
    if ((dt - CachedFile.LastAccess) > OLD_FILE_INTERVAL) or (dt < CachedFile.LastAccess) then
      CachedFile.Free
    else
      inc(i);
  end;
end;

procedure TCachedFileList.CloseFile(AFilename: String; FlushIt: Boolean);
var
  i:          Integer;
  CachedFile: TCachedFile;
begin
  for i := 0 to FList.Count-1 do begin
    CachedFile := FList.Items[i];
    if AnsiCompareText(CachedFile.Filename, AFilename) = 0 then begin
      if FlushIt then CachedFile.FlushToFile;
      CachedFile.Free;
      exit;
    end;
  end;
end;

procedure TCachedFileList.CloseAll(FlushIt: Boolean);
var CachedFile: TCachedFile;
begin
  while FList.Count > 0 do begin
    CachedFile := FList.Items[0];
    if FlushIt then CachedFile.FlushToFile;
    CachedFile.Free;
  end;
end;

{ TFileContextMenu }

procedure TFileContextMenu.DoFileContextMenu(Filename: String; ExVerbs: Boolean);
  procedure Check(hr: HRESULT; Msg: String);
  begin if Failed(hr) then raise Exception.Create(Msg); end;

const
  CMF_EXTENDEDVERBS = $00000100;
  MIN_ID = 1;
  MAX_ID = 30000;
var
  ShM:        IMalloc;
  sfDesktop:  IShellFolder;
  sfPath:     IShellFolder;
  MenuHandle: HMENU;
  WidePath:   WideString;
  pRelIDFile: PItemIDList;
  pAbsIDPath: PItemIDList;
  dummy1,dummy2: Cardinal;
  Id:         Integer;
  InvokeCmdInfo: TCMInvokeCommandInfo;
  Flags:      Cardinal;
begin
  ShM := nil; pRelIDFile := nil; pAbsIDPath := nil;
  MenuHandle := 0; FWindowHandle := 0;

  Check( SHGetMalloc(ShM), 'SHGetMalloc failed' );
  try
    // create a window to handle messages (for IContextMenu2/3)
    FWindowHandle := AllocateHWnd(WndMethod);

    // first get the IShellFolder for the containing folder
    Check( SHGetDesktopFolder( sfDesktop ), 'Get desktop folder failed' );
    WidePath := ExtractFilePath(Filename); dummy2 := 0;
    Check( sfDesktop.ParseDisplayName(FWindowHandle, nil, PWideChar(WidePath), dummy1, pAbsIDPath, dummy2), 'ParseDisplayName(1) failed' );
    Check( sfDesktop.BindToObject(pAbsIDPath, nil, IID_IShellFolder, sfPath), 'BindToObject failed' );
    // get the ID list of the file (relative to its folder)
    WidePath := ExtractFileName(Filename); dummy2 := 0;
    Check( sfPath.ParseDisplayName(FWindowHandle, nil, PWideChar(WidePath), dummy1, pRelIDFile, dummy2), 'ParseDisplayName(2) failed' );

    // get IContextMenu
    FMenuType := 1;
    Check( sfPath.GetUIObjectOf(FWindowHandle, 1, pRelIDFile, IID_IContextMenu, nil, FcmMenu1), 'GetUIObjectOf failed' );
    // see if we can get an IContextMenu2 or 3
    if Succeeded( FcmMenu1.QueryInterface(StringToGUID(SID_IContextMenu3), FcmMenu3) ) then FMenuType := 3
    else if Succeeded( FcmMenu1.QueryInterface(IID_IContextMenu2, FcmMenu2) ) then FMenuType := 2;

    // Create and fill the menu
    MenuHandle := CreatePopupMenu(); if MenuHandle = 0 then raise Exception.Create('CreatePopupMenu failed');
    Flags := CMF_EXPLORE; if ExVerbs then Flags := Flags or CMF_EXTENDEDVERBS;
    case FMenuType of
      3:   Check( FcmMenu3.QueryContextMenu(MenuHandle, 0, MIN_ID, MAX_ID, Flags ), 'QueryContextMenu failed' );
      2:   Check( FcmMenu2.QueryContextMenu(MenuHandle, 0, MIN_ID, MAX_ID, Flags ), 'QueryContextMenu failed' );
      else Check( FcmMenu1.QueryContextMenu(MenuHandle, 0, MIN_ID, MAX_ID, Flags ), 'QueryContextMenu failed' );
    end;

    // Display menu
    Id := Integer( TrackPopupMenu(MenuHandle, TPM_RETURNCMD or TPM_LEFTALIGN, Mouse.CursorPos.X, Mouse.CursorPos.Y, 0, FWindowHandle, nil) );
    if Id = 0 then exit;

    // Carry out action
    FillChar(InvokeCmdInfo, SizeOf(InvokeCmdInfo), 0); InvokeCmdInfo.cbSize := SizeOf(InvokeCmdInfo);
    InvokeCmdInfo.hwnd   := Application.Handle;
    InvokeCmdInfo.lpVerb := PChar(Id - MIN_ID);
    InvokeCmdInfo.nShow  := SW_SHOWNORMAL;
    case FMenuType of
      3:   Check( FcmMenu3.InvokeCommand( InvokeCmdInfo ), 'InvokeCommand failed' );
      2:   Check( FcmMenu2.InvokeCommand( InvokeCmdInfo ), 'InvokeCommand failed' );
      else Check( FcmMenu1.InvokeCommand( InvokeCmdInfo ), 'InvokeCommand failed' );
    end;
  finally
    if MenuHandle <> 0 then DestroyMenu(MenuHandle);
    if FWindowHandle <> 0 then DeallocateHWnd(FWindowHandle);
    ShM.Free( pRelIDFile );
    ShM.Free( pAbsIDPath );
    // don't free interfaces!
  end;
end;

procedure TFileContextMenu.WndMethod(var M: TMessage);
var lRes: LRESULT;
begin
  if FMenuType > 1 then begin
    case M.Msg of
      WM_MENUCHAR:      if (FMenuType > 2) then begin
                          lRes := 0;
                          FcmMenu3.HandleMenuMsg2(M.Msg, M.WParam, M.LParam, lRes);
                          M.Result := lRes;
                          exit;
                        end;
      WM_DRAWITEM,
      WM_MEASUREITEM:   if M.WParam = 0 then begin
                          if FMenuType = 2 then FcmMenu2.HandleMenuMsg(M.Msg, M.WParam, M.LParam)
                                           else FcmMenu3.HandleMenuMsg(M.Msg, M.WParam, M.LParam);
                          M.Result := 1;
                          exit;
                        end;
      WM_INITMENUPOPUP: begin
                          if FMenuType = 2 then FcmMenu2.HandleMenuMsg(M.Msg, M.WParam, M.LParam)
                                           else FcmMenu3.HandleMenuMsg(M.Msg, M.WParam, M.LParam);
                          M.Result := 0;
                          exit;
                        end;
    end;
  end;
  M.Result := DefWindowProc(FWindowHandle, M.Msg, M.WParam, M.LParam);
end;

{ TFrostBoard }

destructor TFrostBoard.Destroy;
begin
  if (Parent <> nil) and (Parent.Child = Self) then Parent.Child := Next;
  while Child <> nil do Child.Free;
  inherited;
end;

end.
