unit FrostSearchResults;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, ExtCtrls, Misc;

type
  TFrmFrostSearchResults = class(TForm)
    LvResults: TListView;
    REdPreview: TRichEdit;
    Splitter1: TSplitter;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure LvResultsColumnClick(Sender: TObject; Column: TListColumn);
    procedure LvResultsCompare(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer);
    procedure LvResultsChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
  private
    FInfoList: TList;
    FInited:   Boolean;
    FSearchText: String;
    FSortBy:     Integer;
    FSortDesc:   Boolean;
    FDisplayed:  Integer;
    procedure Init;
    procedure DisplayMessage;
  public
  end;

type
  TFrostSearchResult = record
    Filename: String;
    Header:   TFrostMessageHeader;
  end;
  PFrostSearchResult = ^TFrostSearchResult;

procedure ShowFrostSearchResults(ResultList: TList; SearchText: String);

implementation

{$R *.dfm}

type
  TFrostFileInfo = record
    SearchResult:  TFrostSearchResult;
    MsgID:         Integer;
    Boardname:     String;
  end;
  PFrostFileInfo = ^TFrostFileInfo;

procedure ShowFrostSearchResults(ResultList: TList; SearchText: String);
var
  Frm: TFrmFrostSearchResults;
  i:   Integer;
  p:   PFrostSearchResult;
  pI:  PFrostFileInfo;
begin
  Frm := TFrmFrostSearchResults.Create(Application);
  for i := 0 to ResultList.Count-1 do begin
    p := ResultList.Items[i];
    New(pI);
    pI^.SearchResult := p^;
    Frm.FInfoList.Add(pI);
  end;
  Frm.FSearchText := SearchText;
  Frm.Show;
end;

procedure TFrmFrostSearchResults.FormCreate(Sender: TObject);
begin
  FInfoList := TList.Create;
end;

procedure TFrmFrostSearchResults.FormDestroy(Sender: TObject);
begin
  while FInfoList.Count > 0 do begin Dispose(FInfoList.Items[0]); FInfoList.Delete(0); end;
  FInfoList.Free;
end;

procedure TFrmFrostSearchResults.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Action := caFree;
end;

procedure TFrmFrostSearchResults.FormShow(Sender: TObject);
begin
  if not FInited then Init;
end;

procedure TFrmFrostSearchResults.Init;
var
  i,iFile: Integer;
  item:    TListItem;
  sFile,s: String;
  pInfo:          PFrostFileInfo;
begin
  LvResults.Items.BeginUpdate;
  try
    for iFile := 0 to FInfoList.Count-1 do begin
      pInfo := FInfoList.Items[iFile];
      s := pInfo^.SearchResult.Filename;
      sFile := ExtractFileName(s);
      s := ExtractFilePath(s); Delete(s,Length(s),1);
      s := ExtractFilePath(s); Delete(s,Length(s),1);
      pInfo^.Boardname := ExtractFileName(s);
      pInfo^.MsgID := 0;
      for i := Length(sFile) downto 1 do begin
        if sFile[i] = '-' then begin
          s := Copy(sFile, i+1, Length(sFile));
          pInfo^.MsgID := StrToIntDef(Copy(s,1,Pos('.',s)-1),0);
          break;
        end;
      end;

      item := LvResults.Items.Add;
      item.Data := pInfo;
      item.Caption := pInfo^.Boardname;
      item.SubItems.Add( FormatDateTime('yyyyy"-"mm"-"dd hh":"nn', pInfo^.SearchResult.Header.DateTime) );
      item.SubItems.Add( IntToStr(pInfo^.MsgID) );
      item.SubItems.Add( pInfo^.SearchResult.Header.From );
      item.SubItems.Add( pInfo^.SearchResult.Header.Subject );
    end;
  finally
    LvResults.Items.EndUpdate;
    FInited := True;
  end;
  LvResults.CustomSort(nil,0);
end;

procedure TFrmFrostSearchResults.DisplayMessage;
var
  pInfo: PFrostFileInfo;
  sXML,s:  String;
  Msg:   TFrostMessage;
  i:     Integer;
begin
  REdPreview.Clear;
  FDisplayed := LvResults.ItemIndex; if FDisplayed < 0 then exit;
  pInfo := LvResults.Items[FDisplayed].Data; if pInfo = nil then exit;

  sXML := GetFrostXMLAsString( pInfo^.SearchResult.Filename );
  if GetFrostMessageFromXML(sXML, Msg) then begin
    s := '*************************************' + #13;
    s := s + 'From   : ' + Msg.Header.From + #13;
    s := s + 'Subject: ' + Msg.Header.Subject + #13;
    s := s + 'Date   : ' + DateTimeToStr(Msg.Header.DateTime) + #13;
    if Msg.FileAttachments <> '' then
      s := s + 'Attached files:'#13 + Msg.FileAttachments + #13;
    s := s + '*************************************' + #13#13;
    s := s + Msg.Body;
  end else begin
    s := '*************************************' + #13;
    s := s + 'Unable to decode message - raw XML displayed' + #13;
    s := s + '*************************************' + #13#13;
    s := s + sXML;
  end;

  REdPreview.Text := s;

  // Highlight search text
  i := 0;
  repeat
    i := REdPreview.FindText(FSearchText, i, Length(REdPreview.Text), []);
    if i >= 0 then begin
      REdPreview.SelStart := i; REdPreview.SelLength := Length(FSearchText);
      REdPreview.SelAttributes.Color := clBlue;
      REdPreview.SelAttributes.Style := [fsBold];
      i := i + Length(FSearchText);
    end;
  until i < 0;
end;

procedure TFrmFrostSearchResults.LvResultsColumnClick(Sender: TObject; Column: TListColumn);
begin
  if Column.Index = FSortBy then
    FSortDesc := not FSortDesc
  else begin
    FSortDesc := False;
    FSortBy := Column.Index;
  end;
  LvResults.CustomSort(nil,0);
end;

procedure TFrmFrostSearchResults.LvResultsCompare(Sender: TObject; Item1,Item2: TListItem; Data: Integer; var Compare: Integer);
var pInfo1,pInfo2: PFrostFileInfo;
begin
  pInfo1 := Item1.Data;
  pInfo2 := Item2.Data;
  Compare := 0;
  case FSortBy of
    0: Compare := AnsiCompareText(pInfo1^.Boardname, pInfo2^.Boardname);
    1: if pInfo1^.SearchResult.Header.DateTime < pInfo2^.SearchResult.Header.DateTime then
         Compare := -1
       else if pInfo1^.SearchResult.Header.DateTime > pInfo2^.SearchResult.Header.DateTime then
         Compare := 1;
    2: Compare := pInfo1^.MsgID - pInfo2^.MsgID;
    3: Compare := AnsiCompareText(pInfo1^.SearchResult.Header.From, pInfo2^.SearchResult.Header.From);
    4: Compare := AnsiCompareText(pInfo1^.SearchResult.Header.Subject, pInfo2^.SearchResult.Header.Subject);
  end;
  if (Compare = 0) and (FSortBy <> 1) then begin
    if pInfo1^.SearchResult.Header.DateTime < pInfo2^.SearchResult.Header.DateTime then
      Compare := -1
    else if pInfo1^.SearchResult.Header.DateTime > pInfo2^.SearchResult.Header.DateTime then
      Compare := 1;
  end;
  if FSortDesc then Compare := -Compare;
end;

procedure TFrmFrostSearchResults.LvResultsChange(Sender: TObject; Item: TListItem; Change: TItemChange);
begin
  if LvResults.ItemIndex <> FDisplayed then DisplayMessage;
end;

end.
