//PROFILE-NO
unit FreenetUtils;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

{
  currently only for CHK@ keys. others may need redirects!

  // additions to FCP specs:
  // all hex digits must be lower case!

}

interface

uses Windows, WinSock, SysUtils, Classes, FixedThread, CryptUtils;

const
  FCP_DEFAULT_ADDR   = INADDR_LOOPBACK;
  FCP_DEFAULT_PORT   = 8481;
  FCP_COMMAND_HEADER: packed Array [0..3] of Byte = (0,0,0,2);

  FORCE_CHECKSUM     = True;  // enable hack to get checksum into splitfile metadata
  PREFER_DATABLOCKS  = 0.50;  // probability to prefer a datablock over a checkblock
                              // 0.5: treat data/checkblocks equally

  FCP_CHUNKSIZE = 1024*1024;  // block size for transfers to/from FCP

  ENABLE_THREAD_LOGGING = True;

  USE_NEWSTYLE_PREPFILE = False;

{$DEFINE USE_NATIVE_SHA1}  // $define to use native SHA1 calculation (else: use FCP)

{$DEFINE STREAM_CHECKBLOCKS} // $define to stream checkblocks directly to file

{-DEFINE FCP_HARDKILLSOCKS} // $define to cut connections to FCP the hard way


{-DEFINE DEBUG_LOGFCPOUT}  // $define to log data sent FROM FCP
{-DEFINE DEBUG_LOGFCPIN}   // $define to log data sent  TO  FCP
{$DEFINE DEBUG_LOG}

{$IFDEF DEBUG_LOGFCPOUT}
  DEBUG_LOGFCPOUTFILE: String = 'FCPBinaryLog-OUT.dat';
{$ENDIF}
{$IFDEF DEBUG_LOGFCPIN}
  DEBUG_LOGFCPINFILE:  String = 'FCPBinaryLog-IN.dat';
{$ENDIF}
{$IFDEF DEBUG_LOG}
  DEBUG_LOGFILE:       String = 'debug.log';
{$ENDIF}

  LOGLVL_DEBUG     =  0;
  LOGLVL_NORMAL    = 10;
  LOGLVL_IMPORTANT = 20;
  LOGLVL_NONE      = 999;

var
  DEFAULT_FEC_DESCRIPTION: String = 'Onion FEC v1.2 file';
  USE_NATIVE_CHK:   Boolean = True;
  USE_NATIVE_FEC:   Boolean = True;
  USE_CACHED_FILES: Boolean = True;
  LOG_LEVEL:        Integer = LOGLVL_NORMAL;

type
  TUpDownloadStats = record
    Valid:     Boolean;
    TimeStart: TDateTime;
    TimeEnd:   TDateTime;
    BytesUp:   Int64;
    BytesDn:   Int64;
  end;
  PUpDownloadStats = ^TUpDownloadStats;

  TMimeResolveMode = (mrmFuqid, mrmFIW, mrmFixed, mrmFrost, mrmRaw, mrmFCP);
  TInsertStyle = record
    Header_Cipher:       String;
    Blocks_Cipher:       String;
    ChkSumAfterDescript: Boolean;
    MimeResolveMode:     TMimeResolveMode;
    FixedMimeType:       String; // only needed for isFixedMime
  end;

const
  InsertStyle_Fuqid: TInsertStyle = (
    Header_Cipher:       CIPHERNAME_TWOFISH;
    Blocks_Cipher:       CIPHERNAME_TWOFISH;
    ChkSumAfterDescript: True;
    MimeResolveMode:     mrmFuqid;
    FixedMimeType:       '';
  );
  InsertStyle_FIW:   TInsertStyle = (
    Header_Cipher:       CIPHERNAME_TWOFISH;
    Blocks_Cipher:       CIPHERNAME_TWOFISH;
    ChkSumAfterDescript: False;
    MimeResolveMode:     mrmFIW;
    FixedMimeType:       '';
  );
  InsertStyle_FProxy: TInsertStyle = (
    Header_Cipher:       CIPHERNAME_RIJNDAEL;
    Blocks_Cipher:       CIPHERNAME_TWOFISH;
    ChkSumAfterDescript: False;
    MimeResolveMode:     mrmFixed;
    FixedMimeType:       ''; // needs to be specified separately!!
  );
  InsertStyle_Frost: TInsertStyle = (
    Header_Cipher:       CIPHERNAME_TWOFISH;
    Blocks_Cipher:       CIPHERNAME_TWOFISH;
    ChkSumAfterDescript: False;
    MimeResolveMode:     mrmFrost;
    FixedMimeType:       '';
  );
  InsertStyle_Raw:   TInsertStyle = (
    Header_Cipher:       CIPHERNAME_TWOFISH; // modifyable
    Blocks_Cipher:       CIPHERNAME_TWOFISH; // (never used) single-file-inserts only, no FEC
    ChkSumAfterDescript: False;
    MimeResolveMode:     mrmRaw;
    FixedMimeType:       '';
  );
  InsertStyle_FCPCmd: TInsertStyle = ( // complete FCP command is in data; (00000002-header is optional) 
    Header_Cipher:       CIPHERNAME_TWOFISH; // (never used) specify cipher in FCP command file if needed
    Blocks_Cipher:       CIPHERNAME_TWOFISH; // (never used) single-file-inserts only, no FEC
    ChkSumAfterDescript: False;
    MimeResolveMode:     mrmFCP;
    FixedMimeType:       '';
  );


var
  gWriteFCPDebugLogOnErrors: Boolean = False;

type
  ENativeImplError = class(Exception);
  EFCPError = class(Exception);
  EFCPTimeout = class(EFCPError);
  TFCPSession = class
  private
    FAddr: TInAddr;
    FPort: Word;
    FSock: TSocket;
    FslMsg: TStringList;
    FRecvd: String;
    FDebug: Array [0..1] of record // 0 = send, 1 = recv
                              Use:     Boolean;
                              Buf:     PChar;
                              BufSize: Integer;
                              BufPos:  Integer;
                            end;
    FDebugInfo: TStringList;
    function GetMsgAsString: String;
    procedure KillSocket;
    procedure LogToDebugBuffer(IsSend: Boolean; pData: PChar; DataLen: Cardinal);
  public
    constructor Create(FCPAddr: u_long = FCP_DEFAULT_ADDR; FCPPort: Word = FCP_DEFAULT_PORT; UseDebugBuffer: Boolean = False);
    destructor Destroy; override;
    procedure ConnectToFCP;
    procedure SendData(pData: Pointer; DataLen: Cardinal; ReuseConnection: Boolean = False);
    procedure SendStringCommand(sCmd: String);
    procedure ReadData(pBuf: Pointer; BufSize: Integer; out BytesRead: Integer);
    function  ReadNextMessage(EndMarker: String = 'EndMessage'): String;
    function  ReadNextLine:    String;
    function  WaitForDataAvailable(TimeoutMillisec: Integer = -1; ConsiderBufferedData: Boolean = True): Boolean;
    procedure WriteDebugBuffer(SuggestedFileName: String = '');

    property Msg: TStringList read FslMsg;
    property MsgAsString: String read GetMsgAsString;
  end;


  TFECSegmentInfo = record
    FECAlgorithm: String;
    FileLength,
    Offset,
    DataBlockOffset,
    BlockCount,
    BlockSize,
    CheckBlockOffset,
    CheckBlockCount,
    CheckBlockSize,
    Segments,
    SegmentNum,
    BlocksRequired: Integer;
    FullSegmentHeader: String;
  end;

  TCheckAbortProc = procedure of object;

  TPartialSave = (psCompleteOnly, psSkipMissing, psZeroMissing, psCopyLastGood);

  // PreparedFileHeader:
  //   MagicCode: DWord; // either PREPAREDINSFILE(2)_MAGIC or PREPAREDGETFILE_MAGIC
  //   NumBlocks: DWord; // total number of blocks (data + check + meta)
  //   MainBlock: DWord; // number of main block (i.e. Splitfile metadata block)
  //   Offsets:   Array [0..NumBlocks-1] of DWord; // from start of file
  //   DoneMap:   Array [0..NumBlocks-1] of Byte; // Bit 7:working? Bit 6:done? Bits5..0: try#
  //   KeyMapLen: DWord; // length of keymap
  //   KeyMap:    Array [0..NumBlocks-1] of C-String;
  //   InfoLen:   DWord; // length of additional info following
  //     Filename:  C-String; // orig. filename
  //     FileLen:   DWord;    // orig. filelen
  //     Checksum:  C-String; // SHA1-checksum
  //     MimeType:  C-String; // MimeType
  //     <new info may go here>
  //   <end of Info section>
  //   SegHdLen:  DWord;    // size of segment header section (sizeof(NumSegHd)+sizeof(SegHd))
  //   NumSegHd:  DWord;    // number of seg.headers following
  //   SegHd:     Array [0..NumSegHd-1] of C-String;
  //   --- download files only ---
  //   BlockMapLen: DWord;    // size of block map section (sizeof(BlockMaps))
  //   BlockMaps: Array [0..NumSegHd-1] of C-String;
  //   --- end download files only ---
  //   <EOF>
  //
  //
  // PreparedFileData:
  //   at each block start:
  //     for type-1-files (FUsesOrigFile = False):
  //       total data length: DWord; // length of data + meta length
  //       meta length:       DWord; // length of metadata string
  //       metadata:          Array [0..meta length-1] of Char // NOT 0-terminated
  //       block data:        Array [0..block length-1] of Byte;
  //   note: block data is not necessarily in block order
  //     for type-2-files (FUsesOrigFile = True):
  //       BlockInOrigFile:   Byte; // 00.. block data here, 01.. in orig.file
  //       OrgFileOffset:     DWord;
  //       OrgFileBlockLen:   DWord; // can be smaller
  //       total data length: DWord; // length of data + meta length
  //       meta length:       DWord; // length of metadata string
  //       metadata:          Array [0..meta length-1] of Char // NOT 0-terminated
  //       block data:        Array [0..block length-1] of Byte; // only if data stored here
const
  PREPFILEDONEMASK_WORKING = $80;
  PREPFILEDONEMASK_DONE    = $40;
  PREPFILEDONEMASK_TRIES   = $3F;
type
  TPreparedFile = class
  private
    FBasename:  String;
    FNumBlocks: DWord;
    FMainBlock: DWord;
    FOffsetList:TList;
    FDoneList:  TList;
    FKeyList:   TStringList;
    FNumSegments: DWord;
    FSegHeaders: Array of String;
    FBlockMaps:  Array of String;
    FInfo:      record
                  OrigFilename: String;
                  FileLength:   DWord;
                  Checksum:     String;
                  MimeType:     String;
                end;
    FOutStream: TFileStream;
    FIsInsertFile: Boolean;
    FUsesOrgFile:  Boolean;
    FRemembered:   record
                     Valid:        Boolean;
                     DataFileSize: Int64;
                     OffsetList:   TList;
                   end;
    function  GetBlockOffset(BlockNum: DWord): DWord;
    procedure SetBlockOffset(BlockNum: DWord; const Value: DWord);
    function  GetKey(BlockNum: DWord): String;
    procedure SetKey(BlockNum: DWord; const Value: String);
    function GetDataFilename: String;
    function GetHeaderFilename: String;
    function GetAllDone: Boolean;
    function GetNumDone: DWord;
    function GetSegHeader(SegmentNum: DWord): String;
    procedure SetSegHeader(SegmentNum: DWord; const Value: String);
    function GetBlockDone(BlockNum: DWord): Boolean;
    function GetBlockRetry(BlockNum: DWord): Integer;
    function GetBlockWorking(BlockNum: DWord): Boolean;
    procedure SetBlockDone(BlockNum: DWord; const Value: Boolean);
    procedure SetBlockRetry(BlockNum: DWord; const Value: Integer);
    procedure SetBlockWorking(BlockNum: DWord; const Value: Boolean);
    function GetBlocksCompletedInSeg(SegmentNum: DWord; IncludeData,IncludeCheck: Boolean): Integer;
    function GetBlocksCompletedInSeg_All(SegmentNum: DWord): Integer;
    function GetBlocksCompletedInSeg_Check(SegmentNum: DWord): Integer;
    function GetBlocksCompletedInSeg_Data(SegmentNum: DWord): Integer;
    function GetBlocksRequiredInSeg(SegmentNum: DWord): Integer;
    function GetIsCheckBlock(BlockNum: DWord): Boolean;
    function GetIsDataBlock(BlockNum: DWord): Boolean;
    function GetCompletionRatioEx(var BlocksLeft: Integer; MaxRetries: Integer): Single;
    function GetCompletionRatio: Single;
    function GetNumWorking: DWord;
    function GetBlocksFailedInSeg(SegmentNum: DWord; MaxRetries: Integer): Integer;
    function GetRawDataOffset(BlockNum: DWord): DWord;
    function GetOutputStream: TFileStream;
    function GetLastDataBlock: DWord;
  public
    constructor Create(AsInsertFile: Boolean; ABasename: String = ''; UseOrigFile: Boolean = False);
    constructor CreateInsert(ABasename: String = ''; UseOrigFile: Boolean = False);
    constructor CreateDownload(ABasename: String = '');

    destructor  Destroy; override;
    procedure Init(ANumBlocks: Integer; ANumSegments: Integer = 0);
    procedure InitFromSplitFileMetadata(FCPAddr: u_long; FCPPort: Word; Metadata: String);
    procedure InitHeal(ANumBlocks: Integer);

    procedure ReadHeader;
    procedure WriteHeader(Debug_Delay: Integer = 0);

    procedure GetData(BlockNum: DWord; out pData: PChar; out DataLen: DWord; out sMetadata: String);
    function  WriteData(BlockNum: DWord; pData: PChar; DataLen: DWord; sMetadata: String; ReferenceOrgFile: Boolean = False; OrgFileOffset: DWord = 0; OrgFileLen: DWord = 0): Cardinal;
    function  GetBlockHeaderSize: DWord;
    procedure CloseDataFile;

    procedure RememberState;
    procedure RevertToRememberedState;

    procedure DeleteFiles;

    procedure SaveOutputFile(Outputfilename: String; CheckAbortProc: TCheckAbortProc = nil; PartialSave: TPartialSave = psCompleteOnly);

    function  CalcSplitFileMetadata(FCPAddr: u_long; FCPPort: Word; InsertStyle: TInsertStyle): String;
    function  GetNextDownloadBlock(MaxTries: Integer; out iBlock: DWord; EvenGetUnnecessaryBlock: Boolean = False): Boolean;
    procedure GetSegmentBlockInfo(SegmentNum: DWord; out FirstBlock,NumData,NumCheck: DWord);

    function  GetHealBlocks(Percent: Integer): String;
    function  GetBlockStatusString: String;

    function  GetProgressString(MaxRetries: Integer = 0): String;
    function  IsSuccessPossible(MaxRetries: Integer = 0): Boolean;

    function  FixLastDatablockPadding(BlockNum: DWord; pData: PChar; DataLen: DWord): Boolean;

    property  Basename:    String read FBasename write FBasename;
    property  HeaderFilename: String read GetHeaderFilename;
    property  DataFilename:   String read GetDataFilename;
    property  NumBlocks:   DWord  read FNumBlocks;
    property  MainBlock:   DWord  read FMainBlock write FMainBlock;
    property  NumSegments: DWord  read FNumSegments;
    property  BlockOffset  [BlockNum: DWord]: DWord   read GetBlockOffset  write SetBlockOffset;
    property  RawDataOffset[BlockNum: DWord]: DWord   read GetRawDataOffset;
    property  BlockDone    [BlockNum: DWord]: Boolean read GetBlockDone    write SetBlockDone;
    property  BlockWorking [BlockNum: DWord]: Boolean read GetBlockWorking write SetBlockWorking;
    property  BlockTries   [BlockNum: DWord]: Integer read GetBlockRetry   write SetBlockRetry;
    property  Key          [BlockNum: DWord]: String read GetKey         write SetKey;
    property  SegHeader[SegmentNum: DWord]: String read GetSegHeader   write SetSegHeader;
    property  AllDone: Boolean read GetAllDone;
    property  NumDone: DWord   read GetNumDone;
    property  NumWorking: DWord read GetNumWorking;
    property  OriginalFilename: String read FInfo.OrigFilename write FInfo.OrigFilename;
    property  FileLength:       DWord  read FInfo.FileLength   write FInfo.FileLength;
    property  SHA1Checksum:     String read FInfo.Checksum     write FInfo.Checksum;
    property  MimeType:         String read FInfo.MimeType     write FInfo.MimeType;
    property  IsInsertFile:     Boolean read FIsInsertFile;
    property  BlocksRequiredInSeg[SegmentNum: DWord]:  Integer read GetBlocksRequiredInSeg;
    property  BlocksCompletedInSeg[SegmentNum: DWord]: Integer read GetBlocksCompletedInSeg_All;
    property  BlocksFailedInSeg[SegmentNum: DWord; MaxRetries: Integer]: Integer read GetBlocksFailedInSeg;
    property  DataBlocksCompletedInSeg[SegmentNum: DWord]:  Integer read GetBlocksCompletedInSeg_Data;
    property  CheckBlocksCompletedInSeg[SegmentNum: DWord]: Integer read GetBlocksCompletedInSeg_Check;
    property  LastDataBlock: DWord read GetLastDataBlock;
    property  CompletionRatio:  Single read GetCompletionRatio;
    property  IsDataBlock [BlockNum: DWord]: Boolean read GetIsDataBlock;
    property  IsCheckBlock[BlockNum: DWord]: Boolean read GetIsCheckBlock;
    property  DataFileOutputStream: TFileStream read GetOutputStream;
  end;


  TTextMsgEvent = procedure (Sender: TObject; Msg: String; Level: Integer = LOGLVL_NORMAL) of object;

  EThreadAborted = class(Exception);

  TFreenetThreadFailReason = (ftfrUnknown,ftfrDNF,ftfrRNF,ftfrInvalid,ftfrCutoff,ftfrTimeout,ftfrBadData,ftfrAborted);

  TFreenetThread = class(TFixedThread)
  private
    FFCPAddr:    u_long;
    FFCPPort:    Word;
    FOnStatus:   TTextMsgEvent;
    FStatusMsg:    String;
    FStatusMsgID:  Cardinal;
    FStatusMsgLvl: Integer;
    FID:         String;
    FUserData:   Pointer;
    FUserID:     Int64;
    FDone:       Boolean;
    FUnnecessary: Boolean;
    FStats:       TUpDownloadStats;
    FFailReason:  TFreenetThreadFailReason;
  protected
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; OnStatus: TTextMsgEvent; AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
    procedure DoStatus(Msg: String; Level: Integer = LOGLVL_NORMAL);
    procedure DoStatusSynched;
    function  NewFCP(UseDebugBuffer: Boolean = False): TFCPSession;
    procedure CheckAbort;
  public
    property ID:       String  read FID;
    property UserData: Pointer read FUserData;
    property UserID:   Int64   read FUserID;
    property Done:         Boolean read FDone;
    property Unnecessary:  Boolean read FUnnecessary write FUnnecessary;
    property StatusMsgID:  Cardinal read FStatusMsgID;
    property Statistics:   TUpDownloadStats read FStats;
    property FailReason:   TFreenetThreadFailReason read FFailReason;

    property OnStatus: TTextMsgEvent read FOnStatus write FOnStatus;
  end;

  TInsertThread = class(TFreenetThread)
  private
    FInsertHTL:  Integer;
    FSkipLocal:  Boolean;
    FInsertData: PChar;
    FInsertLen:  Integer;
    FMetaData:   String;
    FTimeoutMS:  Integer;
    FResult:     Boolean;
    FResultKey:  String;
    FBlockNum:   Integer;
    FIsData:     Boolean;
    FIsCheck:    Boolean;
    FInsertStyle:   TInsertStyle;
    FIsHeaderBlock: Boolean;
    FFreeData:   Boolean;
    FRetries:    Integer; // -1: forever
    FCollided:   Boolean;
    FDebugFakeInsert: Boolean;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; InsertData: PChar; InsertLen: Integer; Metadata: String; InsertHTL: Integer; SkipLocal: Boolean; NumRetries,ABlockNum: Integer; IsDataBlock,IsCheckBlock,IsHeaderBlock,FreeDataOnExit: Boolean; AInsertStyle: TInsertStyle; OnStatus: TTextMsgEvent; TimeoutMinutes: Integer; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    property InsertResult: Boolean read FResult;
    property InsertKey:    String  read FResultKey;
    property KeyCollided:  Boolean read FCollided;
    property BlockNum:     Integer read FBlockNum;
    property Debug_FakeInsert: Boolean read FDebugFakeInsert write FDebugFakeInsert;
  end;

  TDownloadThread = class(TFreenetThread)
  private
    FDownloadKey:String;
    FDownloadHTL:Integer;
    FSkipLocal:  Boolean;
    FResult:     Boolean;
    FTimeoutMS:  Integer;
    // for reference
    FBlockNum:   Integer;
    FIsData:     Boolean;
    FIsCheck:    Boolean;
    // result data - freed when thread is destroyed:
    FResDataLen: Integer;
    FResData:    PChar;
    FResMetadata:String;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; DownloadKey: String; DownloadHTL: Integer; SkipLocal: Boolean; BlockNum: Integer; IsDataBlock,IsCheckBlock: Boolean; OnStatus: TTextMsgEvent; TimeoutMinutes: Integer; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    destructor Destroy; override;
    property DownloadResult: Boolean read FResult;
    property DownloadedMetadata: String  read FResMetadata;
    property DownloadedData:     PChar   read FResData;
    property DownloadedDataLen:  Integer read FResDataLen;
    property BlockNum:     Integer read FBlockNum;
    property IsCheckBlock: Boolean read FIsCheck;
    property IsDataBlock:  Boolean read FIsData;
  end;

  TCheckUpdateThread = class(TFreenetThread)
  private
    FResult:      Boolean;
    FFound:       Boolean;
    FDownloadKey: String;
    FDownloadHTL: Integer;
    FSkipLocal:   Boolean;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; AUpdateKey: String; DownloadHTL: Integer; SkipLocal: Boolean; OnStatus: TTextMsgEvent; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    property UpdateFound: Boolean read FFound;
    property UpdateKey: String read FDownloadKey;
    property ThreadResult: Boolean read FResult;
  end;


  TFECDecodeThread = class(TFreenetThread)
  private
    FPrepBasename:  String;
    FGenHealPerc:   Integer;
    FResult:        Boolean;
    FHealBlocks:    String;
    FSaveAs:        String;
    FPartialSave:   TPartialSave;
    FPartialOnly:   Boolean;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; PrepFileBasename,SaveOutputAs: String; GenHealblocksPercent: Integer; PartialSave: TPartialSave; OnStatus: TTextMsgEvent; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    property DecodeResult: Boolean read FResult;
    property PartiallyDecodedOnly: Boolean read FPartialOnly;
    property HealBlocks: String read FHealBlocks;
  end;

  TChecksumThread = class(TFreenetThread)
  private
    FFilename:      String;
    FChecksum:      String;
    FResult:        Boolean;
    FDontCalc:      Boolean;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; Filename: String; OnStatus: TTextMsgEvent; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    property ThreadResult: Boolean read FResult;
    property Checksum: String read FChecksum;
    property DontCalcChecksum: Boolean read FDontCalc write FDontCalc;
  end;

  TPrepareThread = class(TFreenetThread)
  private
    FFilename:      String;
    FPrepBasename:  String;
    FDoChecksum:    Boolean;
    FPrecalcMeta:   Boolean;
    FUseOrgFile:    Boolean;
    FInsertStyle:   TInsertStyle;
    FResult:        Boolean;
    FResultKey:     String;
  protected
    procedure Execute; override;
  public
    constructor Create(ThreadID: String; FCPAddr: u_long; FCPPort: Word; Filename,APrepBasename: String; MakeChecksum, PrecalcMeta, UseOrgFile: Boolean; AInsertStyle: TInsertStyle; OnStatus: TTextMsgEvent; AUserData: Pointer = nil; AUserID: Int64 = 0; CreateSuspended: Boolean = False);
    property ThreadResult: Boolean read FResult;
    property PrecalcedKey: String  read FResultKey;
    property PrepBasename: String read FPrepBasename;
  end;


  TFreenetFileNumber = record
    LongHash:  Int64;
    ShortHash: Cardinal;
  end;
  EFreenetURI = class (Exception);
  TFreenetURI = class // based on Freenet's FreenetURI.java
  private
    FKeyType:       String;
    FRoutingKeyB64:  String;
    FCryptoKeyB64:   String;
    FMetaInfo:    TStringList;
    FDocName:     String;
    FMetaStrings: TStringList;
    FDontURLEncode: Boolean;
    procedure ClearAll;
    function  URLDecode(const S: String): String;
    function  URLEncode(const S: String): String;
    function  OptionalURLEncode(const S: String): String;
    function  GetMetaInfoAsString: String;
    function  GetMetaStringsAsString: String;
    procedure SetDocName(const Value: String);
    function  GetFileNumber: TFreenetFileNumber;
    function  GetDSFile: String;
  public
    constructor Create;
    destructor Destroy; override;

    function  GetURI(WithPrefix: Boolean = True; WithMeta: Boolean = True; WithDocName: Boolean = True): String;
    procedure SetURI(const Value: String);

    procedure Debug;

    property  DontURLEncode: Boolean read FDontURLEncode write FDontURLEncode;
    property  KeyType:       String read FKeyType;
    property  RoutingKeyB64: String read FRoutingKeyB64;
    property  CryptoKeyB64:  String read FCryptoKeyB64;
    property  DocumentName:  String read FDocName write SetDocName;
    property  MetaInfo:      String read GetMetaInfoAsString;
    property  MetaString:    String read GetMetaStringsAsString;

    property  FileNumber: TFreenetFileNumber read GetFileNumber;
    property  DSFile:        String read GetDSFile;
  end;

function  MakeThreadID(S1, S2: String): String;
function  Hex(val: Integer): String;
procedure FCPMsgToStringList(FCPMsg: String; StrList: TStringList);
function  StringListToFCPMsg(StrList: TStringList; AppendEndMarker: String = 'EndMessage'): String;
procedure SplitKeySubkey(URI: String; out Key,Subkey: String);
function  GetDateRedirKey(TargetURI: String; Increment: Cardinal = $15180; Offset: Cardinal = 0; ForDateTimeUTC: TDateTime = 0): String;
function  InsertStyleToStr(const Style: TInsertStyle): String;
function  StrToInsertStyle(const S: String): TInsertStyle;

// accessible for benchmarking
function  Util_GenerateCHK_Native(pData: PChar; DataLen: Integer; sMetadata: String; Ciphername: String): String;

// accessible for validation
function  Util_NormKey(const sKey: String): String;
function  Util_GenerateCHK(pData: PChar; DataLen: Integer; sMetadata: String; fcp: TFCPSession = nil; Ciphername: String = CIPHERNAME_TWOFISH): String;
procedure Util_GetFECSegmentInfo(SegmentHeaderStrList: TStringList; out SegInfo: TFECSegmentInfo); overload;
procedure Util_GetFECSegmentInfo(SegmentHeader: String; out SegInfo: TFECSegmentInfo); overload;


implementation

uses Dialogs,Forms,Controls,SyncObjs,Math,Settings,SHA1,Base64,FECUtils,
  Misc;

var
  FreenetThreadCritSec: TCriticalSection = nil;

const
  FEC_LIMIT   = 1024*1024; // Minimum: 262144 !
  META_HEADER = 'Version'#10
              + 'Revision=1'#10
              + 'EndPart'#10;

{ Utils }

const
  MimeTypes: Array [0..19] of record Ext,Mime: String; end = (
               (Ext: 'txt';   Mime: 'text/plain';),
               (Ext: 'nfo';   Mime: 'text/plain';),
               (Ext: 'html';  Mime: 'text/html';),
               (Ext: 'htm';   Mime: 'text/html';),
               (Ext: 'mp3';   Mime: 'audio/mpeg';),
               (Ext: 'ogg';   Mime: 'audio/ogg';),
               (Ext: 'mid';   Mime: 'audio/midi';),
               (Ext: 'jpg';   Mime: 'image/jpeg';),
               (Ext: 'jpeg';  Mime: 'image/jpeg';),
               (Ext: 'gif';   Mime: 'image/gif';),
               (Ext: 'png';   Mime: 'image/png';),
               (Ext: 'avi';   Mime: 'video/avi';),
               (Ext: 'asf';   Mime: 'video/asf';),
               (Ext: 'mpg';   Mime: 'video/mpeg';),
               (Ext: 'mpeg';  Mime: 'video/mpeg';),
               (Ext: 'sid';   Mime: 'audio/psid';),
               (Ext: 'zip';   Mime: 'binary/zip-compressed';),
               (Ext: 'iso';   Mime: 'binary/cdimage';),
               (Ext: 'gz';    Mime: 'binary/gzip-compressed';),
               (Ext: 'rar';   Mime: 'application/x-rar-compressed';)
             );

  FIWMimeTypes: Array [0..171] of record Ext,Mime: String; end = (
               (Ext: 'csm';   Mime: 'application/cu-seeme';),
               (Ext: 'cu';    Mime: 'application/cu-seeme';),
               (Ext: 'tsp';   Mime: 'application/dsptype';),
               (Ext: 'xls';   Mime: 'application/excel';),
               (Ext: 'spl';   Mime: 'application/futuresplash';),
               (Ext: 'hqx';   Mime: 'application/mac-binhex40';),
               (Ext: 'doc';   Mime: 'application/msword';),
               (Ext: 'dot';   Mime: 'application/msword';),
               (Ext: 'bin';   Mime: 'application/octet-stream';),
               (Ext: 'oda';   Mime: 'application/oda';),
               (Ext: 'pdf';   Mime: 'application/pdf';),
               (Ext: 'asc';   Mime: 'application/pgp-keys';),
               (Ext: 'pgp';   Mime: 'application/pgp-signature';),
               (Ext: 'ps';    Mime: 'application/postscript';),
               (Ext: 'ai';    Mime: 'application/postscript';),
               (Ext: 'eps';   Mime: 'application/postscript';),
               (Ext: 'ppt';   Mime: 'application/powerpoint';),
               (Ext: 'rtf';   Mime: 'application/rtf';),
               (Ext: 'wp5';   Mime: 'application/wordperfect5.1';),
               (Ext: 'zip';   Mime: 'application/zip';),
               (Ext: 'wk';    Mime: 'application/x-123';),
               (Ext: 'bcpio'; Mime: 'application/x-bcpio';),
               (Ext: 'pgn';   Mime: 'application/x-chess-pgn';),
               (Ext: 'cpio';  Mime: 'application/x-cpio';),
               (Ext: 'deb';   Mime: 'application/x-debian-package';),
               (Ext: 'dcr';   Mime: 'application/x-director';),
               (Ext: 'dir';   Mime: 'application/x-director';),
               (Ext: 'dxr';   Mime: 'application/x-director';),
               (Ext: 'dvi';   Mime: 'application/x-dvi';),
               (Ext: 'pfa';   Mime: 'application/x-font';),
               (Ext: 'pfb';   Mime: 'application/x-font';),
               (Ext: 'gsf';   Mime: 'application/x-font';),
               (Ext: 'pcf';   Mime: 'application/x-font';),
               (Ext: 'pcf.Z'; Mime: 'application/x-font';),
               (Ext: 'gtar';  Mime: 'application/x-gtar';),
               (Ext: 'tgz';   Mime: 'application/x-gtar';),
               (Ext: 'hdf';   Mime: 'application/x-hdf';),
               (Ext: 'phtml'; Mime: 'application/x-httpd-php';),
               (Ext: 'pht';   Mime: 'application/x-httpd-php';),
               (Ext: 'php';   Mime: 'application/x-httpd-php';),
               (Ext: 'php3';  Mime: 'application/x-httpd-php3';),
               (Ext: 'phps';  Mime: 'application/x-httpd-php3-source';),
               (Ext: 'php3p'; Mime: 'application/x-httpd-php3-preprocessed';),
               (Ext: 'class'; Mime: 'application/x-java';),
               (Ext: 'latex'; Mime: 'application/x-latex';),
               (Ext: 'frm';   Mime: 'application/x-maker';),
               (Ext: 'maker'; Mime: 'application/x-maker';),
               (Ext: 'frame'; Mime: 'application/x-maker';),
               (Ext: 'fm';    Mime: 'application/x-maker';),
               (Ext: 'fb';    Mime: 'application/x-maker';),
               (Ext: 'book';  Mime: 'application/x-maker';),
               (Ext: 'fbdoc'; Mime: 'application/x-maker';),
               (Ext: 'mif';   Mime: 'application/x-mif';),
               (Ext: 'com';   Mime: 'application/x-msdos-program';),
               (Ext: 'exe';   Mime: 'application/x-msdos-program';),
               (Ext: 'bat';   Mime: 'application/x-msdos-program';),
               (Ext: 'dll';   Mime: 'application/x-msdos-program';),
               (Ext: 'nc';    Mime: 'application/x-netcdf';),
               (Ext: 'cdf';   Mime: 'application/x-netcdf';),
               (Ext: 'pac';   Mime: 'application/x-ns-proxy-autoconfig';),
               (Ext: 'o';     Mime: 'application/x-object';),
               (Ext: 'pl';    Mime: 'application/x-perl';),
               (Ext: 'pm';    Mime: 'application/x-perl';),
               (Ext: 'shar';  Mime: 'application/x-shar';),
               (Ext: 'swf';   Mime: 'application/x-shockwave-flash';),
               (Ext: 'swfl';  Mime: 'application/x-shockwave-flash';),
               (Ext: 'sit';   Mime: 'application/x-stuffit';),
               (Ext: 'sv4cpio'; Mime: 'application/x-sv4cpio';),
               (Ext: 'sv4crc';  Mime: 'application/x-sv4crc';),
               (Ext: 'tar';   Mime: 'application/x-tar';),
               (Ext: 'gf';    Mime: 'application/x-tex-gf';),
               (Ext: 'pk';    Mime: 'application/x-tex-pk';),
               (Ext: 'PK';    Mime: 'application/x-tex-pk';),
               (Ext: 'texinfo'; Mime: 'application/x-texinfo';),
               (Ext: 'texi';  Mime: 'application/x-texinfo';),
               (Ext: '~';     Mime: 'application/x-trash';),
               (Ext: '%';     Mime: 'application/x-trash';),
               (Ext: 'bak';   Mime: 'application/x-trash';),
               (Ext: 'old';   Mime: 'application/x-trash';),
               (Ext: 'sik';   Mime: 'application/x-trash';),
               (Ext: 't';     Mime: 'application/x-troff';),
               (Ext: 'tr';    Mime: 'application/x-troff';),
               (Ext: 'roff';  Mime: 'application/x-troff';),
               (Ext: 'man';   Mime: 'application/x-troff-man';),
               (Ext: 'me';    Mime: 'application/x-troff-me';),
               (Ext: 'ms';    Mime: 'application/x-troff-ms';),
               (Ext: 'ustar'; Mime: 'application/x-ustar';),
               (Ext: 'src';   Mime: 'application/x-wais-source';),
               (Ext: 'wz';    Mime: 'application/x-wingz';),
               (Ext: 'au';    Mime: 'audio/basic';),
               (Ext: 'snd';   Mime: 'audio/basic';),
               (Ext: 'mid';   Mime: 'audio/midi';),
               (Ext: 'midi';  Mime: 'audio/midi';),
               (Ext: 'mpga';  Mime: 'audio/mpeg';),
               (Ext: 'mpega'; Mime: 'audio/mpeg';),
               (Ext: 'mp2';   Mime: 'audio/mpeg';),
               (Ext: 'mp3';   Mime: 'audio/mpeg';),
               (Ext: 'm3u';   Mime: 'audio/mpegurl';),
               (Ext: 'aif';   Mime: 'audio/x-aiff';),
               (Ext: 'aiff';  Mime: 'audio/x-aiff';),
               (Ext: 'aifc';  Mime: 'audio/x-aiff';),
               (Ext: 'gsm';   Mime: 'audio/x-gsm';),
               (Ext: 'ra';    Mime: 'audio/x-pn-realaudio';),
               (Ext: 'rm';    Mime: 'audio/x-pn-realaudio';),
               (Ext: 'ram';   Mime: 'audio/x-pn-realaudio';),
               (Ext: 'rpm';   Mime: 'audio/x-pn-realaudio-plugin';),
               (Ext: 'wav';   Mime: 'audio/x-wav';),
               (Ext: 'gif';   Mime: 'image/gif';),
               (Ext: 'ief';   Mime: 'image/ief';),
               (Ext: 'jpeg';  Mime: 'image/jpeg';),
               (Ext: 'jpg';   Mime: 'image/jpeg';),
               (Ext: 'jpe';   Mime: 'image/jpeg';),
               (Ext: 'png';   Mime: 'image/png';),
               (Ext: 'tiff';  Mime: 'image/tiff';),
               (Ext: 'tif';   Mime: 'image/tiff';),
               (Ext: 'ras';   Mime: 'image/x-cmu-raster';),
               (Ext: 'bmp';   Mime: 'image/x-ms-bmp';),
               (Ext: 'pnm';   Mime: 'image/x-portable-anymap';),
               (Ext: 'pbm';   Mime: 'image/x-portable-bitmap';),
               (Ext: 'pgm';   Mime: 'image/x-portable-graymap';),
               (Ext: 'ppm';   Mime: 'image/x-portable-pixmap';),
               (Ext: 'rgb';   Mime: 'image/x-rgb';),
               (Ext: 'xbm';   Mime: 'image/x-xbitmap';),
               (Ext: 'xpm';   Mime: 'image/x-xpixmap';),
               (Ext: 'xwd';   Mime: 'image/x-xwindowdump';),
               (Ext: 'csv';   Mime: 'text/comma-separated-values';),
               (Ext: 'html';  Mime: 'text/html';),
               (Ext: 'htm';   Mime: 'text/html';),
               (Ext: 'mml';   Mime: 'text/mathml';),
               (Ext: 'txt';   Mime: 'text/plain';),
               (Ext: 'rtx';   Mime: 'text/richtext';),
               (Ext: 'tsv';   Mime: 'text/tab-separated-values';),
               (Ext: 'h++';   Mime: 'text/x-c++hdr';),
               (Ext: 'hpp';   Mime: 'text/x-c++hdr';),
               (Ext: 'hxx';   Mime: 'text/x-c++hdr';),
               (Ext: 'hh';    Mime: 'text/x-c++hdr';),
               (Ext: 'c++';   Mime: 'text/x-c++src';),
               (Ext: 'cpp';   Mime: 'text/x-c++src';),
               (Ext: 'cxx';   Mime: 'text/x-c++src';),
               (Ext: 'cc';    Mime: 'text/x-c++src';),
               (Ext: 'h';     Mime: 'text/x-chdr';),
               (Ext: 'csh';   Mime: 'text/x-csh';),
               (Ext: 'c';     Mime: 'text/x-csrc';),
               (Ext: 'java';  Mime: 'text/x-java';),
               (Ext: 'moc';   Mime: 'text/x-moc';),
               (Ext: 'p';     Mime: 'text/x-pascal';),
               (Ext: 'pas';   Mime: 'text/x-pascal';),
               (Ext: 'etx';   Mime: 'text/x-setext';),
               (Ext: 'sh';    Mime: 'text/x-sh';),
               (Ext: 'tcl';   Mime: 'text/x-tcl';),
               (Ext: 'tk';    Mime: 'text/x-tcl';),
               (Ext: 'tex';   Mime: 'text/x-tex';),
               (Ext: 'ltx';   Mime: 'text/x-tex';),
               (Ext: 'sty';   Mime: 'text/x-tex';),
               (Ext: 'cls';   Mime: 'text/x-tex';),
               (Ext: 'vcs';   Mime: 'text/x-vCalendar';),
               (Ext: 'vcf';   Mime: 'text/x-vCard';),
               (Ext: 'dl';    Mime: 'video/dl';),
               (Ext: 'fli';   Mime: 'video/fli';),
               (Ext: 'gl';    Mime: 'video/gl';),
               (Ext: 'mpeg';  Mime: 'video/mpeg';),
               (Ext: 'mpg';   Mime: 'video/mpeg';),
               (Ext: 'mpe';   Mime: 'video/mpeg';),
               (Ext: 'qt';    Mime: 'video/quicktime';),
               (Ext: 'mov';   Mime: 'video/quicktime';),
               (Ext: 'asf';   Mime: 'video/x-ms-asf';),
               (Ext: 'asx';   Mime: 'video/x-ms-asf';),
               (Ext: 'avi';   Mime: 'video/x-msvideo';),
               (Ext: 'movie'; Mime: 'video/x-sgi-movie';),
               (Ext: 'vrm';   Mime: 'x-world/x-vrml';),
               (Ext: 'vrml';  Mime: 'x-world/x-vrml';),
               (Ext: 'wrl';   Mime: 'x-world/x-vrml';)
             );


function GetMimeType(Ext: String; FInsertStyle: TInsertStyle): String;
var i: Integer;
begin
  if FInsertStyle.MimeResolveMode = mrmFixed then begin
    Result := FInsertStyle.FixedMimeType; exit;
  end else if FInsertStyle.MimeResolveMode = mrmFIW then begin
    while Copy(Ext,1,1) = '.' do Delete(Ext,1,1);
    for i := Low(FIWMimeTypes) to High(FIWMimeTypes) do
      if CompareText(Ext, FIWMimeTypes[i].Ext) = 0 then begin
        Result := FIWMimeTypes[i].Mime; exit;
      end;
  end else if FInsertStyle.MimeResolveMode = mrmFrost then begin
    Result := 'Frost/FEC'; exit;
  end else begin
    while Copy(Ext,1,1) = '.' do Delete(Ext,1,1);
    for i := Low(MimeTypes) to High(MimeTypes) do
      if CompareText(Ext, MimeTypes[i].Ext) = 0 then begin
        Result := MimeTypes[i].Mime; exit;
      end;
  end;
  Result := 'application/octet-stream';
end;

function  InsertStyleToStr(const Style: TInsertStyle): String;
begin
  Result := '';
  if      Style.Header_Cipher = CIPHERNAME_RIJNDAEL then Result := Result + 'R'
  else if Style.Header_Cipher = CIPHERNAME_TWOFISH  then Result := Result + 'T'
  else raise EConvertError.Create('Unknown header cipher');
  if      Style.Blocks_Cipher = CIPHERNAME_RIJNDAEL then Result := Result + 'R'
  else if Style.Blocks_Cipher = CIPHERNAME_TWOFISH  then Result := Result + 'T'
  else raise EConvertError.Create('Unknown blocks cipher');
  Result := Result + IntToStr(Ord(Style.ChkSumAfterDescript));
  Result := Result + IntToStr(Ord(Style.MimeResolveMode));
  if Style.MimeResolveMode = mrmFixed then Result := Result + '<' + Style.FixedMimeType + '>';
end;

function  StrToInsertStyle(const S: String): TInsertStyle;
var
  i:    Integer;
  sTmp: String;
begin
  if Length(S) < 4 then raise EConvertError.Create('Invalid insert style string');
  if      S[1] = 'R' then Result.Header_Cipher := CIPHERNAME_RIJNDAEL
  else if S[1] = 'T' then Result.Header_Cipher := CIPHERNAME_TWOFISH
  else raise EConvertError.Create('Unknown header cipher');
  if      S[2] = 'R' then Result.Blocks_Cipher := CIPHERNAME_RIJNDAEL
  else if S[2] = 'T' then Result.Blocks_Cipher := CIPHERNAME_TWOFISH
  else raise EConvertError.Create('Unknown blocks cipher');
  Result.ChkSumAfterDescript := Boolean(StrToInt(S[3]));
  Result.MimeResolveMode     := TMimeResolveMode(StrToInt(S[4]));
  if Result.MimeResolveMode = mrmFixed then begin
    if S[5] <> '<' then raise EConvertError.Create('Invalid insert style string (mime type missing)');
    sTmp := Copy(S,6,Length(S));
    i := Pos('>',sTmp);
    if i = 0 then raise EConvertError.Create('Invalid insert style string (mime type unclosed)');
    Result.FixedMimeType := Copy(sTmp,1,i-1);
  end;
end;

function MakeThreadID(S1, S2: String): String;
const
  MAXLEN1 = 20;
  MAXLEN2 = 7;
var i,i2: Integer;
begin
  if Length(S1) > MAXLEN1 then begin
    i := MAXLEN1 div 2 - 1; i2 := MAXLEN1 - i - 3;
    S1 := Copy(S1, 1, i) + '...' + Copy(S1, Length(S1) + 1 - i2, i2);
  end;
  if Length(S2) > MAXLEN2 then S2 := Copy(S2, 1, MAXLEN2);
  Result := Format('%-*s %-*s',[MAXLEN1,S1,MAXLEN2,S2]);
end;

type
  TProfileInfo = record
    StartTicks:  DWord;
    EndTicks:    DWord;
    TotalTicks:  DWord;
    Descript:    String;
  end;

procedure InitStats(var Stats: TUpDownloadStats);
begin
  Stats.Valid     := False;
  Stats.TimeStart := Now;
  Stats.BytesUp   := 0;
  Stats.BytesDn   := 0;
end;

procedure FinishStats(var Stats: TUpDownloadStats);
begin
  Stats.TimeEnd := Now;
  Stats.Valid   := (Stats.TimeEnd >= Stats.TimeStart);
end;

procedure Profile_Start(var P: TProfileInfo; Descript: String);
begin
  P.StartTicks := GetTickCount;
  P.EndTicks   := 0;
  P.TotalTicks := 0;
  P.Descript   := Descript;
end;

procedure Profile_End(var P: TProfileInfo);
var tmp: Int64;
begin
  P.EndTicks := GetTickCount;
  tmp := P.EndTicks - P.StartTicks;
  if tmp < 0 then inc(tmp, $100000000);
  P.TotalTicks := tmp;
end;

function Profile_Info(var P: TProfileInfo): String;
begin
  Result := Format('PROFILE:%s -> %d',[P.Descript,P.TotalTicks]);
end;

{$IFDEF DEBUG_LOG}
procedure DebugLog(Msg: String);
var F: TextFile;
begin
  AssignFile(F, DEBUG_LOGFILE);
  try
    if FileExists(DEBUG_LOGFILE) then Append(F) else Rewrite(F);
    try
      WriteLn(F, Msg);
    finally
      CloseFile(F);
    end;
  except
  end;
end;
{$ENDIF}

function Hex(val: Integer): String;
// return *lower* case hex string
begin
  Result := LowerCase(Format('%x',[val]));
end;

procedure FCPMsgToStringList(FCPMsg: String; StrList: TStringList);
// convert a (text-only!) FCP message to a string list
var
  i: Integer;
begin
  StrList.Clear;
  while Pos(#13#10, FCPMsg) <> 0 do Delete(FCPMsg, Pos(#13#10, FCPMsg), 1);
  while Pos(#13, FCPMsg) <> 0 do FCPMsg[Pos(#13, FCPMsg)] := #10;
  repeat
    i := Pos(#10,FCPMsg);
    if i <> 0 then begin
      StrList.Add(Copy(FCPMsg,1,i-1)); Delete(FCPMsg,1,i);
    end else if FCPMsg <> '' then begin
      StrList.Add(FCPMsg); FCPMsg := '';
    end;
  until i = 0;
end;

function  StringListToFCPMsg(StrList: TStringList; AppendEndMarker: String = 'EndMessage'): String;
var i: Integer;
begin
  Result := '';
  for i := 0 to StrList.Count-1 do
    Result := Result + StrList.Strings[i] + #10;
  if AppendEndMarker <> '' then
    Result := Result + AppendEndMarker + #10;
end;

procedure SplitKeySubkey(URI: String; out Key,Subkey: String);
var i: Integer;
begin
  Key := URI; Subkey := ''; i := Pos('//',Key);
  if i <> 0 then begin
    Subkey := Copy(Key,i+2,Length(Key)); Delete(Key,i,Length(Key));
  end;
end;

function GetDateRedirKey(TargetURI: String; Increment,Offset: Cardinal; ForDateTimeUTC: TDateTime): String;
var
  NowUTC,KeyUTC: Cardinal;
  systim: TSystemTime;
  filtim: TFileTime;
  dt:     TDateTime;
  i64:    Int64;
  i:      Integer;
  s1,s2,s3: String;
begin
  if Increment = 0 then raise Exception.Create('Invalid time increment in date redirect');
  GetSystemTime(systim); // this is UTC
  if not SystemTimeToFileTime(systim, filtim) then raise Exception.Create('SystemTimeToFileTime failed');
  // convert filetime (100-nanosecond intervals since January 1, 1601)
  // to unix time (seconds since January 1, 1970)
  i64 := (Int64(filtim.dwHighDateTime) shl 32) or filtim.dwLowDateTime;
  dec(i64, 116444736000000000);
  NowUTC := i64 div 10000000;
  if ForDateTimeUTC <> 0 then begin
    dt := SystemTimeToDateTime(systim) - ForDateTimeUTC;
    dec(NowUTC, Round(dt*24*60*60));
  end;
  // now work out the actual UTC for the Key
  if NowUTC <= Offset then
    KeyUTC := Offset
  else
    KeyUTC := ((NowUTC - Offset) div Increment) * Increment + Offset;

  s1 := ''; s2 := TargetURI; s3 := '';
  i := Pos('//',s2); if i <> 0 then begin s3 := Copy(s2,i,Length(s2)); Delete(s2,i,Length(s2)); end;
  repeat
    i := Pos('/',s2);
    if i <> 0 then begin s1 := s1 + Copy(s2,1,i); Delete(s2,1,i); end;
  until i = 0;
  if (s1 = '') or (s2 = '') then raise Exception.Create('Invalid target URI in date redirect');

  Result := s1 + Hex(KeyUTC) + '-' + s2 + s3;
end;

function Util_FECMakeMetadata(Headers,Maps: Array of String; Checksum,MimeType: String; InsertStyle: TInsertStyle; fcp: TFCPSession = nil): String;
var
  bCreatedFCP: Boolean;
  sCmd,sMsg,sData: String;
  i,iBlock,iSeg:   Integer;
  DataLen,ChunkLen,BytesLeft,nbytes: Integer;
  pBuf,p: pChar;
  slMeta,slTmp: TStringList;
  sDesc:  String;
  SegInfo: TFECSegmentInfo;
begin
  // Frost doesn't use FECMakeMetadata at all, but does it itself
  if InsertStyle.MimeResolveMode = mrmFrost then begin
    slMeta := nil; slTmp := nil;
    try
      slMeta := TStringList.Create;
      slTmp  := TStringList.Create;
      Util_GetFECSegmentInfo(Headers[0], SegInfo);
      slMeta.Add('Document');
      slMeta.Add('SplitFile.AlgoName=OnionFEC_a_1_2');
      slMeta.Add('SplitFile.Size=' + Hex(SegInfo.FileLength));
      iBlock := 0;
      for iSeg := 0 to SegInfo.Segments-1 do begin
        Util_GetFECSegmentInfo(Headers[iSeg], SegInfo);
        inc(iBlock, SegInfo.BlockCount);
      end;
      slMeta.Add('SplitFile.BlockCount=' + Hex(iBlock));
      iBlock := 0;
      for iSeg := 0 to SegInfo.Segments-1 do begin
        Util_GetFECSegmentInfo(Headers[iSeg], SegInfo);
        FCPMsgToStringList(Maps[iSeg], slTmp);
        for i := 0 to SegInfo.BlockCount-1 do begin
          slMeta.Add('SplitFile.Block.' + Hex(iBlock+1) + '=' + slTmp.Values['Block.' + Hex(i)]);
          inc(iBlock);
        end;
      end;
      iBlock := 0;
      for iSeg := 0 to SegInfo.Segments-1 do begin
        Util_GetFECSegmentInfo(Headers[iSeg], SegInfo);
        inc(iBlock, SegInfo.CheckBlockCount);
      end;
      slMeta.Add('SplitFile.CheckBlockCount=' + Hex(iBlock));
      iBlock := 0;
      for iSeg := 0 to SegInfo.Segments-1 do begin
        Util_GetFECSegmentInfo(Headers[iSeg], SegInfo);
        FCPMsgToStringList(Maps[iSeg], slTmp);
        for i := 0 to SegInfo.CheckBlockCount-1 do begin
          slMeta.Add('SplitFile.CheckBlock.' + Hex(iBlock+1) + '=' + slTmp.Values['Check.' + Hex(i)]);
          inc(iBlock);
        end;
      end;
      slMeta.Add('End');
      Result := META_HEADER + StringListToFCPMsg(slMeta,'');
    finally
      slMeta.Free;
      slTmp.Free;
    end;
    exit;
  end;

  bCreatedFCP := (fcp = nil);
  pBuf := nil;
  try
    if bCreatedFCP then fcp := TFCPSession.Create;

    sData := '';
    for i := 0 to High(Headers) do sData := sData + Headers[i] + Maps[i];

    case InsertStyle.MimeResolveMode of
      mrmFuqid: sDesc := DEFAULT_FEC_DESCRIPTION;
      mrmFrost: begin sDesc := 'FEC file inserted by FROST'; MimeType := 'Frost/FEC'; Checksum := ''; end;
      else      sDesc := 'file';
    end;
    sCmd := 'FECMakeMetadata'#10
          + 'Description='+sDesc+#10;
    if Checksum <> '' then sCmd := sCmd + 'CheckSum=' + Checksum + #10;
    sCmd := sCmd
          + 'MimeType=' + MimeType + #10
          + 'DataLength=' + Hex(Length(sData)) + #10
          + 'Data'#10
          + sData;
    fcp.SendStringCommand(sCmd);

    sMsg := fcp.ReadNextMessage;
    if sMsg <> 'MadeMetadata' then raise Exception.Create('Unexpected message (MadeMetadata expected):' + fcp.MsgAsString);
    DataLen := StrToInt('$' + fcp.Msg.Values['DataLength']);

    GetMem(pBuf, DataLen + 1); // + 1 so we can append a #0
    BytesLeft := DataLen; p := pBuf;
    while BytesLeft > 0 do begin
      if fcp.ReadNextMessage('Data') <> 'DataChunk' then raise Exception.Create('Unexpected message (DataChunk expected):' + fcp.MsgAsString);
      ChunkLen := StrToInt('$' + fcp.Msg.Values['Length']);
      Assert(ChunkLen <= BytesLeft);
      while ChunkLen > 0 do begin
        // fcp.ReadData(p, BytesLeft, nbytes);
        fcp.ReadData(p, ChunkLen, nbytes);
        if nbytes = 0 then raise Exception.Create('Connection died');
        inc(p, nbytes); dec(BytesLeft, nbytes); dec(ChunkLen, nbytes);
      end;
    end;
    (pBuf + DataLen)^ := #0; // 0-terminate for casting to string
    Result := pBuf;

    if FORCE_CHECKSUM and (CheckSum <> '') then begin
      // currently the CheckSum field seems to be ignored in FECMakeMetadata
      // -> manually insert it
      slMeta := TStringList.Create;
      try
        FCPMsgToStringList(Result, slMeta);
        if (slMeta.IndexOfName('Info.Checksum') < 0) and (slMeta.IndexOfName('Info.CheckSum') < 0) then begin
          if not InsertStyle.ChkSumAfterDescript then
            i := slMeta.IndexOfName('Info.Format')
          else
            i := slMeta.IndexOfName('Info.Description');
          if i >= 0 then begin
            slMeta.Insert(i+1, 'Info.Checksum='+Checksum);
            Result := StringListToFCPMsg(slMeta,'');
          end;
        end;
      finally
        slMeta.Free;
      end;
    end;

  finally
    if pBuf <> nil then FreeMem(pBuf);
    if bCreatedFCP then fcp.Free;
  end;
end;

function Util_FECEncode_ViaFCP(SegDataStream: TStream; SegInfo: TFECSegmentInfo; fcp: TFCPSession = nil; CheckAbortProc: TCheckAbortProc = nil): PChar;
// stream must already be positioned to correct offset!
// returns allocated check blocks
  procedure CheckAbort;
  begin
    if Assigned(CheckAbortProc) then CheckAbortProc;
  end;
var
  bCreatedFCP: Boolean;
  SegLen: Integer;
  nbytes: Integer;
  s:      String;
  BlockCount,BlockSize: Integer;
  p:        PChar;
  BytesLeft: Integer;
  pDataChunk:   PChar;
  DataChunkLen:  Integer;
  DataLeft: Integer;
begin
  bCreatedFCP := (fcp = nil);
  Result := nil; pDataChunk := nil;
  try
    if bCreatedFCP then fcp := TFCPSession.Create;

    SegLen := SegInfo.BlockCount * SegInfo.BlockSize;
    fcp.SendStringCommand( 'FECEncodeSegment'#10
                         + 'DataLength=' + Hex(Length(SegInfo.FullSegmentHeader) + SegLen) +#10
                         + 'MetadataLength=' + Hex(Length(SegInfo.FullSegmentHeader)) +#10
                         + 'Data'#10
                         + SegInfo.FullSegmentHeader
                         );

    // read & send data in FCP_CHUNKSIZE chunks
    DataChunkLen := FCP_CHUNKSIZE;
    GetMem(pDataChunk, DataChunkLen);
    BytesLeft := SegLen; DataLeft := SegDataStream.Size - SegDataStream.Position;
    while BytesLeft > 0 do begin
      nbytes := DataChunkLen;
      if BytesLeft < nbytes then nbytes := BytesLeft;
      if nbytes <= DataLeft then begin
        SegDataStream.ReadBuffer(pDataChunk^, nbytes);
        dec(DataLeft, nbytes);
      end else begin
        if DataLeft > 0 then SegDataStream.ReadBuffer(pDataChunk^, DataLeft);
        FillChar((pDataChunk+DataLeft)^,nbytes-DataLeft,0); // 0-pad partial or whole(!) blocks
        DataLeft := 0;
      end;
      dec(BytesLeft,nbytes);
      fcp.SendData(pDataChunk, nbytes, True);
      CheckAbort;
    end;

    // wait for reply
    while not fcp.WaitForDataAvailable(1000) do CheckAbort;
    s := fcp.ReadNextMessage;
    if s <> 'BlocksEncoded' then raise Exception.Create('Unexpected message: ' + fcp.MsgAsString);
    BlockCount := StrToInt('$' + fcp.Msg.Values['BlockCount']);
    BlockSize  := StrToInt('$' + fcp.Msg.Values['BlockSize']);
    if BlockCount <> SegInfo.CheckBlockCount then raise Exception.Create('Unexpected number of check blocks');
    if BlockSize  <> SegInfo.CheckBlockSize  then raise Exception.Create('Unexpected size of check blocks');

    // note: data is chunked; each chunk starts:
    // DataChunk<LF>
    // Length=xxxx<LF>
    // Data<LF>

    // s := fcp.ReadNextLine;
    // if s <> 'DataChunk' then raise Exception.Create('Unexpected line in message');

    // alloc one chunk of mem for all the blocks
    GetMem(Result, BlockCount * BlockSize);
    try
      // read the blocks
      p := Result; BytesLeft := (BlockCount * BlockSize);
      while BytesLeft > 0 do begin
        while not fcp.WaitForDataAvailable(1000) do CheckAbort;
        if fcp.ReadNextLine <> 'DataChunk' then raise Exception.Create('Unexpected line in message');
        s := fcp.ReadNextLine; if Copy(s,1,Length('Length=')) <> 'Length=' then raise Exception.Create('Unexpected line in message');
        Delete(s,1,Length('Length=')); DataChunkLen := StrToInt('$'+s);
        if fcp.ReadNextLine <> 'Data' then raise Exception.Create('Unexpected line in message');

        Assert(DataChunkLen <= BytesLeft);

        while DataChunkLen > 0 do begin
          fcp.ReadData(p, DataChunkLen, nbytes);
          // ShowMessage(Format('Got %d bytes: %.2x %.2x %.2x ... %.2x %.2x %.2x',[nbytes,ord(p^),ord((p+1)^),ord((p+2)^), ord((p+nbytes-3)^),ord((p+nbytes-2)^),ord((p+nbytes-1)^)]));
          if nbytes = 0 then raise Exception.Create('Connection died');
          inc(p,nbytes); dec(DataChunkLen,nbytes); dec(BytesLeft,nbytes);
        end;
        CheckAbort;
      end;
    except
      Dispose(Result);
      raise;
    end;

  finally
    if bCreatedFCP then fcp.Free;
    if pDataChunk <> nil then FreeMem(pDataChunk);
  end;
end;

function Util_FECEncode_Native(FecEnc: TFECEncoder; SegDataStream: TStream; SegInfo: TFECSegmentInfo; CheckAbortProc: TCheckAbortProc = nil): PChar;
// stream must already be positioned to correct offset!
// returns allocated check blocks
var
  CheckBlockList: TList;
  i: Integer;
  p: PChar;
begin
  FecEnc.Init(SegInfo.FileLength);
  GetMem(Result, SegInfo.CheckBlockCount * SegInfo.CheckBlockSize);
  try
    CheckBlockList := TList.Create;
    try
      p := Result;
      for i := 0 to SegInfo.CheckBlockCount-1 do begin
        CheckBlockList.Add(p); inc(p, SegInfo.CheckBlockSize);
      end;
      FecEnc.Encode(SegInfo.SegmentNum-1, SegDataStream, CheckBlockList);
    finally
      CheckBlockList.Free;
    end;
  except
    FreeMem(Result);
    raise;
  end;
end;

function Util_FECEncode(SegDataStream: TStream; SegInfo: TFECSegmentInfo; fcp: TFCPSession = nil; CheckAbortProc: TCheckAbortProc = nil): PChar;
var FecEnc: TFECEncoder;
begin
  if USE_NATIVE_FEC then begin
    FecEnc := CreateFECEncoderFromName(SegInfo.FECAlgorithm);
    if FecEnc <> nil then
      try
        Result := Util_FECEncode_Native(FecEnc, SegDataStream, SegInfo, CheckAbortProc);
      finally
        FecEnc.Free;
      end
    else
      Result := Util_FECEncode_ViaFCP(SegDataStream, SegInfo, fcp, CheckAbortProc);
  end else
    Result := Util_FECEncode_ViaFCP(SegDataStream, SegInfo, fcp, CheckAbortProc);
end;

procedure Util_FECEncode_ToStream(SegDataStream: TStream; SegInfo: TFECSegmentInfo; OutputStream: TStream; OutputOffsets: TList; fcp: TFCPSession = nil; CheckAbortProc: TCheckAbortProc = nil);
// like Util_FECEncode, but saves checkblocks directly to a stream
// OutputOffsets must contain exactly num.checkblocks offsets
var
  pCheck,p: PChar;
  i:        Integer;
  FecEnc:   TFECEncoder;
begin
  Assert(SegInfo.CheckBlockCount = OutputOffsets.Count);

  if USE_NATIVE_FEC then begin
    FecEnc := CreateFECEncoderFromName(SegInfo.FECAlgorithm);
    if FecEnc <> nil then
      try
        FecEnc.Init(SegInfo.FileLength);
        FecEnc.Encode(SegInfo.SegmentNum-1, SegDataStream, nil, nil, OutputStream, OutputOffsets);
        exit;
      finally
        FecEnc.Free;
      end;
  end;

  pCheck := Util_FECEncode(SegDataStream, SegInfo, fcp, CheckAbortProc);
  try
    p := pCheck;
    for i := 0 to SegInfo.CheckBlockCount-1 do begin
      OutputStream.Seek(Integer(OutputOffsets.Items[i]), soFromBeginning);
      OutputStream.WriteBuffer(p^, SegInfo.CheckBlockSize);
      inc(p, SegInfo.CheckBlockSize);
    end;
  finally
    FreeMem(pCheck);
  end;
end;

function Util_GenerateCHK_ViaFCP(pData: PChar; DataLen: Integer; sMetadata: String; fcp: TFCPSession; Ciphername: String): String;
var
  bCreatedFCP: Boolean;
begin
  bCreatedFCP := (fcp = nil);
  try
    if bCreatedFCP then fcp := TFCPSession.Create;

    fcp.SendStringCommand( 'GenerateCHK'#10
                         + 'Cipher=' + Ciphername + #10
                         + 'DataLength=' + Hex(DataLen + Length(sMetadata)) + #10
                         + 'MetadataLength=' + Hex(Length(sMetadata)) + #10
                         + 'Data'#10
                         + sMetadata
                         );
    if DataLen > 0 then fcp.SendData(pData, DataLen, True);
    if fcp.ReadNextMessage <> 'Success' then raise Exception.Create('Failed to calculate CHK');
    Result := fcp.Msg.Values['URI'];
  finally
    if bCreatedFCP then fcp.Free;
  end;
end;

function Util_GenerateCHK_Native(pData: PChar; DataLen: Integer; sMetadata: String; Ciphername: String): String;
const
  LOG2_MINSIZE = 10;
  LOG2_MAXSIZE = 62;
  HASH_EQUALS  = #$FE;
  HASH_NEWLINE = #$FF;
  CHK_KEYNUM   = $0302;
  SAFETY: packed Array [0..1] of Cardinal = ($DEADBEEF,$ABADCAFE);

  function  HexDump(pData: Pointer; DataLen: Integer): String;
  var
    p:     PChar;
    sH,sA: String;
    i,off: Integer;
  begin
    Result := ''; sH := ''; sA := ''; p := pData; off := 0;
    for i := 0 to DataLen-1 do begin
      sH := sH + ' ' + IntToHex(Ord(p^),2);
      if p^ in [#32..#127] then sA := sA + p^ else sA := sA + '.';
      inc(p);
      if ((i and 15) = 15) or (i = DataLen-1) then begin
        Result := Result + Format('%.8X:%-50s %s',[off,sH,sA]) + #13#10;
        inc(off,16); sH := ''; sA := '';
      end;
    end;
  end;

  procedure FillHead(var p: PByte; pData: PByte; DataLen: Integer); overload;
  begin
    p^ := HiByte(DataLen);     inc(p);
    p^ := LoByte(DataLen);     inc(p);
    Move(pData^, p^, DataLen); inc(p,DataLen);
  end;

  procedure FillHead(var p: PByte; num: Cardinal); overload;
  var i,fieldsize: Integer;
  begin
    fieldsize := (Freenet_log2(num+1)+7) shr 3; if fieldsize = 0 then fieldsize := 1;
    p^ := HiByte(fieldsize);  inc(p);
    p^ := LoByte(fieldsize);  inc(p);
    i := 8*(fieldsize-1);
    while i >= 0 do begin
      p^ := (num shr i) and $FF; inc(p);
      dec(i,8);
    end;
  end;

  function BytesToHex(pData: PByte; DataLen: Integer): String;
  const HEXDIGS = '0123456789abcdef';
  begin
    Result := '';
    while DataLen > 0 do begin
      Result := Result + HEXDIGS[1+ ((pData^ shr 4) and $0F)] + HEXDIGS[1+ (pData^ and $0F)];
      inc(pData); dec(DataLen);
    end;
  end;

var
  pBuf,p:   PByte;
  Cipher:   TCipher;
  PCFBMode: TPeriodicCipherFeedbackMode;
  dig:      TSHA1Digest;
  sha:      TSHA1Context;
  OrigDataSHA: TSHA1Context;
  OrigDataDIG: TSHA1Digest;
  i,x:      Integer;
  Len,PadLen:   Cardinal;
  PaddedLength: Cardinal;
  PartSize:     Cardinal;
  SHAList:      TList;
  pSHA:         PSHA1Context;
  count,chunk:  Cardinal;
  BufSize:      Cardinal;
  pCryptKey:    PByte;
  CryptKeyLen:  Cardinal;
  ic:           Integer;
  b:            Byte;
  sEncHeader:   String;
  sIniDigest:   String;
  s:            String;
  TotalLen:     Cardinal;
  MetaLen:      Cardinal;
  pTmp:         PByte;
begin
  MetaLen  := Length(sMetadata);
  TotalLen := MetaLen + Cardinal(DataLen);
  pBuf := nil; pCryptKey := nil; Cipher := nil; PCFBMode := nil; SHAList := nil;
  try
    x := Freenet_log2(TotalLen); if x < LOG2_MINSIZE then x := LOG2_MINSIZE;
    PaddedLength := 1 shl x;
    BufSize := PaddedLength + SizeOf(SAFETY);
    GetMem(pBuf, BufSize); FillChar(pBuf^, BufSize, 0);
    Move(SAFETY, (PChar(pBuf) + Bufsize - SizeOf(SAFETY))^, SizeOf(SAFETY));

    Cipher := CreateCipherByName(Ciphername);
    CryptKeyLen := Cipher.KeySize shr 3;
    GetMem(pCryptKey, CryptKeyLen);

    // calc hash of plain, unpadded data
    SHA1Reset(sha);
    SHA1Input(sha, PChar(sMetadata), MetaLen);
    SHA1Input(sha, pData,            DataLen);
    OrigDataSHA := sha; // remember for later - BEFORE getting the digest!
    SHA1Result(sha, OrigDataDIG);

    // calc cryptokey
    ic := 0; Len := CryptKeyLen; p := pCryptKey;
    while Len > 0 do begin
      SHA1Reset(sha);
      inc(ic);
      b := 0; for i := 1 to ic do SHA1Input(sha, @b, 1);
      SHA1Input(sha, @OrigDataDIG, SizeOf(OrigDataDIG));
      SHA1Result(sha, dig);
      if Len > SizeOf(dig) then begin
        Move(dig, p^, SizeOf(dig)); inc(p, SizeOf(dig)); dec(Len, SizeOf(dig));
      end else begin
        Move(dig, p^, Len); inc(p, Len); Len := 0;
      end;
    end;

    Cipher.Initialize(pCryptKey, CryptKeyLen);
    PCFBMode := TPeriodicCipherFeedbackMode.Create(Cipher);

    // Document header
    p := pBuf;
    Util_CalcSHA1Digest(pCryptKey, CryptKeyLen, dig);
    FillHead(p, @dig, SizeOf(dig));
    FillHead(p, TotalLen);
    FillHead(p, MetaLen);
    p^ := 0; inc(p);
    p^ := 0; inc(p);

    Len := PChar(p) - PChar(pBuf);
    PadLen := (1 shl Freenet_log2(Len)) - Len;
    if PadLen > 0 then begin
      SHA1Reset(sha);
      SHA1Input(sha, PChar(pBuf), Len);
      Util_RollingHashPad(p, PadLen, sha); inc(p, PadLen);
    end;

    Len := PChar(p) - PChar(pBuf); p := pBuf;
    for i := 0 to Len-1 do begin
      p^ := PCFBMode.Encipher(p^); inc(p);
    end;
    sEncHeader := BytesToHex(pBuf, Len);

    //ShowDebugLines(HexDump(pBuf, PChar(p) - PChar(pBuf)), 'Encrypted header');

    // encode doc

    PadLen := PaddedLength - TotalLen;
    PartSize := PaddedLength;
    if PartSize > 16384 then begin
      PartSize := PaddedLength shr 7;
      if PartSize < 16384 then PartSize := 16384;
    end;

    {
    // pad out plaintext
    p := pBuf; FillChar(pBuf^, BUFSIZE, 0);
    Move(PLAINTEXT, p^, SizeOf(PLAINTEXT)); inc(p, SizeOf(PLAINTEXT));
    SHA1Reset(sha);
    SHA1Input(sha, PChar(pBuf), SizeOf(PLAINTEXT));
    Util_RollingHashPad(p, PadLen, sha); inc(p, PadLen);

    //ShowDebugLines(HexDump(pBuf, PChar(p) - PChar(pBuf)), 'Padded plaintext');

    // encrypt padded plaintext
    Len := PChar(p) - PChar(pBuf); p := pBuf;
    for i := 0 to Len-1 do begin
      p^ := PCFBMode.Encipher(p^); inc(p);
    end;
    }
    // encrypt unpadded plaintext to buffer
    p := pBuf;
    if MetaLen > 0 then begin
      pTmp := PByte(PChar(sMetadata));
      for i := 0 to MetaLen-1 do begin
        p^ := PCFBMode.Encipher(pTmp^); inc(p); inc(pTmp);
      end;
    end;
    if DataLen > 0 then begin
      pTmp := PByte(pData);
      for i := 0 to DataLen-1 do begin
        p^ := PCFBMode.Encipher(pTmp^); inc(p); inc(pTmp);
      end;
    end;
    if PadLen > 0 then begin
      // pad plaintext directly into buffer; use undigested hash calced before
      Util_RollingHashPad(p, PadLen, OrigDataSHA);
      // encrypt padding
      for i := 0 to PadLen-1 do begin
        p^ := PCFBMode.Encipher(p^); inc(p);
      end;
    end;

    //ShowDebugLines(HexDump(pBuf, PChar(p) - PChar(pBuf)), 'Encrypted plaintext');

    // calc initial digest (progressive hash)
    SHAList := TList.Create;
    p := pBuf; count := 0; New(pSHA); SHA1Reset(pSHA^);
    Len := PaddedLength;
    while Len > 0 do begin
      chunk := PartSize - count; if chunk > Len then chunk := Len;
      if chunk = 0 then begin
        SHAList.Insert(0,pSHA); New(pSHA); SHA1Reset(pSHA^); count := 0;
      end else begin
        SHA1Input(pSHA^, PChar(p), chunk); inc(p,chunk);
        inc(count,chunk); dec(Len,chunk);
      end;
    end;
    while SHAList.Count > 0 do begin
      SHA1Result(pSHA^, dig); Dispose(pSHA);
      pSHA := SHAList.Items[0]; SHAList.Delete(0);
      SHA1Input(pSHA^, @dig, SizeOf(dig));
    end;
    SHA1Result(pSHA^, dig); Dispose(pSHA);
    sIniDigest := SHA1DigestToLoHexString(dig);

    //ShowMessage('Initial digest: ' + SHA1DigestToLoHexString(dig));

    // calculate storables hash
    s := 'Document-header'  + HASH_EQUALS + sEncHeader    + HASH_NEWLINE
       + 'Initial-digest'   + HASH_EQUALS + sIniDigest    + HASH_NEWLINE
       + 'Part-size'        + HASH_EQUALS + Hex(PartSize) + HASH_NEWLINE
       + 'Symmetric-cipher' + HASH_EQUALS + Ciphername    + HASH_NEWLINE;
    Util_CalcSHA1Digest(PChar(s), Length(s), dig);

    // almost have the routing key.. append log2size and keynumber
    p := pBuf;
    Move(dig, p^, SizeOf(dig)); inc(p, SizeOf(dig));
    p^ := Freenet_log2(PaddedLength); inc(p);
    p^ := HiByte(CHK_KEYNUM); inc(p);
    p^ := LoByte(CHK_KEYNUM); inc(p);

    //ShowDebugLines(HexDump(pBuf, PChar(p) - PChar(pBuf)), 'Routing key');

    Len := PChar(p) - PChar(pBuf);
    Result := 'freenet:CHK@' + Base64encode(PChar(pBuf), Len) + ',' + Base64encode(PChar(pCryptKey), CryptKeyLen);

    // ShowMessage('Key: ' + s);

    if not CompareMem(@SAFETY, PChar(pBuf) + Bufsize - SizeOf(SAFETY), SizeOf(SAFETY)) then
      raise Exception.Create('Buffer overflow detected in Util_GenerateCHK_Native!');
  finally
    if pBuf <> nil then FreeMem(pBuf);
    if pCryptKey <> nil then FreeMem(pCryptKey);
    Cipher.Free;
    PCFBMode.Free;
    SHAList.Free;
  end;
end;

function Util_GenerateCHK(pData: PChar; DataLen: Integer; sMetadata: String; fcp: TFCPSession = nil; Ciphername: String = CIPHERNAME_TWOFISH): String;
begin
  if USE_NATIVE_CHK then
    Result := Util_GenerateCHK_Native(pData, DataLen, sMetadata, Ciphername)
  else
    Result := Util_GenerateCHK_ViaFCP(pData, DataLen, sMetadata, fcp, Ciphername);
end;

type
  TDynamicStringArray = Array of String;

function  Util_FECSegmentFile_ViaFCP(FileLen: Integer; out SegmentHeaders: TDynamicStringArray; fcp: TFCPSession = nil): Integer;
// Headers are SetLength'ed to NumSegs here
// NumSegs is also returned
var
  sMsg: String;
  bCreatedFCP: Boolean;
  NumSegs,i: Integer;
begin
  SetLength(SegmentHeaders, 0); Result := 0;
  bCreatedFCP := (fcp = nil);
  try
    try
      if bCreatedFCP then fcp := TFCPSession.Create;

      fcp.SendStringCommand( 'FECSegmentFile'#10
                           + 'AlgoName=OnionFEC_a_1_2'#10
                           + 'FileLength=' + Hex(FileLen) + #10
                           + 'EndMessage'#10
                           );

      NumSegs := 0;
      repeat
        sMsg := fcp.ReadNextMessage;
        if sMsg = '' then break;
        if sMsg <> 'SegmentHeader' then raise Exception.Create('FECSegmentFile: unexpected header');

        if NumSegs = 0 then begin
          NumSegs := StrToIntDef('$'+fcp.Msg.Values['Segments'], 0);
          if NumSegs = 0 then raise Exception.Create('FECSegmentFile: invalid message');
          SetLength(SegmentHeaders, NumSegs);
          for i := 0 to NumSegs-1 do SegmentHeaders[i] := '';
        end;

        i := StrToIntDef('$'+fcp.Msg.Values['SegmentNum'], -1);
        if (i < 0) or (i >= NumSegs) then raise Exception.Create('FECSegmentFile: invalid message');
        SegmentHeaders[i] := StringListToFCPMsg(fcp.Msg);
      until False;

      if NumSegs = 0 then raise Exception.Create('FECSegmentFile failed');
      for i := 0 to NumSegs-1 do
        if SegmentHeaders[i] = '' then
          raise Exception.Create('FECSegmentFile incomplete');

      Result := NumSegs;
    finally
      if bCreatedFCP then fcp.Free;
    end;
  except
    SetLength(SegmentHeaders, 0);
    raise;
  end;
end;

function  Util_FECSegmentFile_Native(FecEnc: TFECEncoder; FileLen: Integer; out SegmentHeaders: TDynamicStringArray): Integer;
var
  i:  Integer;
  sh: TFECSegmentHeader;
  sl: TStringList;
begin
  SetLength(SegmentHeaders, 0); sl := nil;
  try
    try
      FecEnc.Init(FileLen);
      Result := FecEnc.NumSegments;
      SetLength(SegmentHeaders, FecEnc.NumSegments);

      sl := TStringList.Create;
      for i := 0 to FecEnc.NumSegments-1 do begin
        sh := FecEnc.GetSegmentHeader(i);
        // build string in same order as FCP does (shouldn't matter)
        sl.Clear;
        sl.Add('SegmentHeader');
        sl.Add('CheckBlockOffset='  +Hex(sh.CheckBlockOffset));
        sl.Add('SegmentNum='        +Hex(sh.SegmentNum));
        sl.Add('FECAlgorithm='      +    sh.AlgoName);
        sl.Add('Segments='          +Hex(sh.Segments));
        sl.Add('DataBlockOffset='   +Hex(sh.DataBlockOffset));
        sl.Add('CheckBlockSize='    +Hex(sh.CheckBlockSize));
        sl.Add('Offset='            +Hex(sh.Offset));
        sl.Add('BlockCount='        +Hex(sh.BlockCount));
        sl.Add('BlockSize='         +Hex(sh.BlockSize));
        sl.Add('BlocksRequired='    +Hex(sh.BlocksRequired));
        sl.Add('FileLength='        +Hex(sh.FileLength));
        sl.Add('CheckBlockCount='   +Hex(sh.CheckBlockCount));
        SegmentHeaders[i] := StringListToFCPMsg(sl); // appends 'EndMessage'
      end;
    finally
      sl.Free;
    end;
  except
    SetLength(SegmentHeaders, 0);
    raise;
  end;
end;

function  Util_FECSegmentFile(FileLen: Integer; out SegmentHeaders: TDynamicStringArray; fcp: TFCPSession = nil): Integer;
var FecEnc: TFECEncoder;
begin
  if USE_NATIVE_FEC then begin
    FecEnc := CreateFECEncoderFromName(ALGONAME_ONIONFEC_A_1_2);
    if FecEnc <> nil then
      try
        Result := Util_FECSegmentFile_Native(FecEnc, FileLen, SegmentHeaders);
      finally
        FecEnc.Free;
      end
    else
      Result := Util_FECSegmentFile_ViaFCP(FileLen, SegmentHeaders, fcp);
  end else
    Result := Util_FECSegmentFile_ViaFCP(FileLen, SegmentHeaders, fcp);
end;

function Util_FECSegmentSplitfile_Native(SplitfileMetadata: String; out SegmentHeaders,BlockMaps: TDynamicStringArray): Integer;
var
  FecEnc:          TFECEncoder;
  slMeta,slBlocks: TStringList;
  iSeg,iBlock:     Integer;
  SegInfo:         TFECSegmentInfo;
  s:               String;
begin
  SetLength(SegmentHeaders, 0); SetLength(BlockMaps, 0); Result := 0;

  try
    slMeta := nil; slBlocks := nil; FecEnc := nil;
    try
      slMeta := TStringList.Create;
      slMeta.Text := SplitfileMetadata;

      FecEnc := CreateFECEncoderFromName(slMeta.Values['SplitFile.AlgoName']);
      if FecEnc = nil then exit;

      // use FECSegmentFile to get the segment headers
      Result := Util_FECSegmentFile_Native(FecEnc, StrToInt('$'+slMeta.Values['SplitFile.Size']), SegmentHeaders);

      // now build the BlockMaps
      slBlocks := TStringList.Create;
      SetLength(BlockMaps, Result);
      for iSeg := 0 to Result-1 do begin
        Util_GetFECSegmentInfo(SegmentHeaders[iSeg], SegInfo);
        slBlocks.Clear;
        slBlocks.Add('BlockMap');
        for iBlock := 0 to SegInfo.BlockCount-1 do begin
          s := slMeta.Values['SplitFile.Block.' + Hex(SegInfo.DataBlockOffset + iBlock + 1)];
          if s = '' then raise ENativeImplError.Create('Error in Util_FECSegmentSplitfile_Native');
          slBlocks.Add('Block.' + Hex(iBlock) + '=' + s);
        end;
        for iBlock := 0 to SegInfo.CheckBlockCount-1 do begin
          s := slMeta.Values['SplitFile.CheckBlock.' + Hex(SegInfo.CheckBlockOffset + iBlock + 1)];
          if s = '' then raise ENativeImplError.Create('Error in Util_FECSegmentSplitfile_Native');
          slBlocks.Add('Check.' + Hex(iBlock) + '=' + s);
        end;
        BlockMaps[iSeg] := StringListToFCPMsg(slBlocks); // appends 'EndMessage'
      end;

    finally
      slMeta.Free;
      slBlocks.Free;
      FecEnc.Free;
    end;
  except
    SetLength(SegmentHeaders, 0); SetLength(BlockMaps, 0);
    raise;
  end;
end;

function Util_FECSegmentSplitfile(SplitfileMetadata: String; out SegmentHeaders,BlockMaps: TDynamicStringArray; fcp: TFCPSession = nil): Integer;
// Headers,Maps are SetLength'ed to NumSegs here
// NumSegs is also returned
var
  bCreatedFCP: Boolean;
  sCmd,sMsg: String;
  iSeg,NumSegs: Integer;
begin
  if USE_NATIVE_FEC then begin
    try
      Result := Util_FECSegmentSplitfile_Native(SplitfileMetadata, SegmentHeaders, BlockMaps);
    except
      Result := 0;
    end;
    if Result <> 0 then exit;
  end;

  SetLength(SegmentHeaders, 0); SetLength(BlockMaps, 0); Result := 0;
  bCreatedFCP := (fcp = nil);
  try
    try
      if bCreatedFCP then fcp := TFCPSession.Create;

      sCmd := 'FECSegmentSplitFile'#10
            + 'DataLength=' + Hex(Length(SplitfileMetadata)) + #10
            + 'Data'#10
            + SplitfileMetadata;
      fcp.SendStringCommand(sCmd);

      iSeg := 0; NumSegs := 0;
      repeat
        sMsg := fcp.ReadNextMessage; // Segment header
        if sMsg <> 'SegmentHeader' then raise Exception.Create('Unexpected message:' + fcp.MsgAsString);
        if iSeg = 0 then begin
          NumSegs := StrToInt('$' + fcp.Msg.Values['Segments']);
          SetLength(SegmentHeaders, NumSegs); SetLength(BlockMaps, NumSegs);
        end;
        Assert(iSeg = StrToInt('$' + fcp.Msg.Values['SegmentNum']), 'Segment headers not in sequence');
        SegmentHeaders[iSeg] := StringListToFCPMsg(fcp.Msg);

        sMsg := fcp.ReadNextMessage; // Block map
        if sMsg <> 'BlockMap' then raise Exception.Create('Unexpected message:' + fcp.MsgAsString);
        BlockMaps[iSeg] := StringListToFCPMsg(fcp.Msg);

        inc(iSeg);
      until iSeg >= NumSegs;

      Result := NumSegs;
    finally
      if bCreatedFCP then fcp.Free;
    end;
  except
    SetLength(SegmentHeaders, 0); SetLength(BlockMaps, 0);
    raise;
  end;
end;

procedure Util_GetFECSegmentInfo(SegmentHeaderStrList: TStringList; out SegInfo: TFECSegmentInfo); overload;
var
  sl: TStringList;
  i:  Integer;
begin
  sl := SegmentHeaderStrList;
  SegInfo.FECAlgorithm      := sl.Values['FECAlgorithm'];
  SegInfo.FileLength        := StrToInt('$'+sl.Values['FileLength']);
  SegInfo.Offset            := StrToInt('$'+sl.Values['Offset']);
  SegInfo.DataBlockOffset   := StrToInt('$'+sl.Values['DataBlockOffset']);
  SegInfo.BlockCount        := StrToInt('$'+sl.Values['BlockCount']);
  SegInfo.BlockSize         := StrToInt('$'+sl.Values['BlockSize']);
  SegInfo.CheckBlockOffset  := StrToInt('$'+sl.Values['CheckBlockOffset']);
  SegInfo.CheckBlockCount   := StrToInt('$'+sl.Values['CheckBlockCount']);
  SegInfo.CheckBlockSize    := StrToInt('$'+sl.Values['CheckBlockSize']);
  SegInfo.Segments          := StrToInt('$'+sl.Values['Segments']);
  SegInfo.SegmentNum        := StrToInt('$'+sl.Values['SegmentNum']);
  SegInfo.BlocksRequired    := StrToInt('$'+sl.Values['BlocksRequired']);
  SegInfo.FullSegmentHeader := '';
  for i := 0 to sl.Count-1 do
    SegInfo.FullSegmentHeader := SegInfo.FullSegmentHeader + sl.Strings[i] + #10;
  if sl.Strings[sl.Count-1] <> 'EndMessage' then
    SegInfo.FullSegmentHeader := SegInfo.FullSegmentHeader + 'EndMessage'#10;
end;

procedure Util_GetFECSegmentInfo(SegmentHeader: String; out SegInfo: TFECSegmentInfo); overload;
var sl: TStringList;
begin
  sl := TStringList.Create;
  try
    FCPMsgToStringList(SegmentHeader, sl);
    Util_GetFECSegmentInfo(sl, SegInfo);
  finally
    sl.Free;
  end;
end;

function Util_NormKey(const sKey: String): String;
begin
  Result := sKey;
  if CompareText(Copy(Result,1,Length('freenet:')),'freenet:') = 0 then
    Delete(Result,1,Length('freenet:'));
  if CompareText(Copy(Result,1,4),'CHK@') <> 0 then exit;
  if Pos('/',Result) <> 0 then Delete(Result,Pos('/',Result),Length(Result));
end;


{ TFCPSession }

constructor TFCPSession.Create(FCPAddr: u_long; FCPPort: Word; UseDebugBuffer: Boolean);
var i: Integer;
begin
  inherited Create;
  FAddr.S_addr := htonl(FCPAddr);
  FPort := FCPPort;
  FSock := INVALID_SOCKET;
  FRecvd := '';
  for i := 0 to 1 do begin
    FDebug[i].Use     := UseDebugBuffer;
    FDebug[i].Buf     := nil;
    FDebug[i].BufSize := 0;
    FDebug[i].BufPos  := 0;
  end;
  FDebugInfo := nil;
end;

destructor TFCPSession.Destroy;
begin
  KillSocket;
  if FslMsg <> nil then FslMsg.Free;
  if FDebug[0].Buf <> nil then FreeMem(FDebug[0].Buf);
  if FDebug[1].Buf <> nil then FreeMem(FDebug[1].Buf);
  FDebugInfo.Free;
  inherited;
end;

procedure TFCPSession.KillSocket;
{$IFDEF FCP_HARDKILLSOCKS}
var linger: TLinger;
{$ENDIF}
begin
  if FSock = INVALID_SOCKET then exit;
  {$IFDEF FCP_HARDKILLSOCKS}
    linger.l_onoff  := 1;
    linger.l_linger := 0;
    setsockopt(FSock, SOL_SOCKET, SO_LINGER, @linger, SizeOf(linger));
  {$ENDIF}
  closesocket(FSock);
  FSock := INVALID_SOCKET;
end;

procedure TFCPSession.LogToDebugBuffer(IsSend: Boolean; pData: PChar; DataLen: Cardinal);
var
  dbgSize: Integer;
  dbgIdx:  Integer;
  pDebug:  PChar;
  c:       Char;
begin
  if IsSend then dbgIdx := 0 else dbgIdx := 1;
  if (not FDebug[dbgIdx].Use) or (DataLen = 0) then exit;

  if (FDebug[dbgIdx].BufPos + Integer(DataLen)) > FDebug[dbgIdx].BufSize then begin
    dbgSize := DataLen; if dbgSize < (1024*1024) then dbgSize := 1024*1024; // inc. buf size by nbytes, but at least by 1M
    inc(FDebug[dbgIdx].BufSize, dbgSize);
    ReallocMem(FDebug[dbgIdx].Buf, FDebug[dbgIdx].BufSize);
  end;
  pDebug := FDebug[dbgIdx].Buf + FDebug[dbgIdx].BufPos;
  Move(pData^, pDebug^, DataLen);
  if IsSend then c := '>' else c := '<';
  if FDebugInfo = nil then FDebugInfo := TStringList.Create;
  FDebugInfo.Add( Format('%s %s %.8x %.8x',[FormatDateTime('yyyy-mm-dd hh:nn:ss.zzz',Now),c,FDebug[dbgIdx].BufPos,DataLen]) );
  inc(FDebug[dbgIdx].BufPos, DataLen);
end;

procedure TFCPSession.ConnectToFCP;
var
  sin: TSockAddrIn;
begin
  // create socket
  FSock := socket(AF_INET, SOCK_STREAM, 0);
  if FSock = INVALID_SOCKET then raise EFCPError.CreateFmt('socket failed: %d',[WSAGetLastError]);

  // connect
  FillChar(sin, SizeOf(sin), 0);
  sin.sin_family := AF_INET;
  sin.sin_port   := htons(FPort);
  sin.sin_addr   := FAddr;
  if 0 <> connect(FSock, sin, SizeOf(sin)) then raise EFCPError.CreateFmt('connect failed: %d',[WSAGetLastError]);
end;

procedure TFCPSession.SendData(pData: Pointer; DataLen: Cardinal; ReuseConnection: Boolean);
const
  SENDSIZE = FCP_CHUNKSIZE;
var
  p:   PChar;
  len: Cardinal;
  nbytes: Integer;
{$IFDEF DEBUG_LOGFCPIN}
  F: File;
  bNewFile: Boolean;
{$ENDIF}
begin
  if ReuseConnection then
    Assert(FSock <> INVALID_SOCKET, 'No session created')
  else
    Assert(FSock = INVALID_SOCKET, 'Session already used');
  try
    if not ReuseConnection then ConnectToFCP;

    // send data
    p := PChar(pData); len := DataLen;
    while len > 0 do begin
      if len > SENDSIZE then nbytes := SENDSIZE else nbytes := len;
      nbytes := send(FSock, p^, nbytes, 0);
      if nbytes = SOCKET_ERROR then raise EFCPError.CreateFmt('send failed: %d',[WSAGetLastError]);

      LogToDebugBuffer(True, p, nbytes);

{$IFDEF DEBUG_LOGFCPIN}
      if (nbytes > 0) then begin
        AssignFile(F, DEBUG_LOGFCPINFILE);
        try
          bNewFile := not FileExists(DEBUG_LOGFCPINFILE);
          if bNewFile then Rewrite(F,1) else begin FileMode := 2; Reset(F,1); end;
          try
            if not bNewFile then Seek(F,FileSize(F));
            BlockWrite(F, p^, nbytes);
          finally
            CloseFile(F);
          end;
        except
        end;
      end;
{$ENDIF}

      dec(len, nbytes); inc(p,nbytes);
    end;

  except
    KillSocket;
    raise;
  end;
end;

procedure TFCPSession.SendStringCommand(sCmd: String);
// prefixes 0x 00 00 00 02 and sends sCmd
// EndMessage (if necessary) has to be specified manually
var
  pData: pChar;
  Len:   Integer;
begin
//showmessage(scmd);
  Len := SizeOf(FCP_COMMAND_HEADER) + Length(sCmd);
  GetMem(pData, Len);
  try
    Move(FCP_COMMAND_HEADER, pData^, SizeOf(FCP_COMMAND_HEADER));
    Move(PChar(sCmd)^, (pData + SizeOf(FCP_COMMAND_HEADER))^, Length(sCmd));
    SendData(pData, Len);
  finally
    FreeMem(pData);
  end;
end;

procedure TFCPSession.ReadData(pBuf: Pointer; BufSize: Integer; out BytesRead: Integer);
{$IFDEF DEBUG_LOGFCPOUT}
var
  F: File;
  bNewFile: Boolean;
  //sDbg: String;
{$ENDIF}
begin
  Assert(FSock <> INVALID_SOCKET, 'No session created');

  // if there's something in FRecvd (from ReadNextMessage) return this first
  if Length(FRecvd) > 0 then begin // interprete as binary string!
    if Length(FRecvd) > BufSize then BytesRead := BufSize else BytesRead := Length(FRecvd);
    Move(PChar(FRecvd)^, pBuf^, BytesRead);
    Delete(FRecvd,1,BytesRead);

    //sDbg := format('*** returned %d bytes from buf, remaining: %d ***',[bytesread,Length(FRecvd)]);
    //OutputDebugString(pchar(sDbg));
  end else begin
    BytesRead := recv(FSock, pBuf^, BufSize, 0);
    if BytesRead = SOCKET_ERROR then raise EFCPError.CreateFmt('recv failed: %d',[WSAGetLastError]);

    //sDbg := format('*** received %d bytes ***',[bytesread]);
    //OutputDebugString(pchar(sDbg));

    LogToDebugBuffer(False, pBuf, BytesRead);

{$IFDEF DEBUG_LOGFCPOUT}
    if (BytesRead > 0) then begin
      AssignFile(F, DEBUG_LOGFCPOUTFILE);
      try
        bNewFile := not FileExists(DEBUG_LOGFCPOUTFILE);
        if bNewFile then Rewrite(F,1) else begin FileMode := 2; Reset(F,1); end;
        try
          if not bNewFile then Seek(F,FileSize(F));
          BlockWrite(F, pBuf^, BytesRead);
        finally
          CloseFile(F);
        end;
      except
      end;
    end;
{$ENDIF}
  end;
end;

function TFCPSession.WaitForDataAvailable(TimeoutMillisec: Integer; ConsiderBufferedData: Boolean): Boolean;
// Wait util data is available or Timeout ms have passed
// Return True if data available, False if timed out
var
  readfds:  TFDSet;
  tval:     TTimeval;
  pTimeout: PTimeval;
  res:      Integer;
begin
  Assert(FSock <> INVALID_SOCKET, 'No session created');

  if ConsiderBufferedData and (Length(FRecvd) > 0) then begin
    Result := True; exit; // buffered data available
  end;

  FD_ZERO(readfds);
  FD_SET(FSock, readfds);
  if TimeoutMillisec < 0 then
    pTimeout := nil
  else begin
    tval.tv_sec  := TimeoutMillisec div 1000;
    tval.tv_usec := (TimeoutMillisec mod 1000) * 1000;
    pTimeout := @tval;
  end;
  res := select(1, @readfds, nil, nil, pTimeout);
  if res = SOCKET_ERROR then raise EFCPError.CreateFmt('select failed: %d',[WSAGetLastError]);
  Result := res <> 0;
end;


function TFCPSession.ReadNextLine: String;
// reads complete line (LF terminated)
var
  rbuf:   packed array [0..1024] of Char;
  nbytes: Integer;
  i,iPos: Integer;
  s:      String;
  sLine:  String;
begin
  Result := ''; sLine := '';
  try
    repeat
      iPos := Pos(#10,PChar(sLine)); // interprete as c-string!
      if iPos <> 0 then begin
        s := Copy(sLine,1,iPos-1); Delete(sLine,1,iPos);
        // if Copy(s,Length(s),1) = #13 then Delete(s,Length(s),1);
        Result := s; exit;
      end;

      ReadData(@rbuf, SizeOf(rbuf)-1, nbytes);
      rbuf[nbytes] := #0;
      // FRecvd := FRecvd + rbuf;
      for i := 0 to nbytes-1 do sLine := sLine + rbuf[i]; // do this so we copy also #0-bytes
      {
      if nbytes > 0 then begin
        i := Length(sLine);
        SetLength(sLine, i+nbytes);
        Move(rbuf[0],(PChar(sLine)+i)^,nbytes+1); // copy 0-byte too
      end;
      }
    until nbytes = 0;
  finally
    if sLine <> '' then Insert(sLine, FRecvd, 1); // push back unprocessed data
  end;
end;

function  TFCPSession.ReadNextMessage(EndMarker: String): String;
// reads complete msg into Msg
// 1st line of Msg is Result
var s: String;
begin
  Result := '';
  if FslMsg = nil then FslMsg := TStringList.Create;
  FslMsg.Clear;

  repeat
    s := ReadNextLine;
    if s <> '' then begin
      if s = EndMarker then begin
        // Message complete
        exit;
      end else begin
        if FslMsg.Count = 0 then Result := s;
        FslMsg.Add(s);
      end;
    end;
  until s = '';
end;

function TFCPSession.GetMsgAsString: String;
// return Msg as string (for logging output)
var i: Integer;
begin
  Result := '[';
  for i := 0 to Msg.Count-1 do begin
    if i > 0 then Result := Result + ',';
    Result := Result + '"' + Msg.Strings[i] + '"';
  end;
  Result := Result + ']';
end;

procedure TFCPSession.WriteDebugBuffer(SuggestedFileName: String);
var
  fn,s:  String;
  i,cnt: Integer;
  str:   TFileStream;
begin
  if SuggestedFileName = '' then SuggestedFileName := 'fcpdebug_' + FormatDateTime('yyyy-mm-dd_hh.nn.ss',Now);
  if ExtractFilePath(SuggestedFileName) = '' then
    Insert(ExtractFilePath(Application.ExeName), SuggestedFileName, 1);
  fn := SuggestedFileName; cnt := 0;
  while FileExists(fn + '_send.log') or FileExists(fn + '_recv.log') or FileExists(fn + '_info.log')  do begin
    inc(cnt); fn := SuggestedFileName + '_' + IntToStr(cnt);
  end;

  for i := 0 to 1 do begin
    if FDebug[i].BufPos > 0 then begin
      if i = 0 then s := fn + '_send.log' else s := fn + '_recv.log';
      str := CreateFileStream(s, fmCreate or fmShareExclusive);
      str.WriteBuffer(FDebug[i].Buf^, FDebug[i].BufPos);
      str.Free;
    end;
  end;
  if Assigned(FDebugInfo) then FDebugInfo.SaveToFile(fn + '_info.log');
end;

{ TFreenetThread }

constructor TFreenetThread.Create(
  ThreadID: String; FCPAddr: u_long; FCPPort: Word;
  OnStatus: TTextMsgEvent; AUserData: Pointer; AUserID: Int64;
  CreateSuspended: Boolean);
begin
  inherited Create(True); // suspended
  FID        := ThreadID;
  FFCPAddr := FCPAddr; FFCPPort := FCPPort;
  FOnStatus := OnStatus;
  FUserData  := AUserData;
  FUserID    := AUserID;
  FDone      := False;
  FUnnecessary := False;
  InitStats(FStats);
  if not CreateSuspended then Resume;
end;

procedure TFreenetThread.DoStatus(Msg: String; Level: Integer);
const MsgID: Cardinal = 0;
begin
  if not ENABLE_THREAD_LOGGING then exit;
  if Level < LOG_LEVEL then exit;

  // if Terminated then exit;
  if Assigned(FOnStatus) then begin
    if MsgID = $FFFFFFFF then MsgID := 0 else inc(MsgID);
    FStatusMsgID  := MsgID;
    FStatusMsg    := Msg;
    FStatusMsgLvl := Level;
    Synchronize(DoStatusSynched);
  end;
end;

procedure TFreenetThread.DoStatusSynched;
begin
  FOnStatus(Self,FStatusMsg,FStatusMsgLvl);
end;

function TFreenetThread.NewFCP(UseDebugBuffer: Boolean): TFCPSession;
begin
  Result := TFCPSession.Create(FFCPAddr,FFCPPort,UseDebugBuffer);
end;

procedure TFreenetThread.CheckAbort;
begin
  if Terminated then raise EThreadAborted.Create('Aborted');
end;

{ TInsertThread }

constructor TInsertThread.Create(
  ThreadID: String;
  FCPAddr: u_long; FCPPort: Word;
  InsertData: PChar; InsertLen: Integer; Metadata: String;
  InsertHTL: Integer; SkipLocal: Boolean; NumRetries,ABlockNum: Integer;
  IsDataBlock,IsCheckBlock,IsHeaderBlock,FreeDataOnExit: Boolean;
  AInsertStyle: TInsertStyle; OnStatus: TTextMsgEvent; TimeoutMinutes: Integer;
  AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FInsertData    := InsertData; FInsertLen := InsertLen; FMetadata := Metadata;
  FInsertHTL     := InsertHTL; FSkipLocal := SkipLocal;
  FRetries       := NumRetries;
  FTimeoutMS     := TimeoutMinutes * 60 * 1000;
  FBlockNum      := ABlockNum;
  FIsData        := IsDataBlock;
  FIsCheck       := IsCheckBlock;
  FInsertStyle   := AInsertStyle;
  FIsHeaderBlock := IsHeaderBlock;
  FFreeData      := FreeDataOnExit;
  FResult        := False;
  FResultKey     := '';
  FCollided      := False;
  FDebugFakeInsert := False;
  if not CreateSuspended then Resume;
end;

procedure TInsertThread.Execute;
var
  sHead:  String;
  fcp:    TFCPSession;
  sMsg:   String;
  iTry:   Integer;
  cipher: String;
  t0,t:   Int64;
  bTimedOut: Boolean;
  sStat,sTry: String;
begin
  FResult := False; FResultKey := ''; FDone := False; iTry := 0; FFailReason := ftfrUnknown;
  try
    try
      while not (FResult or Terminated) do begin // until we succeed
        inc(iTry); FFailReason := ftfrUnknown;
        if iTry > 1 then sTry := Format(' (try %d)',[iTry]) else sTry := '';
        if (FInsertLen <> 0) and (FMetadata <> '') then
          DoStatus(Format('Inserting %d bytes data and %d bytes metadata%s, HTL=%d',[FInsertLen,Length(FMetadata),sTry,FInsertHTL]))
        else if (FInsertLen = 0) and (FMetadata <> '') then
          DoStatus(Format('Inserting %d bytes metadata%s, HTL=%d',[Length(FMetadata),sTry,FInsertHTL]))
        else
          DoStatus(Format('Inserting %d bytes%s, HTL=%d',[FInsertLen,sTry,FInsertHTL]));

        if FIsHeaderBlock then cipher := FInsertStyle.Header_Cipher else cipher := FInsertStyle.Blocks_Cipher;

        FStats.BytesUp := Length(FMetadata) + FInsertLen;

        if FDebugFakeInsert then begin
          FResultKey := Util_GenerateCHK(FInsertData, FInsertLen, FMetadata, nil, cipher);
          FResult    := True;
          DoStatus('Fake inserted as ' + FResultKey, LOGLVL_DEBUG);
        end else begin
          if FInsertStyle.MimeResolveMode <> mrmFCP then begin
            sHead := 'ClientPut'#10
                   + 'HopsToLive=' + Hex(FInsertHTL) + #10
                   + 'URI=CHK@'#10
                   + 'Cipher=' + cipher + #10;
            if FSkipLocal then sHead := sHead + 'RemoveLocalKey=true'#10;
            sHead := sHead
                   + 'DataLength=' + Hex(FInsertLen + Length(FMetadata)) + #10;
            if FMetadata <> '' then sHead := sHead + 'MetadataLength=' + Hex(Length(FMetadata)) + #10;
            sHead := sHead
                   + 'Data'#10
                   + FMetadata;
          end else
            sHead := '';

          fcp := nil;
          try
            fcp := NewFCP;

            t0 := GetTickCount; bTimedOut := False;

            FreenetThreadCritSec.Enter; // don't want more than 1 thread sending large amounts of data to FCP at the same time
            try
              try
                if FInsertStyle.MimeResolveMode <> mrmFCP then begin
                  fcp.SendStringCommand(sHead);                // Send header
                  fcp.SendData(FInsertData, FInsertLen, True); // and data
                end else begin
                  // 00000002 header is optional in FCP command file
                  if (FInsertLen > 4) and CompareMem(FInsertData, @(FCP_COMMAND_HEADER[0]), 4) then
                    fcp.SendData(FInsertData, FInsertLen, False)
                  else begin
                    fcp.SendData(@(FCP_COMMAND_HEADER[0]), 4, False);
                    fcp.SendData(FInsertData, FInsertLen, True);
                  end;
                end;
              except
                FFailReason := ftfrCutoff;
                raise;
              end;
            finally
              FreenetThreadCritSec.Leave;
            end;

            // wait for reply
            repeat
              while (not fcp.WaitForDataAvailable(1000)) and (not Terminated) and (not bTimedOut) do begin
                t := GetTickCount - t0; if t < 0 then inc(t, $100000000);
                if (t > FTimeoutMS) and (FTimeoutMS > 0) then bTimedOut := True;
              end;
              if Terminated or bTimedOut then break; // break repeat loop

              sMsg := fcp.ReadNextMessage;
              if sMsg = '' then begin FFailReason := ftfrCutoff; break; end;

              DoStatus(fcp.MsgAsString, LOGLVL_DEBUG);

              t0 := GetTickCount; // reset timeout

              if (sMsg = 'Success')
              or (sMsg = 'KeyCollision') then begin
                FResult    := True;
                FResultKey := fcp.Msg.Values['URI'];
                FCollided  := (sMsg = 'KeyCollision');
              end else if sMsg = 'RouteNotFound' then begin
                FFailReason := ftfrRNF;
                break;
              end;
            until FResult or Terminated or bTimedOut;

            if FResult then begin
              FinishStats(FStats);
              sStat := Format('Successfully inserted: %s (HTL %d)', [FResultKey, FInsertHTL]);
            end else if Terminated and FUnnecessary then begin
              FFailReason := ftfrAborted;
              sStat := 'Insert thread aborted.';
            end else if Terminated then
              sStat := 'Insert thread failed (terminated).'
            else if bTimedOut then begin
              FFailReason := ftfrTimeout;
              sStat := 'Insert thread timed out.';
            end else if FFailReason = ftfrRNF then
              sStat := 'RNF - Insert thread failed.'
            else
              sStat := Format('Insert thread failed. (%d)',[Ord(FFailReason)]);

            if (not FResult) and (not Terminated) then begin
              if (iTry > FRetries) and (FRetries >= 0) then
                Terminate
              else
                sStat := sStat + ' Retrying...';
            end;
            DoStatus(sStat);

          finally
            fcp.Free;
          end;
        end;
      end;

    finally
      if FFreeData then begin FreeMem(FInsertData); FInsertData := nil; end;
    end;
  except
    on E: Exception do begin
      FResult := False;
      DoStatus('Fatal error in insert thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
    end;
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;

{ TDownloadThread }

constructor TDownloadThread.Create(ThreadID: String; FCPAddr: u_long;
  FCPPort: Word; DownloadKey: String; DownloadHTL: Integer; SkipLocal: Boolean;
  BlockNum: Integer; IsDataBlock, IsCheckBlock: Boolean; OnStatus: TTextMsgEvent;
  TimeoutMinutes: Integer; AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FDownloadKey := DownloadKey;
  FDownloadHTL := DownloadHTL;
  FSkipLocal   := SkipLocal;
  FTimeoutMS   := TimeoutMinutes * 60 * 1000;
  FBlockNum  := BlockNum;
  FIsData    := IsDataBlock;
  FIsCheck   := IsCheckBlock;
  FResult    := False;
  FResData    := nil;
  FResDataLen := 0;
  FResMetadata:= '';
  if not CreateSuspended then Resume;
end;

destructor TDownloadThread.Destroy;
begin
  if FResData <> nil then FreeMem(FResData);
  inherited;
end;

procedure TDownloadThread.Execute;
var
  sCmd:   String;
  fcp,fcpCheck: TFCPSession;
  sMsg:   String;
  p,pBuf: PChar;
  c:      Char;
  TotalLen,MetaLen,PureDataLen,ChunkLen,ChunkLeft,BytesLeft,nbytes: Integer;
  sKey1,sKey2,sKey3: String;
  ok:                Boolean;
  t0,t:      Int64;
  bTimedOut: Boolean;
begin
  FResult := False; FResDataLen := 0; FResData := nil; FResMetadata := ''; FDone := False; FFailReason := ftfrUnknown;
  fcp := nil; pBuf := nil;
  try
    try
      DoStatus(Format('Download thread started with HTL %d for %s',[FDownloadHTL,FDownloadKey]));

      sCmd := 'ClientGet'#10
            + 'URI=' + FDownloadKey + #10
            + 'HopsToLive=' + Hex(FDownloadHTL) + #10;
      if FSkipLocal then sCmd := sCmd + 'RemoveLocalKey=true'#10;
      sCmd := sCmd
            + 'EndMessage'#10;

      fcp := NewFCP(gWriteFCPDebugLogOnErrors);

      t0 := GetTickCount; bTimedOut := False;

      try
        fcp.SendStringCommand(sCmd);                 // Send request
      except
        FFailReason := ftfrCutoff;
        raise;
      end;

      p := nil; BytesLeft := 0; ChunkLen := 0; MetaLen := 0; PureDataLen := 0;
      // Waiting state
      repeat
        while not fcp.WaitForDataAvailable(1000) and (not Terminated) and (not bTimedOut) do begin
          t := GetTickCount - t0; if t < 0 then inc(t, $100000000);
          if (t > FTimeoutMS) and (FTimeoutMS > 0) then bTimedOut := True;
        end;
        if Terminated or bTimedOut then break; // break repeat loop

        sMsg := fcp.ReadNextMessage;
        if sMsg = '' then begin FFailReason := ftfrCutoff; break; end;

        t0 := GetTickCount; // reset timeout

        DoStatus(fcp.MsgAsString, LOGLVL_DEBUG);

        if sMsg = 'DataFound' then begin
          try
            TotalLen := StrToInt('$'+fcp.Msg.Values['DataLength']);
            if fcp.Msg.IndexOfName('MetadataLength') >= 0 then
              MetaLen  := StrToInt('$'+fcp.Msg.Values['MetadataLength'])
            else
              MetaLen  := 0;
            PureDataLen := TotalLen - MetaLen;
            GetMem(pBuf, TotalLen);
            p := pBuf; BytesLeft := TotalLen;
          except
            DoStatus('Invalid data found message', LOGLVL_IMPORTANT);
            FFailReason := ftfrInvalid;
            Terminate; break;
          end;

          // Receiving state
          repeat
            while not fcp.WaitForDataAvailable(1000) and (not Terminated) and (not bTimedOut) do begin
              t := GetTickCount - t0; if t < 0 then inc(t, $100000000);
              if (t > FTimeoutMS) and (FTimeoutMS > 0) then bTimedOut := True;
            end;
            if Terminated or bTimedOut then break; // break repeat loop
            sMsg := fcp.ReadNextLine;   // only read 1st line
            if sMsg = '' then begin FFailReason := ftfrCutoff; Terminate; break; end;

            t0 := GetTickCount; // reset timeout

            if sMsg = 'Restarted' then begin
              // Restarted puts us back to waiting state - forget everything so far received
              if pBuf <> nil then FreeMem(pBuf);
              pBuf := nil;
              break; // break out of receiving state
            end;

            if sMsg <> 'DataChunk' then begin
              DoStatus('Unexpected message received: ' + sMsg, LOGLVL_IMPORTANT);
              FFailReason := ftfrInvalid;
              Terminate; break;
            end;

            // read rest of DataChunk message
            while not fcp.WaitForDataAvailable(1000) and (not Terminated) and (not bTimedOut) do begin
              t := GetTickCount - t0; if t < 0 then inc(t, $100000000);
              if (t > FTimeoutMS) and (FTimeoutMS > 0) then bTimedOut := True;
            end;
            if Terminated or bTimedOut then break; // break repeat loop
            t0 := GetTickCount; // reset timeout
            fcp.ReadNextMessage('Data');
            try
              ChunkLen := StrToInt('$'+fcp.Msg.Values['Length']);
            except
              DoStatus('Invalid data chunk message', LOGLVL_IMPORTANT);
              FFailReason := ftfrInvalid;
              Terminate; break;
            end;

            // read data
            ChunkLeft := ChunkLen;
            Assert(pBuf <> nil);
            Assert(ChunkLeft <= BytesLeft);
            while ChunkLeft > 0 do begin
              fcp.ReadData(p, ChunkLeft, nbytes);
              if nbytes = 0 then begin FFailReason := ftfrCutoff; Terminate; end;
              inc(p, nbytes); dec(ChunkLeft, nbytes); dec(BytesLeft, nbytes);
              inc(FStats.BytesDn, nbytes);
            end;

            // are we done?
            if BytesLeft = 0 then begin
              if MetaLen <> 0 then begin
                p := pBuf + MetaLen;
                c := p^; p^ := #0; FResMetadata := pBuf; p^ := c;
                Move(p^, pBuf^, PureDataLen);
                ReallocMem(pBuf, PureDataLen);
              end;

              {
              if GlobalSettings.VerifyCHK then begin
                // not with metadata
                if FResMetadata = '' then begin
                  sKey1 := Util_NormKey(FDownloadKey);
                  if Copy(sKey1,1,4) = 'CHK@' then begin
                    fcpCheck := NewFCP;
                    try
                      sKey2 := Util_NormKey(Util_GenerateCHK(pBuf, PureDataLen, FResMetadata, fcpCheck));
                      if sKey1 <> sKey2 then raise Exception.Create('WARNING: Paranoid check: CHK of downloaded key does not match requested key: '+sKey1+' vs. '+sKey2);
                    finally
                      fcpCheck.Free;
                    end;
                  end;
                end;
              end;
              }
              if GlobalSettings.VerifyCHK then begin
                sKey1 := Util_NormKey(FDownloadKey);
                if Copy(sKey1,1,4) = 'CHK@' then begin
                  fcpCheck := NewFCP;
                  try
                    sKey2 := Util_NormKey(Util_GenerateCHK(pBuf, PureDataLen, FResMetadata, fcpCheck, CIPHERNAME_TWOFISH));
                    ok := (sKey1 = sKey2);
                    //if ok then DoStatus('Twofish verification passed') else DoStatus('Twofish verification failed');
                  finally
                    fcpCheck.Free;
                  end;
                  if not ok then begin
                    fcpCheck := NewFCP;
                    try
                      sKey3 := Util_NormKey(Util_GenerateCHK(pBuf, PureDataLen, FResMetadata, fcpCheck, CIPHERNAME_RIJNDAEL));
                      ok := (sKey1 = sKey3);
                      //if ok then DoStatus('Rijndael verification passed') else DoStatus('Rijndael verification failed');
                    finally
                      fcpCheck.Free;
                    end;
                    if not ok then begin
                      FFailReason := ftfrBadData;
                      raise Exception.Create('WARNING: Paranoid check: CHK of downloaded key does not match requested key: '+sKey1+' vs. '+sKey2+' or '+sKey3);
                    end;
                  end;
                end;
              end;

              FResDataLen := PureDataLen;
              FResData    := pBuf;
              FResult     := True;
            end;

          until FResult or Terminated or bTimedOut;

        end else if sMsg <> 'Restarted' then begin
          if sMsg = 'DataNotFound' then FFailReason := ftfrDNF
          else if sMsg = 'RouteNotFound' then FFailReason := ftfrRNF
          else FFailReason := ftfrUnknown;
          Terminate;
        end;

      until FResult or Terminated or bTimedOut;

      if FResult then begin
        FinishStats(FStats);
        if FResMetadata = '' then
          DoStatus(Format('Successfully retrieved %d bytes (HTL %d)',[FResDataLen, FDownloadHTL]))
        else if FResDataLen = 0 then
          DoStatus(Format('Successfully retrieved %d bytes metadata (HTL %d)',[Length(FResMetadata), FDownloadHTL]))
        else
          DoStatus(Format('Successfully retrieved %d bytes data and %d bytes metadata (HTL %d)',[FResDataLen,Length(FResMetadata), FDownloadHTL]));
      end else begin
        if bTimedOut then begin
          FFailReason := ftfrTimeout;
          DoStatus('Download thread timed out.');
        end else if Terminated and FUnnecessary then begin
          FFailReason := ftfrAborted;
          DoStatus('Download thread aborted.');
        end else if FFailReason = ftfrDNF then
          DoStatus('DNF - Download thread failed.')
        else if FFailReason = ftfrRNF then
          DoStatus('RNF - Download thread failed.')
        else begin
          DoStatus(Format('Download thread failed. (%d)',[Ord(FFailReason)]));
          if gWriteFCPDebugLogOnErrors and Assigned(fcp) then fcp.WriteDebugBuffer;
        end;
      end;

    except
      on E: Exception do begin
        FResult := False;
        DoStatus('Fatal error in download thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
        if gWriteFCPDebugLogOnErrors and Assigned(fcp) then fcp.WriteDebugBuffer;
      end;
    end;
  finally
    fcp.Free;
    if (not FResult) and (pBuf <> nil) then FreeMem(pBuf);
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;

{ TCheckUpdateThread }


constructor TCheckUpdateThread.Create(ThreadID: String; FCPAddr: u_long;
  FCPPort: Word; AUpdateKey: String; DownloadHTL: Integer;
  SkipLocal: Boolean; OnStatus: TTextMsgEvent; AUserData: Pointer; AUserID: Int64;
  CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FDownloadKey := AUpdateKey;
  FDownloadHTL := DownloadHTL;
  FSkipLocal   := SkipLocal;
  FResult    := False;
  if not CreateSuspended then Resume;
end;

procedure TCheckUpdateThread.Execute;
var
  sCmd: String;
  fcp:  TFCPSession;
  sMsg: String;
begin
  FResult := False; FFound := False; FDone := False;
  fcp := nil;
  try
    try
      DoStatus(Format('CheckUpdate thread started with HTL %d for %s',[FDownloadHTL,FDownloadKey]));
      sCmd := 'ClientGet'#10
            + 'URI=' + FDownloadKey + #10
            + 'HopsToLive=' + Hex(FDownloadHTL) + #10;
      if FSkipLocal then sCmd := sCmd + 'RemoveLocalKey=true'#10;
      sCmd := sCmd
            + 'EndMessage'#10;

      fcp := NewFCP;

      fcp.SendStringCommand(sCmd);                 // Send request

      // Waiting state
      repeat
        while not fcp.WaitForDataAvailable(1000) do
          if Terminated then break; // break while loop
        if Terminated then break; // break repeat loop

        sMsg := fcp.ReadNextMessage;
        if sMsg = '' then break;

        DoStatus(fcp.MsgAsString, LOGLVL_DEBUG);

        if sMsg = 'DataFound' then begin
          FResult := True; FFound := True;
        end else if sMsg = 'DataNotFound' then begin
          FResult := True; FFound := False;
        end else if sMsg <> 'Restarted' then
          Terminate;
      until FResult or Terminated;

      if FResult then begin
        if FFound then
          DoStatus('Update found')
        else
          DoStatus('Update not found')
      end else begin
        if Terminated and FUnnecessary then
          DoStatus('CheckUpdate thread aborted.')
        else
          DoStatus('CheckUpdate thread failed.');
      end;

    finally
      fcp.Free;
    end;
  except
    on E: Exception do begin
      FResult := False;
      DoStatus('Fatal error in CheckUpdate thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
    end;
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;


{ TFECDecodeThread }

constructor TFECDecodeThread.Create(ThreadID: String; FCPAddr: u_long;
  FCPPort: Word; PrepFileBasename,SaveOutputAs: String;
  GenHealblocksPercent: Integer; PartialSave: TPartialSave; OnStatus: TTextMsgEvent;
  AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FPrepBasename := PrepFileBasename;
  FSaveAs       := SaveOutputAs;
  FGenHealPerc  := GenHealblocksPercent;
  FPartialSave  := PartialSave;
  FResult    := False;
  if not CreateSuspended then Resume;
end;

procedure TFECDecodeThread.Execute;

  function MakeListString(Lst: TList): String;
  var i: Integer;
  begin
    Result := '';
    for i := 0 to Lst.Count-1 do
      if i = 0 then
        Result := Result + Hex(Integer(Lst.Items[i]))
      else
        Result := Result + ',' + Hex(Integer(Lst.Items[i]));
  end;

var
  sCmd:   String;
  fcp,fcpCheck: TFCPSession;
  p,pBuf,pData: PChar;
  nbytes: Integer;
  PrepFile: TPreparedFile;
  i,iSeg,iBlock,iBlockInSeg,FirstBlockInSeg: Integer;
  SegInfo: TFECSegmentInfo;
  BlockList,CheckList,RequestList,HealList: TList;
  DataLen,ChunkLen,BytesLeft,BlockLeft,TotalLen: DWord;
  sMeta:   String;
  bDidMetaWarn: Boolean;
  iDecCnt:      Integer;
  sKey1,sKey2:  String;
  HealBlocksWanted: Integer;
  bMissData:        Boolean;
  AvailableList,OffsetList: TList;
  Len: Integer;
  FecDec: TFECDecoder;
  DoNativeFEC: Boolean;
begin
  FResult := False; FDone := False; FHealBlocks := '';
  fcp := nil; pBuf := nil; PrepFile := nil; pData := nil;
  BlockList := nil; CheckList := nil; RequestList := nil; HealList := nil;
  AvailableList := nil; OffsetList := nil;
  try
    try
      DoStatus('Decode thread started');

      PrepFile := TPreparedFile.CreateDownload(FPrepBasename);
      PrepFile.ReadHeader;

      BlockList   := TList.Create;
      CheckList   := TList.Create;
      RequestList := TList.Create;
      HealList    := TList.Create; // contains absolute block numbers

      AvailableList := TList.Create;
      OffsetList    := TList.Create;

      bDidMetaWarn := False;

      FirstBlockInSeg := 0; iDecCnt := 0;
      for iSeg := 0 to PrepFile.NumSegments-1 do begin
        // build BlockList,CheckList,RequestList
        Util_GetFECSegmentInfo(PrepFile.SegHeader[iSeg], SegInfo);
        BlockList.Clear; CheckList.Clear; RequestList.Clear; // do not clear HealList!

        // Hmm.. should we only use Checkblocks for healing? For now: use data blocks too
        HealBlocksWanted := Ceil(FGenHealPerc*(SegInfo.BlockCount+SegInfo.CheckBlockCount)/100);
        if HealBlocksWanted < 0 then HealBlocksWanted := 0;
        if HealBlocksWanted > SegInfo.BlockCount+SegInfo.CheckBlockCount then
          HealBlocksWanted := SegInfo.BlockCount+SegInfo.CheckBlockCount;

        iBlock := FirstBlockInSeg; bMissData := False;
        for iBlockInSeg := 0 to SegInfo.BlockCount-1 do begin
          if PrepFile.BlockDone[iBlock] then
            BlockList.Add(Pointer(iBlockInSeg))
          else begin
            bMissData := True;
            RequestList.Add(Pointer(iBlockInSeg));
            if HealBlocksWanted > 0 then begin
              HealList.Add(Pointer(iBlock)); // absolute block number
              dec(HealBlocksWanted);
            end;
          end;
          inc(iBlock);
        end;
        for iBlockInSeg := SegInfo.BlockCount to SegInfo.BlockCount+SegInfo.CheckBlockCount-1 do begin
          if PrepFile.BlockDone[iBlock] then
            CheckList.Add(Pointer(iBlockInSeg))
          else if HealBlocksWanted > 0 then begin
            // request decoding of this checkblock for healing
            HealList.Add(Pointer(iBlock)); // absolute block number
            RequestList.Add(Pointer(iBlockInSeg)); // relative block number
            dec(HealBlocksWanted);
          end;
          inc(iBlock);
        end;

        // does it matter if we send more check blocks than necessary?
        // not for fred-fec, but for the builtin implementation!
        while (BlockList.Count + CheckList.Count) > SegInfo.BlocksRequired do
          if CheckList.Count > 0 then
            CheckList.Delete(CheckList.Count-1)
          else
            BlockList.Delete(BlockList.Count-1);

        if SegInfo.BlocksRequired > (BlockList.Count + CheckList.Count) then begin
          if FPartialSave <> psCompleteOnly then begin
            // partial decode requested, but we don't have enough blocks for this seg
            FPartialOnly := True;
            DoStatus(Format('Not enough blocks to decode segment %d', [iSeg+1]));
            RequestList.Clear; HealList.Clear;
          end else
            raise Exception.Create('Not enough blocks to decode');
        end;

        if (CheckList.Count = 0) or (not bMissData) then begin
          // cannot decode from datablocks only
          // neither can we decode if we have all datablocks
          RequestList.Clear; HealList.Clear;
        end;

        if RequestList.Count > 0 then begin
          DoStatus(Format('Decoding %d blocks for segment %d/%d ...',[RequestList.Count,iSeg+1,PrepFile.NumSegments]));
          DoNativeFEC := USE_NATIVE_FEC and IsNativeFECPossible(SegInfo.FECAlgorithm);

          if DoNativeFEC then begin

            Assert(pBuf = nil);
            FecDec := nil;
            try
              FecDec := CreateFECDecoderFromName(SegInfo.FECAlgorithm);
              if FecDec = nil then raise Exception.Create('Could not create native FEC decoder');

              FecDec.Init(SegInfo.FileLength);

              Len := SegInfo.BlockSize;
              if SegInfo.CheckBlockSize > Len then Len := SegInfo.CheckBlockSize;
              GetMem(pBuf, Len); FillChar(pBuf^, Len, 0);

              PrepFile.RememberState;
              try
                AvailableList.Clear;
                for i := 0 to BlockList.Count-1 do AvailableList.Add(BlockList.Items[i]);
                for i := 0 to CheckList.Count-1 do AvailableList.Add(CheckList.Items[i]);
                OffsetList.Clear;
                for i := 0 to SegInfo.BlockCount-1 do begin
                  iBlock := FirstBlockInSeg + i;
                  if PrepFile.BlockDone[iBlock] then
                    OffsetList.Add( Pointer(PrepFile.RawDataOffset[iBlock]) )
                  else
                    // write dummy data
                    OffsetList.Add( Pointer(PrepFile.WriteData(iBlock, pBuf, SegInfo.BlockSize, '')) );
                end;
                for i := SegInfo.BlockCount to SegInfo.BlockCount+SegInfo.CheckBlockCount-1 do begin
                  iBlock := FirstBlockInSeg + i;
                  if CheckList.IndexOf(Pointer(i)) >= 0 then
                    OffsetList.Add( Pointer(PrepFile.RawDataOffset[iBlock]) )
                  else
                    OffsetList.Add( nil ); // don't need offsets for unused checkblocks
                end;

                FecDec.Decode(iSeg, PrepFile.DataFileOutputStream, AvailableList, OffsetList);
                for i := 0 to SegInfo.BlockCount-1 do begin
                  iBlock := FirstBlockInSeg + i;
                  if not PrepFile.BlockDone[iBlock] then begin
                    PrepFile.BlockDone[iBlock] := True;
                    inc(iDecCnt);
                  end;
                end;
                PrepFile.WriteHeader;   // CAUTION: possible race cond.
                PrepFile.CloseDataFile;

                // we don't generate check-heal blocks here;
                // would require to reencode the whole file and it's not necessary
                // - we just pick some of the newly decoded datablocks
                HealList.Clear;

              except
                PrepFile.RevertToRememberedState;
                raise;
              end;
            finally
              if pBuf <> nil then FreeMem(pBuf); pBuf := nil;
              FecDec.Free;
            end;

          end else begin

            // do FEC via FCP
            Assert(fcp = nil);
            fcp := NewFCP; // create new fcp session for each segment!
            try
              // send FCP command
              sCmd := 'FECDecodeSegment'#10
                    + 'BlockList='     + MakeListString(BlockList)   + #10
                    + 'CheckList='     + MakeListString(CheckList)   + #10
                    + 'RequestedList=' + MakeListString(RequestList) + #10
                    + 'DataLength=' + Hex(Length(SegInfo.FullSegmentHeader) + SegInfo.BlockSize * BlockList.Count + SegInfo.CheckBlockSize * CheckList.Count) + #10
                    + 'Data'#10
                    + SegInfo.FullSegmentHeader
                    ;
              fcp.SendStringCommand(sCmd);

              // send data and check blocks
              TotalLen := 0;
              for i := 0 to BlockList.Count+CheckList.Count-1 do begin
                if i < BlockList.Count then
                  iBlock := FirstBlockInSeg + Integer(BlockList.Items[i])
                else
                  iBlock := FirstBlockInSeg + Integer(CheckList.Items[i-BlockList.Count]);

                if Terminated then break;
                PrepFile.GetData(iBlock, pData, DataLen, sMeta);
                if Terminated then break;
                try
                  if (sMeta <> '') and (not bDidMetaWarn) then begin
                    DoStatus('WARNING: Split file contains metadata - ignored');
                    bDidMetaWarn := True;
                  end;
                  Assert(DataLen > 0, 'Data length = 0');

                  // send data in FCP_CHUNKSIZE blocks
                  FreenetThreadCritSec.Enter; // don't want more than 1 thread sending large amounts of data to FCP at the same time
                  try
                    p := pData; BytesLeft := DataLen;
                    while BytesLeft > 0 do begin
                      if Terminated then break;
                      ChunkLen := FCP_CHUNKSIZE; if ChunkLen > BytesLeft then ChunkLen := BytesLeft;
                      fcp.SendData(p, ChunkLen, True);
                      inc(p,ChunkLen); inc(TotalLen,ChunkLen); dec(BytesLeft,ChunkLen);
                    end;
                  finally
                    FreenetThreadCritSec.Leave;
                  end;
                  if Terminated then break;
                finally
                  if pData <> nil then FreeMem(pData); pData := nil;
                end;
              end;
              if Terminated then break;

              Assert(TotalLen = DWord(SegInfo.BlockSize * BlockList.Count + SegInfo.CheckBlockSize * CheckList.Count), 'block size mismatch');

              // all data has been sent, now wait for the decoded blocks
              while not fcp.WaitForDataAvailable(1000) do
                if Terminated then break; // break while loop
              if Terminated then break; // break for loop
              if fcp.ReadNextMessage <> 'BlocksDecoded' then raise Exception.Create('Unexpected message: ' + fcp.MsgAsString);
              Assert( StrToInt('$' + fcp.Msg.Values['BlockCount']) = RequestList.Count, 'Returned block count differs from requested');
              Assert( StrToInt('$' + fcp.Msg.Values['BlockSize'])  = SegInfo.BlockSize, 'Returned block size differs from expected');

              Assert(pBuf = nil);
              GetMem(pBuf,SegInfo.BlockSize);
              ChunkLen := 0;
              for i := 0 to RequestList.Count-1 do begin
                p := pBuf; BlockLeft := SegInfo.BlockSize;
                while BlockLeft > 0 do begin
                  if ChunkLen = 0 then begin
                    while not fcp.WaitForDataAvailable(1000) do
                      if Terminated then break; // break while loop
                    if Terminated then break; // break while BlockLeft loop
                    if fcp.ReadNextMessage('Data') <> 'DataChunk' then raise Exception.Create('Unexpected message: ' + fcp.MsgAsString);
                    ChunkLen := StrToInt('$' + fcp.Msg.Values['Length']);
                  end;
                  nbytes := ChunkLen; if nbytes > Integer(BlockLeft) then nbytes := BlockLeft;
                  fcp.ReadData(p, nbytes, nbytes);
                  if nbytes = 0 then raise Exception.Create('Connection died');
                  inc(p,nbytes); dec(BlockLeft,nbytes); dec(ChunkLen,nbytes);
                end;
                if Terminated then break;
                // got complete block. save it to prepfile
                iBlock := FirstBlockInSeg + Integer(RequestList.Items[i]);

                {
                if VERIFY_CHK_KEYS then begin
                  fcpCheck := TFCPSession.Create(FFCPAddr, FFCPPort);
                  try
                    sKey1 := Util_NormKey(PrepFile.Key[iBlock]);
                    sKey2 := Util_NormKey(Util_GenerateCHK(pBuf, SegInfo.BlockSize, '', fcpCheck));
                    if sKey1 <> sKey2 then raise Exception.Create('WARNING: Paranoid check: CHK of decoded key does not match requested key: '+sKey1+' vs. '+sKey2);
                  finally
                    fcpCheck.Free;
                  end;
                end;
                }

                PrepFile.WriteData(iBlock, pBuf, SegInfo.BlockSize, '');
                PrepFile.BlockDone[iBlock] := True;
                if HealList.IndexOf(Pointer(iBlock)) >= 0 then PrepFile.BlockTries[iBlock] := 0; // mark healblocks as tries=0
                PrepFile.WriteHeader; // CAUTION - possible race cond.
                inc(iDecCnt);
                if PrepFile.IsDataBlock[iBlock] then
                  DoStatus(Format('Reconstructed block %d', [iBlock]))
                else
                  DoStatus(Format('Reconstructed check-block %d', [iBlock]));
              end;
            finally
              PrepFile.CloseDataFile;
              fcp.Free; fcp := nil;
              if pBuf <> nil then FreeMem(pBuf); pBuf := nil;
            end;

          end;
        end;

        FirstBlockInSeg := FirstBlockInSeg + SegInfo.BlockCount + SegInfo.CheckBlockCount;
        if Terminated then break;
      end;

      if not Terminated then begin
        // generate Healblocklist-String
        for i := 0 to HealList.Count-1 do
          FHealBlocks := FHealBlocks + ',' + Hex(Integer(HealList.Items[i]));
        if FHealBlocks <> '' then Delete(FHealBlocks,1,1); // remove first ','
      end;

      FResult := not Terminated;
      if FResult then begin
        DoStatus(Format('Reconstructed %d blocks successfully.',[iDecCnt]));
        if FHealBlocks <> '' then DoStatus('Generated heal blocks: ' + FHealBlocks);

        if FSaveAs <> '' then begin // save output file
          DoStatus('Saving output file');
          PrepFile.SaveOutputFile(FSaveAs, CheckAbort, FPartialSave);
        end;
      end else
        DoStatus('Decode thread failed');

    finally
      fcp.Free;
      PrepFile.Free;
      BlockList.Free;
      CheckList.Free;
      RequestList.Free;
      HealList.Free;
      AvailableList.Free;
      OffsetList.Free;
      if pBuf  <> nil then FreeMem(pBuf);
      if pData <> nil then FreeMem(pData);
    end;
  except
    on E: Exception do begin
      FResult := False;
      DoStatus('Fatal error in decode thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
    end;
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;

{ TChecksumThread }

constructor TChecksumThread.Create(ThreadID: String; FCPAddr: u_long;
  FCPPort: Word; Filename: String;
  OnStatus: TTextMsgEvent; AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FFilename := Filename;
  FResult   := False;
  FChecksum := '';
  FDontCalc := False;
  if not CreateSuspended then Resume;
end;

procedure TChecksumThread.Execute;
var
  fcp:    TFCPSession;
  pDataChunk: PChar;
  DataChunkLen,BytesLeft,nbytes: Integer;
  Str:    TFileStream;
{$IFDEF USE_NATIVE_SHA1}
  sha:    TSHA1Context;
  Digest: TSHA1Digest;
{$ENDIF}
begin
  if FDontCalc then begin // if used as dummy thread, immediately return
    DoStatus('Checksum thread: dummy executing');
    FResult   := True;
    FChecksum := '';
    Terminate;
    FDone := True;
    exit;
  end;

  FResult := False; FDone := False;
  fcp := nil; pDataChunk := nil; Str := nil;
  try
    try
      DoStatus('Checksum thread started');

      Str := CreateFileStream(FFilename, fmOpenRead or fmShareDenyWrite);

      {$IFDEF USE_NATIVE_SHA1}
        SHA1Reset(sha);
      {$ELSE}
        fcp := NewFCP;
      {$ENDIF}

      DataChunkLen := FCP_CHUNKSIZE;
      GetMem(pDataChunk, DataChunkLen);

      {$IFNDEF USE_NATIVE_SHA1} FreenetThreadCritSec.Enter; {$ENDIF} // don't want more than 1 thread sending large amounts of data to FCP at the same time
      try
        {$IFNDEF USE_NATIVE_SHA1}
          fcp.SendStringCommand( 'GenerateSHA1'#10
                               + 'DataLength=' + Hex(Str.Size) + #10
                               + 'Data'#10
                               );
        {$ENDIF}
        BytesLeft := Str.Size;
        while BytesLeft > 0 do begin
          if BytesLeft > DataChunkLen then nbytes := DataChunkLen else nbytes := BytesLeft;
          Str.ReadBuffer(pDataChunk^, nbytes);
          {$IFDEF USE_NATIVE_SHA1}
            SHA1Input(sha, pDataChunk, nbytes);
          {$ELSE}
            fcp.SendData(pDataChunk, nbytes, True);
          {$ENDIF}
          dec(BytesLeft,nbytes);
          if Terminated then break;
        end;
      finally
        {$IFNDEF USE_NATIVE_SHA1} FreenetThreadCritSec.Leave; {$ENDIF}
      end;

      {$IFDEF USE_NATIVE_SHA1}
        if not Terminated then begin
          SHA1Result(sha, Digest);
          FChecksum := SHA1DigestToLoHexString(Digest);
        end;
      {$ELSE}
        if not Terminated then
          while not fcp.WaitForDataAvailable(1000) do
            if Terminated then break; // break while loop

        if not Terminated then begin
          if fcp.ReadNextMessage = 'Success' then begin
            FChecksum := fcp.Msg.Values['SHA1']
          end else
            raise Exception.Create('Unexpected message: ' + fcp.MsgAsString);
        end;
      {$ENDIF}

      FResult := (FChecksum <> '') and (not Terminated);
      if FResult then
        DoStatus('Calculated checksum succesfully.')
      else
        DoStatus('Checksum calculation failed');

    finally
      fcp.Free;
      Str.Free;
      if pDataChunk <> nil then FreeMem(pDataChunk);
    end;
  except
    on E: Exception do begin
      FResult := False;
      DoStatus('Fatal error in checksum thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
    end;
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;

{ TPrepareThread }

constructor TPrepareThread.Create(ThreadID: String; FCPAddr: u_long;
  FCPPort: Word; Filename, APrepBasename: String; MakeChecksum, PrecalcMeta, UseOrgFile: Boolean;
  AInsertStyle: TInsertStyle; OnStatus: TTextMsgEvent; AUserData: Pointer; AUserID: Int64; CreateSuspended: Boolean);
begin
  inherited Create(ThreadID,FCPAddr,FCPPort,OnStatus,AUserData,AUserID,True); // suspended
  FFilename     := Filename;
  FPrepBasename := APrepBasename;
  FDoChecksum   := MakeChecksum;
  FPrecalcMeta  := PrecalcMeta;
  FUseOrgFile   := UseOrgFile;
  FInsertStyle  := AInsertStyle;
  FResult       := False;
  FResultKey    := '';
  if not CreateSuspended then Resume;
end;

procedure TPrepareThread.Execute;
// this is basically a stripped-down version of filemanager.insertfile
  function CalcAndSetKey(PrepFile: TPreparedFile; pData: PChar; DataLen: Integer; sMeta: String; iBlockNum: Integer; IsHeaderBlock: Boolean): String;
  var
    fcp:  TFCPSession;
    cipher: String;
  begin
    fcp := NewFCP;
    try
      if IsHeaderBlock then cipher := FInsertStyle.Header_Cipher else cipher := FInsertStyle.Blocks_Cipher;
      Result := Util_GenerateCHK(pData, DataLen, sMeta, fcp, cipher);
      PrepFile.Key[iBlockNum] := Result;
    finally
      fcp.Free;
    end;
  end;
var
  fcp:     TFCPSession;
  Str:     TFileStream;
  sChkSum: String;
  ChkSumThread: TChecksumThread;
  PrepFile: TPreparedFile;
  sMeta:    String;
  DataLen:  DWord;
  pBlock,pCheck: PChar;
  NumSegs:  Integer;
  SegHeaders: TDynamicStringArray;
  i,iSeg:      Integer;
  TotalBlocks: Integer;
  SegInfo:    TFECSegmentInfo;
  PrepBlockNum: Integer;
  OrgLen:       DWord;
  OrgOff:       DWord;
  FileSize:     DWord;
  OutOffsets:   TList;
{$IFDEF STREAM_CHECKBLOCKS}
  OutPos:       Cardinal;
{$ELSE}
  p: PChar;
{$ENDIF}
begin
  FResult := False; FDone := False; FResultKey := '';
  fcp := nil; Str := nil; PrepFile := nil; pBlock := nil; pCheck := nil;
  OutOffsets := nil;
  try
    try
      DoStatus('Prepare thread started');

      // open input file
      Str := CreateFileStream(FFilename, fmOpenRead or fmShareDenyWrite);
      FileSize := Str.Size;

      // create prep.file
      PrepFile := TPreparedFile.CreateInsert(FPrepBasename, FUseOrgFile);
      PrepFile.OriginalFilename := FFilename;
      PrepFile.FileLength       := FileSize;
      PrepFile.MimeType         := GetMimeType(ExtractFileExt(FFilename), FInsertStyle);

      // no FEC ?
      if (FileSize < FEC_LIMIT) or (FInsertStyle.MimeResolveMode in [mrmRaw,mrmFCP] {raw inserts, FCP command files}) then begin
        PrepFile.Init(1);
        if FInsertStyle.MimeResolveMode in [mrmFrost,mrmRaw,mrmFCP] then begin
          sMeta := '';
        end else begin
          if FInsertStyle.MimeResolveMode = mrmFIW then begin
            sMeta := 'Version'#10
                   + 'Revision=1'#10;
          end else begin
            sMeta := META_HEADER
                   + 'Document'#10
                   + 'Info.Format=' + PrepFile.MimeType + #10;
            if FInsertStyle.MimeResolveMode <> mrmFuqid then
              sMeta := sMeta + 'Info.Description=file'#10;
          end;
          sMeta := sMeta + 'End'#10;
        end;

        DataLen := FileSize;
        GetMem(pBlock, DataLen);
        Str.ReadBuffer(pBlock^, DataLen);
        PrepFile.WriteData(0, pBlock, DataLen, sMeta, FUseOrgFile, 0, DataLen);
        if FPrecalcMeta then FResultKey := CalcAndSetKey(PrepFile,pBlock,DataLen,sMeta,0,True);
        PrepFile.MainBlock := 0;
        PrepFile.WriteHeader; // ok
      end else begin
        // prepare for FEC

        // calc checksum
        sChkSum := '';
        if FDoChecksum then begin
          ChkSumThread := TChecksumThread.Create(
                            FID, // + '(ChkSum)',
                            FFCPAddr, FFCPPort, FFilename, FOnStatus, nil, 0, True
                          );
          try
            ChkSumThread.Resume;
            while not ChkSumThread.Done do begin
              if Terminated then begin ChkSumThread.Terminate; break; end;
              Sleep(200);
            end;
            ChkSumThread.WaitFor;
            if ChkSumThread.ThreadResult then sChkSum := ChkSumThread.Checksum;
          finally
            ChkSumThread.Free;
          end;
          if (not Terminated) and (sChkSum = '') then
            DoStatus('Failed to calculate checksum! Continuing anyway.');
        end;
        PrepFile.SHA1Checksum := sChkSum;

        // create segment headers and init prepfile
        CheckAbort;
        fcp := NewFCP;
        try
          NumSegs := Util_FECSegmentFile(FileSize, SegHeaders, fcp);
          TotalBlocks := 1; // header block
          for i := 0 to NumSegs-1 do begin
            Util_GetFECSegmentInfo(SegHeaders[i], SegInfo);
            inc(TotalBlocks, SegInfo.BlockCount + SegInfo.CheckBlockCount);
          end;
          PrepFile.Init(TotalBlocks, NumSegs);
          if FPrecalcMeta then begin
            PrepFile.MainBlock := 0; // header is first block
            PrepBlockNum := 1;
          end else begin
            PrepFile.MainBlock := PrepFile.NumBlocks-1; // header is last block
            PrepBlockNum := 0;
          end;
          for i := 0 to NumSegs-1 do PrepFile.SegHeader[i] := SegHeaders[i];
        finally
          fcp.Free; fcp := nil;
        end;

        // FEC encode each segment and write to prepfile
        for iSeg := 0 to NumSegs-1 do begin
          CheckAbort;

          Util_GetFECSegmentInfo(SegHeaders[iSeg], SegInfo);

          Assert(fcp = nil); Assert(pBlock = nil); Assert(pCheck = nil);
          try
            fcp := NewFCP;

            {$IFNDEF STREAM_CHECKBLOCKS}
              // FEC-encode
              DoStatus(Format('Segment %d/%d: FEC-encoding ...',[iSeg+1,NumSegs]));
              Str.Seek(SegInfo.Offset, soFromBeginning);
              pCheck := Util_FECEncode(Str, SegInfo, fcp, CheckAbort);
              DoStatus(Format('Segment %d/%d: Created %d redundant blocks',[iSeg+1,NumSegs,SegInfo.CheckBlockCount]));
              CheckAbort;
            {$ENDIF}

            // save data blocks
            DoStatus(Format('Segment %d/%d: Saving %d data blocks ...',[iSeg+1,NumSegs,SegInfo.BlockCount]));
            GetMem(pBlock, SegInfo.BlockSize);
            OrgOff := SegInfo.Offset;
            Str.Seek(OrgOff, soFromBeginning);
            for i := 0 to SegInfo.BlockCount-1 do begin
              CheckAbort;
              if FPrecalcMeta or (not FUseOrgFile) then begin
                if OrgOff + Cardinal(SegInfo.BlockSize) <= FileSize then
                  Str.ReadBuffer(pBlock^, SegInfo.BlockSize)
                else begin
                  FillChar(pBlock^, SegInfo.BlockSize, 0);
                  if OrgOff < FileSize then Str.ReadBuffer(pBlock^, FileSize - OrgOff);
                end;
              end;
              if FUseOrgFile then begin
                OrgLen := SegInfo.BlockSize;
                if OrgOff + OrgLen > FileSize then OrgLen := FileSize - OrgOff;
                PrepFile.WriteData(PrepBlockNum, nil, SegInfo.BlockSize, '', True, OrgOff, OrgLen);
              end else
                PrepFile.WriteData(PrepBlockNum, pBlock, SegInfo.BlockSize, '');
              if FPrecalcMeta then CalcAndSetKey(PrepFile, pBlock, SegInfo.BlockSize, '', PrepBlockNum,False);
              inc(PrepBlockNum);
              inc(OrgOff, SegInfo.BlockSize);
            end;

            {$IFDEF STREAM_CHECKBLOCKS}
              DoStatus(Format('Segment %d/%d: Preparing for %d check blocks ...',[iSeg+1,NumSegs,SegInfo.CheckBlockCount]));
              // fill data file with zeroes to get offsets
              if OutOffsets = nil then OutOffsets := TList.Create else OutOffsets.Clear;
              GetMem(pCheck, SegInfo.CheckBlockSize);
              FillChar(pCheck^, SegInfo.CheckBlockSize, 0);
              for i := 0 to SegInfo.CheckBlockCount-1 do begin
                OutOffsets.Add( Pointer( PrepFile.WriteData(PrepBlockNum + i, pCheck, SegInfo.CheckBlockSize, '') ) );
              end;
              // encode
              DoStatus(Format('Segment %d/%d: FEC-encoding and saving %d check blocks ...',[iSeg+1,NumSegs,SegInfo.CheckBlockCount]));
              OutPos := PrepFile.DataFileOutputStream.Position; // remember pos!
              try
                Str.Seek(SegInfo.Offset, soFromBeginning);
                Util_FECEncode_ToStream(Str, SegInfo, PrepFile.DataFileOutputStream, OutOffsets, fcp, CheckAbort);
                if FPrecalcMeta then begin
                  for i := 0 to SegInfo.CheckBlockCount-1 do begin
                    PrepFile.DataFileOutputStream.Seek(Integer(OutOffsets.Items[i]), soFromBeginning);
                    PrepFile.DataFileOutputStream.ReadBuffer(pCheck^, SegInfo.CheckBlockSize);
                    CalcAndSetKey(PrepFile, pCheck, SegInfo.CheckBlockSize, '', PrepBlockNum + i,False);
                  end;
                end;
              finally
                PrepFile.DataFileOutputStream.Seek(OutPos, soFromBeginning); // reset pos!
              end;
              inc(PrepBlockNum, SegInfo.CheckBlockCount);
            {$ELSE}
              // save check blocks
              DoStatus(Format('Segment %d/%d: Saving %d check blocks ...',[iSeg+1,NumSegs,SegInfo.CheckBlockCount]));
              p := pCheck;
              for i := 0 to SegInfo.CheckBlockCount-1 do begin
                CheckAbort;
                PrepFile.WriteData(PrepBlockNum, p, SegInfo.CheckBlockSize, '');
                if FPrecalcMeta then CalcAndSetKey(PrepFile, p, SegInfo.CheckBlockSize, '', PrepBlockNum,False);
                inc(PrepBlockNum);
                inc(p, SegInfo.CheckBlockSize);
              end;
            {$ENDIF}
            
          finally
            fcp.Free; fcp := nil;
            if pBlock <> nil then FreeMem(pBlock); pBlock := nil;
            if pCheck <> nil then FreeMem(pCheck); pCheck := nil;
          end;
        end; // next segment

        if FPrecalcMeta then begin
          // calculate and save header block (metadata) and key
          DoStatus('Calculating metadata');
          sMeta := PrepFile.CalcSplitFileMetadata(FFCPAddr,FFCPPort,FInsertStyle);
          PrepFile.WriteData(PrepFile.MainBlock, nil, 0, sMeta);
          FResultKey := CalcAndSetKey(PrepFile, nil, 0, sMeta, PrepFile.MainBlock,True);
        end else begin
          // save an empty block as header
          PrepFile.WriteData(PrepFile.MainBlock, nil, 0, '');
        end;

        // save the prepfile header
        PrepFile.WriteHeader; // ok
      end;

      FResult := (not Terminated);
      if FResult then
        DoStatus('Prepared file succesfully.')
      else
        DoStatus('Preparation failed');

    finally
      fcp.Free;
      Str.Free;
      PrepFile.Free;
      OutOffsets.Free;
      if pBlock <> nil then FreeMem(pBlock);
      if pCheck <> nil then FreeMem(pCheck);
    end;
  except
    on E: Exception do begin
      FResult := False;
      DoStatus('Fatal error in prepare thread: ' + E.ClassName + ': ' + E.Message, LOGLVL_IMPORTANT);
    end;
  end;
  Terminate; // sets Terminated=True
  FDone := True;
end;


{ TPreparedFile }

const
  PREPAREDINSFILE_MAGIC  = $DA7AF173;
  PREPAREDINSFILE2_MAGIC = $DA2AF123;
  PREPAREDGETFILE_MAGIC  = $F173DA7A;

constructor TPreparedFile.Create(AsInsertFile: Boolean; ABasename: String; UseOrigFile: Boolean);
begin
  inherited Create;
  FBasename := ABasename;
  FOffsetList := TList.Create;
  FDoneList   := TList.Create;
  FKeyList    := TStringList.Create;
  FRemembered.OffsetList := TList.Create;
  SetLength(FSegHeaders,0);
  SetLength(FBlockMaps, 0);
  FIsInsertFile := AsInsertFile;
  FUsesOrgFile := UseOrigFile;
  Assert((not FUsesOrgFile) or FIsInsertFile, 'UseOrigFile only supported for insert files!');
end;

constructor TPreparedFile.CreateInsert(ABasename: String; UseOrigFile: Boolean);
begin
  Create(True, ABasename, UseOrigFile);
end;

constructor TPreparedFile.CreateDownload(ABasename: String);
begin
  Create(False, ABasename);
end;

destructor TPreparedFile.Destroy;
begin
  CloseDataFile;
  FOffsetList.Free;
  FDoneList.Free;
  FKeyList.Free;
  FRemembered.OffsetList.Free;
  SetLength(FSegHeaders,0);
  SetLength(FBlockMaps, 0);
  inherited;
end;

procedure TPreparedFile.Init(ANumBlocks,ANumSegments: Integer);
var i: Integer;
begin
  FNumBlocks := ANumBlocks;
  FNumSegments := ANumSegments;
  FOffsetList.Clear;
  FDoneList.Clear;
  FKeyList.Clear;
  for i := 0 to FNumBlocks-1 do begin
    FOffsetList.Add(Pointer(-1));
    FDoneList.Add(nil);
    FKeyList.Add('');
  end;
  SetLength(FSegHeaders, FNumSegments);
  SetLength(FBlockMaps,  FNumSegments);
  for i := 0 to FNumSegments-1 do begin FSegHeaders[i] := ''; FBlockMaps[i] := ''; end;
end;

procedure TPreparedFile.InitHeal(ANumBlocks: Integer);
begin
  Init(ANumBlocks, 0);
  // should we set NumSegs to 1 and generate a pseudo Segmentheader?
end;

function TPreparedFile.GetHeaderFilename: String;
begin
  Assert(FBasename <> '', 'Basename not specified');
  Result := FBasename + '.dhd';
end;

function TPreparedFile.GetDataFilename: String;
begin
  Assert(FBasename <> '', 'Basename not specified');
  Result := FBasename + '.dat';
end;

procedure TPreparedFile.ReadHeader;
var
  Str:   TFileStream;
  CachedFile: TCachedFile;
  len,buflen,alen,keylen,infolen,seghdlen,blkmaplen: DWord;
  i:     Integer;
  p,pBuf,pKeys,pInfo,pSegs,pMaps: PChar;
begin
  FNumBlocks := 0;
  FOffsetList.Clear;
  FDoneList.Clear;
  FKeyList.Clear;
  FInfo.OrigFilename := '';
  FInfo.FileLength   := 0;
  FInfo.Checksum     := '';
  FInfo.MimeType     := '';
  FNumSegments       := 0;
  SetLength(FSegHeaders, 0);
  SetLength(FBlockMaps,  0);

  pBuf := nil; Str := nil;
  try
    if USE_CACHED_FILES then begin
      CachedFile := gCachedFiles.OpenFile(HeaderFilename, True);
      buflen := CachedFile.Size;
      GetMem(pBuf, buflen+1);
      CachedFile.GetContents(pBuf, buflen);
    end else begin
      Str := CreateFileStream(HeaderFilename, fmOpenRead or fmShareDenyWrite);
      buflen := Str.Size;
      GetMem(pBuf, buflen+1);
      Str.ReadBuffer(pBuf^, buflen);
    end;
    (pBuf+buflen)^ := #0;
    p := pBuf;

    alen := 3*SizeOf(DWord); Assert(buflen >= alen);
    // Magic
    if FIsInsertFile then begin
      FUsesOrgFile := (PDWord(p)^ = PREPAREDINSFILE2_MAGIC);
      Assert((PDWord(p)^ = PREPAREDINSFILE_MAGIC) or (PDWord(p)^ = PREPAREDINSFILE2_MAGIC), 'Invalid data file'); inc(p, SizeOf(DWord));
    end else begin
      FUsesOrgFile := False;
      Assert(PDWord(p)^ = PREPAREDGETFILE_MAGIC, 'Invalid data file'); inc(p, SizeOf(DWord));
    end;
    // NumBlocks
    FNumBlocks := PDWord(p)^; inc(p, SizeOf(DWord));
    // MainBlock
    FMainBlock := PDWord(p)^; inc(p, SizeOf(DWord));

    inc(alen, FNumBlocks*(SizeOf(DWord)+SizeOf(Byte)) + SizeOf(DWord)); Assert(buflen >= alen);

    // Offsets
    for i := 0 to FNumBlocks-1 do begin
      FOffsetList.Add(Pointer(PDWord(p)^)); inc(p, SizeOf(DWord));
    end;
    // DoneMap
    for i := 0 to FNumBlocks-1 do begin
      FDoneList.Add(Pointer(PByte(p)^)); inc(p, SizeOf(Byte));
    end;
    // Keys
    keylen := PDWord(p)^; inc(p, SizeOf(DWord));
    inc(alen, keylen + SizeOf(DWord)); Assert(buflen >= alen);
    pKeys := p;
    for i := 0 to FNumBlocks-1 do begin
      len := StrLen(p);
      Assert(p+len < pKeys+keylen, 'Data file corrupt');
      FKeyList.Add(p);
      inc(p,len+1)
    end;
    Assert(p = pKeys+keylen, 'Data file corrupt');

    // Info
    infolen := PDWord(p)^; inc(p, SizeOf(DWord));
    inc(alen, infolen + SizeOf(DWord)); Assert(buflen >= alen);
    pInfo := p;
    len := StrLen(p);
    Assert(p+len+SizeOf(DWord) < pInfo+infolen, 'Data file corrupt');
    FInfo.OrigFilename := p; inc(p,len+1);
    FInfo.FileLength   := PDWord(p)^; inc(p, SizeOf(DWord));

    if p < (pInfo + infolen) then begin
      len := StrLen(p); Assert(p+len < pInfo+infolen, 'Data file corrupt');
      FInfo.Checksum := p; inc(p,len+1);
    end;
    if p < (pInfo + infolen) then begin
      len := StrLen(p); Assert(p+len < pInfo+infolen, 'Data file corrupt');
      FInfo.MimeType := p; inc(p,len+1);
    end;

    Assert (p <= pInfo + infolen);
    if (p < pInfo + infolen) then p := pInfo + infolen; // skip over unknown info

    // SegHeaders
    seghdlen := PDWord(p)^; inc(p, SizeOf(DWord));
    inc(alen, seghdlen); Assert(buflen >= alen);
    pSegs := p;
    FNumSegments := PDWord(p)^; inc(p, SizeOf(DWord));
    SetLength(FSegHeaders, FNumSegments);
    for i := 0 to FNumSegments-1 do begin
      len := StrLen(p);
      Assert(p+len < pSegs+seghdlen, 'Data file corrupt');
      FSegHeaders[i] := p;
      inc(p,len+1)
    end;
    Assert(p = pSegs+seghdlen, 'Data file corrupt');

    // BlockMaps (only present for download files)
    SetLength(FBlockMaps, FNumSegments);
    if not FIsInsertFile then begin
      blkmaplen := PDWord(p)^; inc(p, SizeOf(DWord));
      inc(alen, blkmaplen); Assert(buflen >= alen);
      pMaps := p;
      for i := 0 to FNumSegments-1 do begin
        len := StrLen(p);
        Assert(p+len < pMaps+blkmaplen, 'Data file corrupt');
        FBlockMaps[i] := p;
        inc(p,len+1)
      end;
      Assert(p = pMaps+blkmaplen, 'Data file corrupt');
    end else begin
      for i := 0 to FNumSegments-1 do FBlockMaps[i] := '';
    end;

  finally
    if pBuf <> nil then FreeMem(pBuf);
    Str.Free;
    // do NOT free CachedFile
  end;
end;

procedure TPreparedFile.WriteHeader(Debug_Delay: Integer);
var
  Str: TFileStream;
  CachedFile: TCachedFile;
  len,keyslen,infolen,seghdlen,blkmaplen,blkmaptotlen: DWord;
  i:   Integer;
  p,pBuf: PChar;
begin
  pBuf := nil; Str := nil; CachedFile := nil;
  try
    if USE_CACHED_FILES then begin
      CachedFile := gCachedFiles.OpenFile(HeaderFilename, False);
    end else begin
      Str := CreateFileStream(HeaderFilename, fmCreate or fmShareExclusive);
    end;
    keyslen  := 0; for i := 0 to FNumBlocks-1 do inc(keyslen, Length(FKeyList.Strings[i])+1);
    infolen  := Length(FInfo.OrigFilename)+1 + SizeOf(DWord) + Length(FInfo.Checksum)+1 + Length(FInfo.MimeType)+1;
    seghdlen := SizeOf(DWord); for i := 0 to FNumSegments-1 do inc(seghdlen, Length(FSegHeaders[i])+1);
    blkmaplen := 0; blkmaptotlen := 0;
    if not FIsInsertFile then begin
      for i := 0 to FNumSegments-1 do inc(blkmaplen, Length(FBlockMaps[i])+1);
      blkmaptotlen := SizeOf(DWord) + blkmaplen;
    end;

    len := 3*SizeOf(DWord)                         // Magic,NumBlocks,MainBlock
         + FNumBlocks*(SizeOf(DWord)+SizeOf(Byte)) // Offsets,DoneMap
         + SizeOf(DWord) + keyslen                 // Keylen,Keymap
         + SizeOf(DWord) + infolen                 // Infolen,Info
         + SizeOf(DWord) + seghdlen                // SegHdLen,SegHeaderBlock
         + blkmaptotlen                            // (BlockMapSize+Data)
         ;
    GetMem(pBuf, len); p := pBuf;
    // Magic
    if FIsInsertFile then begin
      if FUsesOrgFile then begin
        PDword(p)^ := PREPAREDINSFILE2_MAGIC; inc(p, SizeOf(DWord));
      end else begin
        PDword(p)^ := PREPAREDINSFILE_MAGIC; inc(p, SizeOf(DWord));
      end;
    end else begin
      PDword(p)^ := PREPAREDGETFILE_MAGIC; inc(p, SizeOf(DWord));
    end;
    // NumBlocks
    PDword(p)^ := FNumBlocks; inc(p, SizeOf(DWord));
    // MainBlock
    PDword(p)^ := FMainBlock; inc(p, SizeOf(DWord));
    // Offsets
    for i := 0 to FNumBlocks-1 do begin
      PDword(p)^ := DWord(FOffsetList.Items[i]); inc(p, SizeOf(DWord));
    end;
    // DoneMap
    for i := 0 to FNumBlocks-1 do begin
      PByte(p)^ := Byte(FDoneList.Items[i]); inc(p, SizeOf(Byte));
    end;
    // Keys
    PDword(p)^ := keyslen; inc(p, SizeOf(DWord));
    for i := 0 to FNumBlocks-1 do begin
      StrPCopy(p, FKeyList.Strings[i]); inc(p, Length(FKeyList.Strings[i])+1);
    end;
    // Info
    PDword(p)^ := infolen; inc(p, SizeOf(DWord));
    StrPCopy(p, FInfo.OrigFilename); inc(p, Length(FInfo.OrigFilename)+1);
    PDword(p)^ := FInfo.FileLength;  inc(p, SizeOf(DWord));
    StrPCopy(p, FInfo.Checksum);     inc(p, Length(FInfo.Checksum)+1);
    StrPCopy(p, FInfo.MimeType);     inc(p, Length(FInfo.MimeType)+1);
    // Segheaders
    PDword(p)^ := seghdlen; inc(p, SizeOf(DWord));
    PDword(p)^ := FNumSegments; inc(p, SizeOf(DWord));
    for i := 0 to FNumSegments-1 do begin
      StrPCopy(p, FSegHeaders[i]); inc(p, Length(FSegHeaders[i])+1);
    end;
    // BlockMaps
    if not FIsInsertFile then begin
      PDword(p)^ := blkmaplen; inc(p, SizeOf(DWord));
      for i := 0 to FNumSegments-1 do begin
        StrPCopy(p, FBlockMaps[i]); inc(p, Length(FBlockMaps[i])+1);
      end;
    end;

    Assert(p = (pBuf+len));

    if USE_CACHED_FILES then
      CachedFile.SetContents(pBuf, len)
    else
      Str.WriteBuffer(pBuf^, len);

    if Debug_Delay <> 0 then Sleep(Debug_Delay);
  finally
    if pBuf <> nil then FreeMem(pBuf);
    Str.Free;
    // do NOT free CachedFile
  end;
end;

function TPreparedFile.GetBlockOffset(BlockNum: DWord): DWord;
begin
  Assert(BlockNum < FNumBlocks);
  Result := DWord(FOffsetList.Items[BlockNum]);
end;

procedure TPreparedFile.SetBlockOffset(BlockNum: DWord; const Value: DWord);
begin
  Assert(BlockNum < FNumBlocks);
  FOffsetList.Items[BlockNum] := Pointer(Value);
end;

function TPreparedFile.GetRawDataOffset(BlockNum: DWord): DWord;
begin
  Result := GetBlockOffset(BlockNum) + GetBlockHeaderSize;
end;


function TPreparedFile.GetBlockDone(BlockNum: DWord): Boolean;
begin
  Assert(BlockNum < FNumBlocks);
  Result := (Byte(FDoneList.Items[BlockNum]) and PREPFILEDONEMASK_DONE) <> 0;
end;

procedure TPreparedFile.SetBlockDone(BlockNum: DWord; const Value: Boolean);
begin
  Assert(BlockNum < FNumBlocks);
  if Value then
    FDoneList.Items[BlockNum] := Pointer( Byte(FDoneList.Items[BlockNum]) or PREPFILEDONEMASK_DONE )
  else
    FDoneList.Items[BlockNum] := Pointer( Byte(FDoneList.Items[BlockNum]) and not PREPFILEDONEMASK_DONE );
end;

function TPreparedFile.GetBlockWorking(BlockNum: DWord): Boolean;
begin
  Assert(BlockNum < FNumBlocks);
  Result := (Byte(FDoneList.Items[BlockNum]) and PREPFILEDONEMASK_WORKING) <> 0;
end;

procedure TPreparedFile.SetBlockWorking(BlockNum: DWord; const Value: Boolean);
begin
  Assert(BlockNum < FNumBlocks);
  if Value then
    FDoneList.Items[BlockNum] := Pointer( Byte(FDoneList.Items[BlockNum]) or PREPFILEDONEMASK_WORKING )
  else
    FDoneList.Items[BlockNum] := Pointer( Byte(FDoneList.Items[BlockNum]) and not PREPFILEDONEMASK_WORKING );
end;

function TPreparedFile.GetBlockRetry(BlockNum: DWord): Integer;
begin
  Assert(BlockNum < FNumBlocks);
  Result := (Byte(FDoneList.Items[BlockNum]) and PREPFILEDONEMASK_TRIES);
end;

procedure TPreparedFile.SetBlockRetry(BlockNum: DWord; const Value: Integer);
var
  b,v: Byte;
begin
  Assert(BlockNum < FNumBlocks);
  b := Byte(FDoneList.Items[BlockNum]);
  v := Value;
  if v > $3F then v := $3F;
  FDoneList.Items[BlockNum] := Pointer((b and not PREPFILEDONEMASK_TRIES) or v);
end;


function TPreparedFile.GetKey(BlockNum: DWord): String;
begin
  Assert(BlockNum < FNumBlocks);
  Result := FKeyList.Strings[BlockNum];
end;

procedure TPreparedFile.SetKey(BlockNum: DWord; const Value: String);
begin
  Assert(BlockNum < FNumBlocks);
  FKeyList.Strings[BlockNum] := Value;
end;

function TPreparedFile.GetSegHeader(SegmentNum: DWord): String;
begin
  Assert(SegmentNum < FNumSegments);
  Result := FSegHeaders[SegmentNum];
end;

procedure TPreparedFile.SetSegHeader(SegmentNum: DWord; const Value: String);
begin
  Assert(SegmentNum < FNumSegments);
  FSegHeaders[SegmentNum] := Value;
end;


procedure TPreparedFile.GetData(BlockNum: DWord; out pData: PChar; out DataLen: DWord; out sMetadata: String);
// Alloc and read data; caller must free it later
var
  Str,StrOrg:  TFileStream;
  NewInfo: packed record RefsOrgFile: Byte; OrgFileOffset, OrgFileLen: DWord; end;
  Len:     packed record TotalLen,MetaLen: DWord; end;
  pMeta:   PChar;
begin
  Assert(BlockNum < FNumBlocks);

  pData := nil; DataLen := 0; sMetadata := '';

  pMeta := nil; StrOrg := nil;
  Str := CreateFileStream(DataFilename, fmOpenRead or fmShareDenyWrite);
  try
    Assert(BlockOffset[BlockNum] <= Cardinal(Str.Size - 2*SizeOf(DWord)));
    Str.Seek(BlockOffset[BlockNum], soFromBeginning);
    if FUsesOrgFile then
      Str.ReadBuffer(NewInfo, SizeOf(NewInfo))
    else begin
      NewInfo.RefsOrgFile := 0; NewInfo.OrgFileOffset := 0; NewInfo.OrgFileLen := 0;
    end;
    Str.ReadBuffer(Len, SizeOf(Len));

    if Len.MetaLen <> 0 then begin
      GetMem(pMeta, Len.MetaLen+1);
      Str.ReadBuffer(pMeta^, Len.MetaLen);
      (pMeta+Len.MetaLen)^ := #0;
      sMetadata := pMeta;
    end;

    DataLen := Len.TotalLen - Len.MetaLen;
    if DataLen <> 0 then begin
      GetMem(pData, DataLen); // FillChar(pData^, DataLen, $AA);
      try
        if NewInfo.RefsOrgFile <> 0 then begin
          if DataLen > NewInfo.OrgFileLen then FillChar(pData^, DataLen, 0); // padding ; insert bug was here ( < instead of > )
          StrOrg := CreateFileStream(FInfo.OrigFilename, fmOpenRead or fmShareDenyWrite);
          StrOrg.Seek(NewInfo.OrgFileOffset, soFromBeginning);
          StrOrg.ReadBuffer(pData^, NewInfo.OrgFileLen);
        end else
          Str.ReadBuffer(pData^, DataLen);
      except
        FreeMem(pData); pData := nil;
        raise;
      end;
    end;
  finally
    if pMeta <> nil then FreeMem(pMeta);
    Str.Free;
    StrOrg.Free;
  end;
end;

procedure TPreparedFile.CloseDataFile;
begin
  if Assigned(FOutStream) then FOutStream.Free;
  FOutStream := nil;
end;

function TPreparedFile.GetOutputStream: TFileStream;
begin
  if not Assigned(FOutStream) then begin
    if FileExists(DataFileName) then
      FOutStream := CreateFileStream(DataFileName, fmOpenReadWrite or fmShareExclusive)
    else
      FOutStream := CreateFileStream(DataFileName, fmCreate or fmShareExclusive);
  end;
  Result := FOutStream;
end;

function  TPreparedFile.WriteData(BlockNum: DWord; pData: PChar; DataLen: DWord; sMetadata: String; ReferenceOrgFile: Boolean; OrgFileOffset,OrgFileLen: DWord): Cardinal;
// save data to data file, note block offset
// Result: Offset of beginning of raw data/metadata (past header) in output stream
var
  MetaLen,TotalLen: DWord;
  b:                Byte;
begin
  Assert(BlockNum <= FNumBlocks, 'Write data: more data than blocks');
  Assert(BlockOffset[BlockNum] = $FFFFFFFF, 'Write data: block has already been written');
  Assert((not ReferenceOrgFile) or FUsesOrgFile, 'Write data: cannot reference orig.file in old-type prepfile');

  GetOutputStream;

  FOutStream.Seek(0, soFromEnd);
  BlockOffset[BlockNum] := FOutStream.Position;

  MetaLen  := Length(sMetadata);
  TotalLen := DataLen + MetaLen;
  if FUsesOrgFile then begin
    // new style block
    if ReferenceOrgFile then b := 1 else b := 0;
    FOutStream.WriteBuffer(b, SizeOf(Byte));
    FOutStream.WriteBuffer(OrgFileOffset, SizeOf(DWord));
    FOutStream.WriteBuffer(OrgFileLen,    SizeOf(DWord));
  end;
  FOutStream.WriteBuffer(TotalLen, SizeOf(DWord));
  FOutStream.WriteBuffer(MetaLen, SizeOf(DWord));
  Result := FOutStream.Position;
  if MetaLen > 0 then FOutStream.WriteBuffer(PChar(sMetadata)^, MetaLen);
  if not ReferenceOrgFile then begin
    if DataLen > 0 then FOutStream.WriteBuffer(pData^, DataLen);
  end;
end;

procedure TPreparedFile.RememberState;
var i: Integer;
begin
  try
    if (not Assigned(FOutStream)) and FileExists(DataFileName) then
      GetOutputStream;
    if Assigned(FOutStream) then
      FRemembered.DataFileSize := FOutStream.Size
    else
      FRemembered.DataFileSize := 0;
    FRemembered.OffsetList.Clear;
    for i := 0 to FOffsetList.Count-1 do FRemembered.OffsetList.Add(FOffsetList.Items[i]);
    FRemembered.Valid := True;
  finally
    CloseDataFile;
  end;
end;

procedure TPreparedFile.RevertToRememberedState;
var
  F: File;
  i: Integer;
begin
  Assert(FRemembered.Valid, 'No remembered state');
  CloseDataFile;
  if FRemembered.DataFileSize = 0 then begin
    if FileExists(DataFileName) then DeleteFile(DataFileName);
  end else begin
    AssignFile(F, DataFileName);
    FileMode := 2; Reset(F,1);
    Seek(F, FRemembered.DataFileSize);
    Truncate(F);
    CloseFile(F);
  end;
  for i := 0 to FRemembered.OffsetList.Count-1 do FOffsetList.Items[i] := FRemembered.OffsetList.Items[i];
end;


function TPreparedFile.GetBlockHeaderSize: DWord;
type
  TNewInfo = packed record RefsOrgFile: Byte; OrgFileOffset, OrgFileLen: DWord; end;
  TLen     = packed record TotalLen,MetaLen: DWord; end;
begin
  if FUsesOrgFile then
    Result := SizeOf(TNewInfo) + SizeOf(TLen)
  else
    Result := SizeOf(TLen);
end;

procedure TPreparedFile.DeleteFiles;
begin
  if USE_CACHED_FILES then gCachedFiles.CloseFile(HeaderFilename, False);
  DeleteFile(HeaderFilename);
  DeleteFile(DataFilename);
end;

function TPreparedFile.GetAllDone: Boolean;
var i: Integer;
begin
  if FIsInsertFile then begin
    Result := True;
    for i := 0 to FNumBlocks-1 do
      if not BlockDone[i] then begin
        Result := False; break;
      end;
  end else begin
    // return if we have enough data to fec-decode
    Result := True;
    for i := 0 to FNumSegments-1 do
      if BlocksCompletedInSeg[i] < BlocksRequiredInSeg[i] then begin
        Result := False; break;
      end;
  end;
end;

function TPreparedFile.GetNumDone: DWord;
var i: Integer;
begin
  Result := 0;
  for i := 0 to FNumBlocks-1 do
    if BlockDone[i] then inc(Result);
end;

function TPreparedFile.GetNumWorking: DWord;
var i: Integer;
begin
  Result := 0;
  for i := 0 to FNumBlocks-1 do
    if BlockWorking[i] then inc(Result);
end;


function TPreparedFile.CalcSplitFileMetadata(FCPAddr: u_long; FCPPort: Word; InsertStyle: TInsertStyle): String;
// calculate split file metadata
  function RemovePrefix(s: String): String;
  begin
    if CompareText(Copy(s,1,8), 'freenet:') = 0 then
      Result := Copy(s,9,Length(s))
    else
      Result := s;
  end;

var
  Maps:   Array of String;
  iBlock,iSeg:        Integer;
  sSegHead,sMap,sKey: String;
  slSeg:              TStringList;
  i,NumData,NumCheck: Integer;
  sMime:              String;
  fcp:                TFCPSession;
begin
  // this code relies on the fact that MainBlock is either 0 or (NumBlocks-1)
  // and the sequence of the other blocks is in the following order
  // Seg1 data, Seg1 check, Seg2 data, Seg2 check, ..., SegN data SegN check

  // Safe to be called even if the MainBlock data is not yet written

  Assert((FMainBlock = 0) or (FMainBlock = FNumBlocks-1));

  slSeg := nil; fcp := nil;
  SetLength(Maps, FNumSegments);
  try
    slSeg := TStringList.Create;
    if FMainBlock = 0 then iBlock := 1 else iBlock := 0; // first data block

    for iSeg := 0 to FNumSegments-1 do begin
      sSegHead := FSegHeaders[iSeg];
      FCPMsgToStringList(sSegHead, slSeg);
      NumData  := StrToInt('$' + slSeg.Values['BlockCount']);
      NumCheck := StrToInt('$' + slSeg.Values['CheckBlockCount']);

      sMap := 'BlockMap'#10;
      for i := 0 to NumData-1 do begin
        sKey := Key[iBlock]; Assert(sKey <> '', 'Key missing for block');
        sMap := sMap + Format('Block.%s=%s'#10,[Hex(i),sKey]);
        inc(iBlock);
      end;
      for i := 0 to NumCheck-1 do begin
        sKey := Key[iBlock]; Assert(sKey <> '', 'Key missing for block');
        sMap := sMap + Format('Check.%s=%s'#10,[Hex(i),sKey]);
        inc(iBlock);
      end;
      sMap := sMap + 'EndMessage'#10;
      Maps[iSeg] := sMap;
    end;

    sMime := FInfo.MimeType;
    if sMime = '' then sMime := GetMimeType(ExtractFileExt(FInfo.OrigFilename), InsertStyle);

    fcp := TFCPSession.Create(FCPAddr, FCPPort);
    Result := Util_FECMakeMetadata(FSegHeaders,Maps,FInfo.Checksum,sMime,InsertStyle,fcp);

  finally
    slSeg.Free;
    fcp.Free;
    SetLength(Maps, 0);
  end;
end;

procedure TPreparedFile.InitFromSplitFileMetadata(FCPAddr: u_long; FCPPort: Word; Metadata: String);
// init from splitfile metadata (for a splitfile download)
var
  fcp: TFCPSession;
  Headers,Maps: TDynamicStringArray;
  nSegs,i:      Integer;
  nBlocks:      Integer;
  SegInfo:      TFECSegmentInfo;
  iBlockInSeg,iBlock: Integer; // INTEGER for the for-loop!
  slMap:        TStringList;
begin
  Assert(Metadata <> '');
  SetLength(Headers,0); SetLength(Maps,0);
  slMap := nil;
  fcp := TFCPSession.Create(FCPAddr,FCPPort);
  try
    nSegs := Util_FECSegmentSplitFile(Metadata, Headers, Maps, fcp);
    Assert(nSegs > 0);
    nBlocks := 0;
    for i := 0 to nSegs-1 do begin
      Util_GetFECSegmentInfo(Headers[i], SegInfo);
      inc(nBlocks, SegInfo.BlockCount + SegInfo.CheckBlockCount);
    end;
    Assert(nBlocks > 0);
    Init(nBlocks, nSegs);

    // store headers, blockmaps, keys
    slMap := TStringList.Create;
    iBlock := 0;
    for i := 0 to FNumSegments-1 do begin
      FSegHeaders[i] := Headers[i]; FBlockMaps[i] := Maps[i];
      // init keys
      Util_GetFECSegmentInfo(FSegHeaders[i], SegInfo);
      slMap.Clear;
      FCPMsgToStringList(FBlockMaps[i], slMap);
      for iBlockInSeg := 0 to SegInfo.BlockCount-1 do begin
        Key[iBlock] := slMap.Values['Block.'+Hex(iBlockInSeg)];
        inc(iBlock);
      end;
      for iBlockInSeg := 0 to SegInfo.CheckBlockCount-1 do begin
        Key[iBlock] := slMap.Values['Check.'+Hex(iBlockInSeg)];
        inc(iBlock);
      end;
    end;


  finally
    fcp.Free;
    slMap.Free;
    SetLength(Headers,0); SetLength(Maps,0);
  end;
end;


function TPreparedFile.GetNextDownloadBlock(MaxTries: Integer; out iBlock: DWord; EvenGetUnnecessaryBlock: Boolean): Boolean;
// determine next block to download. Return if any found
// block must not have had more than MaxTries tries
var
  i,iSeg,iPass,mintries,FirstCheckIdx: Integer;
  FirstBlock,NumData,NumCheck: DWord;
  TmpList:    TList;
begin
  // note: code for PREFER_DATABLOCKS currently broken!

  Result := False;
  TmpList := TList.Create;
  try
    for iSeg := 0 to FNumSegments-1 do begin
      if (BlocksCompletedInSeg[iSeg] >= BlocksRequiredInSeg[iSeg]) and (not EvenGetUnnecessaryBlock) then continue;

      GetSegmentBlockInfo(iSeg, FirstBlock, NumData, NumCheck);

      mintries := $FF; // actual max. try# is $3F, so this is always bigger
      FirstCheckIdx := -1;
      for iPass := 1 to 2 do begin
        // in pass 1 we determine the mintries, in pass 2 we fill the list
        for i := FirstBlock to FirstBlock+NumData+NumCheck-1 do begin
          if not (BlockWorking[i] or BlockDone[i]) then begin
            if iPass = 1 then begin
              if BlockTries[i] < mintries then mintries := BlockTries[i];
            end else begin
              if BlockTries[i] = mintries then begin
                TmpList.Add(Pointer(i));
                if (FirstCheckIdx < 0) and (i >= Integer(FirstBlock+NumData)) then
                  FirstCheckIdx := TmpList.Count-1; // remember pos. of first checkblock
              end;
            end;
          end;
        end;
        if (iPass = 1) and (mintries > MaxTries) then break; // break the iPass loop
      end;
    end;

    if TmpList.Count > 0 then begin
      // if we have data and checkblocks for disposition:
      {
      if FirstCheckIdx >= 1 then begin
        if Random <= PREFER_DATABLOCKS then
          i := Random(FirstCheckIdx) // pick a data block
        else
          i := FirstCheckIdx + Random(TmpList.Count-FirstCheckIdx);
      end else
        i := Random(TmpList.Count);
      }
      i := Random(TmpList.Count);
      iBlock := Integer(TmpList.Items[i]);
      Result := True;
      exit;
    end;
  finally
    TmpList.Free;
  end;
end;


procedure TPreparedFile.GetSegmentBlockInfo(SegmentNum: DWord; out FirstBlock, NumData, NumCheck: DWord);
var
  iSeg:    Integer;
  SegInfo: TFECSegmentInfo;
begin
  Assert(SegmentNum < FNumSegments);

  FirstBlock := 0;
  for iSeg := 0 to Integer(SegmentNum) do begin
    Assert(FSegHeaders[iSeg] <> '', 'Information not available');
    Util_GetFECSegmentInfo(FSegHeaders[iSeg], SegInfo);
    if iSeg < Integer(SegmentNum) then
      inc(FirstBlock, SegInfo.BlockCount + SegInfo.CheckBlockCount)
    else begin
      NumData  := SegInfo.BlockCount;
      NumCheck := SegInfo.CheckBlockCount;
    end;
  end;
end;

function TPreparedFile.GetBlocksRequiredInSeg(SegmentNum: DWord): Integer;
var SegInfo: TFECSegmentInfo;
begin
  Assert(SegmentNum < FNumSegments);
  Assert(FSegHeaders[SegmentNum] <> '', 'Information not available');
  Util_GetFECSegmentInfo(FSegHeaders[SegmentNum], SegInfo);
  Result := SegInfo.BlocksRequired;
end;

function TPreparedFile.GetLastDataBlock: DWord;
var
  FirstBlock:        DWord;
  NumData,NumCheck:  DWord;
begin
  GetSegmentBlockInfo(NumSegments-1, FirstBlock, NumData, NumCheck);
  Result := FirstBlock+NumData-1;
end;

function TPreparedFile.GetBlocksCompletedInSeg(SegmentNum: DWord; IncludeData,IncludeCheck: Boolean): Integer;
var
  FirstBlock:        DWord;
  NumData,NumCheck:  DWord;
  iBlock:            Integer;
begin
  GetSegmentBlockInfo(SegmentNum, FirstBlock, NumData, NumCheck);

  Result := 0;
  if IncludeData then begin
    for iBlock := FirstBlock to FirstBlock+NumData-1 do begin
      if BlockDone[iBlock] then inc(Result);
    end;
  end;
  if IncludeCheck then begin
    for iBlock := FirstBlock+NumData to FirstBlock+NumData+NumCheck-1 do begin
      if BlockDone[iBlock] then inc(Result);
    end;
  end;
end;

function TPreparedFile.GetBlocksCompletedInSeg_All(SegmentNum: DWord): Integer;
begin
  Result := GetBlocksCompletedInSeg(SegmentNum, True, True);
end;

function TPreparedFile.GetBlocksCompletedInSeg_Data(SegmentNum: DWord): Integer;
begin
  Result := GetBlocksCompletedInSeg(SegmentNum, True, False);
end;

function TPreparedFile.GetBlocksCompletedInSeg_Check(SegmentNum: DWord): Integer;
begin
  Result := GetBlocksCompletedInSeg(SegmentNum, False, True);
end;

function TPreparedFile.GetBlocksFailedInSeg(SegmentNum: DWord; MaxRetries: Integer): Integer;
var
  FirstBlock:        DWord;
  NumData,NumCheck:  DWord;
  iBlock:            Integer;
begin
  GetSegmentBlockInfo(SegmentNum, FirstBlock, NumData, NumCheck);

  Result := 0;
  for iBlock := FirstBlock to FirstBlock+NumData+NumCheck-1 do begin
    if not (BlockDone[iBlock] or BlockWorking[iBlock]) then begin
      if BlockTries[iBlock] > MaxRetries then inc(Result);
    end;
  end;
end;

function TPreparedFile.GetIsDataBlock(BlockNum: DWord): Boolean;
var
  iSeg: Integer;
  SegInfo: TFECSegmentInfo;
  FirstBlock: Integer;
begin
  Assert(not FIsInsertFile, 'Information not available for insert files');

  FirstBlock := 0;
  for iSeg := 0 to FNumSegments-1 do begin
    Util_GetFECSegmentInfo(FSegHeaders[iSeg], SegInfo);
    if BlockNum < DWord(FirstBlock + SegInfo.BlockCount) then begin
      Result := True; exit;
    end;
    if BlockNum < DWord(FirstBlock + SegInfo.BlockCount + SegInfo.CheckBlockCount) then begin
      Result := False; exit;
    end;
    inc(FirstBlock, SegInfo.BlockCount + SegInfo.CheckBlockCount);
  end;

  raise Exception.Create('Unable to determine block type');
end;

function TPreparedFile.GetIsCheckBlock(BlockNum: DWord): Boolean;
begin
  Result := not GetIsDataBlock(BlockNum);              
end;


function TPreparedFile.GetCompletionRatioEx(var BlocksLeft: Integer; MaxRetries: Integer): Single;
var
  iSeg,i: Integer;
  SegDone,SegNeed,Done,Total: Integer;
  FirstBlock,NumData,NumCheck: DWord;
begin
  if FIsInsertFile then begin
    Result := NumDone / FNumBlocks;
    BlocksLeft := FNumBlocks - NumDone;
  end else begin
    Done := 0; Total := 0; Result := 0; BlocksLeft := 0;
    for iSeg := 0 to FNumSegments-1 do begin
      SegDone := BlocksCompletedInSeg[iSeg];
      SegNeed := BlocksRequiredInSeg[iSeg];
      if SegDone > SegNeed then SegDone := SegNeed;
      Done  := Done  + SegDone;
      Total := Total + SegNeed;
      if SegDone < SegNeed then begin
        GetSegmentBlockInfo(iSeg, FirstBlock, NumData, NumCheck);
        for i := FirstBlock to FirstBlock+NumData+NumCheck-1 do
          if (not BlockDone[i]) and (BlockTries[i] <= MaxRetries) then inc(BlocksLeft);
      end;
    end;
    if Total <> 0 then Result := Done / Total;
  end;
end;

function TPreparedFile.GetCompletionRatio: Single;
var dummy: Integer;
begin
  Result := GetCompletionRatioEx(dummy,0);
end;

function TPreparedFile.GetProgressString(MaxRetries: Integer = 0): String;
var
  ratio:      Single;
  BlocksLeft: Integer;
begin
  ratio := GetCompletionRatioEx(BlocksLeft, MaxRetries);
  Result := Format('%d%%, %d',[Round(100 * ratio),BlocksLeft]);
end;

function TPreparedFile.IsSuccessPossible(MaxRetries: Integer): Boolean;
var
  iSeg: Integer;
  SegInfo: TFECSegmentInfo;
begin
  Result := True;
  if FIsInsertFile or (FNumSegments = 0) then exit;

  for iSeg := 0 to FNumSegments-1 do begin
    Util_GetFECSegmentInfo(FSegHeaders[iSeg], SegInfo);
    if BlocksFailedInSeg[iSeg, MaxRetries] > (SegInfo.BlockCount + SegInfo.CheckBlockCount - SegInfo.BlocksRequired) then begin
      Result := False; exit;
    end;
  end;
end;

procedure TPreparedFile.SaveOutputFile(Outputfilename: String; CheckAbortProc: TCheckAbortProc; PartialSave: TPartialSave);
var
  OutStr: TFileStream;
  iSeg,iBlock: Integer;
  FirstBlockInSeg: Integer;
  SegInfo:     TFECSegmentInfo;
  pData,pTmp:        PChar;
  DataLen,BytesLeft: DWord;
  LastGoodBlock:     Integer;
  sMeta:             String;
  TmpLen:            DWord;
begin
  Assert(not FIsInsertFile);
  OutStr := CreateFileStream(Outputfilename, fmCreate or fmShareExclusive);
  try
    FirstBlockInSeg := 0; BytesLeft := 0;
    LastGoodBlock := -1;
    for iSeg := 0 to FNumSegments-1 do begin
      Util_GetFECSegmentInfo(FSegHeaders[iSeg], SegInfo);
      if iSeg = 0 then BytesLeft := SegInfo.FileLength;
      for iBlock := FirstBlockInSeg to FirstBlockInSeg + SegInfo.BlockCount - 1 do begin
        pData := nil;
        try
          if not BlockDone[iBlock] then begin
            case PartialSave of
              psSkipMissing:  if BytesLeft >= Cardinal(SegInfo.BlockSize) then dec(BytesLeft, SegInfo.BlockSize) else BytesLeft := 0;
              psZeroMissing:  begin
                                DataLen := SegInfo.BlockSize;
                                GetMem(pData, DataLen);
                                FillChar(pData^, DataLen, 0);
                              end;
              psCopyLastGood: begin
                                DataLen := SegInfo.BlockSize;
                                GetMem(pData, DataLen);
                                FillChar(pData^, DataLen, 0);
                                if LastGoodBlock >= 0 then begin
                                  pTmp := nil;
                                  try
                                    GetData(LastGoodBlock, pTmp, TmpLen, sMeta);
                                    if TmpLen > DataLen then TmpLen := DataLen;
                                    Move(pTmp^, pData^, TmpLen);
                                  finally
                                    if pTmp <> nil then FreeMem(pTmp);
                                  end;
                                end;
                              end;
              else            raise Exception.Create('Not all data blocks present');
            end;
          end else begin
            LastGoodBlock := iBlock; // remember the last good block
            GetData(iBlock, pData, DataLen, sMeta);
          end;
          if pData <> nil then begin
            Assert(DataLen <> 0, 'Data block size is 0');
            if DataLen > BytesLeft then DataLen := BytesLeft;
            OutStr.WriteBuffer(pData^, DataLen);
            dec(BytesLeft, DataLen);
          end;
          if Assigned(CheckAbortProc) then CheckAbortProc;
        finally
          if pData <> nil then FreeMem(pData); pData := nil;
        end;
        if BytesLeft = 0 then break;
      end;
      FirstBlockInSeg := FirstBlockInSeg + SegInfo.BlockCount + SegInfo.CheckBlockCount;
      if BytesLeft = 0 then break;
    end;
  finally
    OutStr.Free;
  end;
end;

function TPreparedFile.GetHealBlocks(Percent: Integer): String;
// return a list of heal blocks as comma-separated string (hex-numbers!)
var iBlock,nBlocks,Cnt,iPass: Integer;
begin
  // we prefer for healblocks done-blocks with 0 tries
  Result := '';
  nBlocks := Ceil(DWord(Percent) * FNumBlocks / 100); Cnt := 0;
  if nBlocks = 0 then exit;
  for iPass := 1 to 2 do begin
    for iBlock := 0 to FNumBlocks-1 do begin
      if BlockDone[iBlock] and ((iPass = 1) = (BlockTries[iBlock] = 0)) then begin
        if Result <> '' then Result := Result + ',';
        Result := Result + Hex(iBlock);
        inc(Cnt); if Cnt >= nBlocks then break;
      end;
    end;
    if Cnt >= nBlocks then break;
  end;
end;

function TPreparedFile.GetBlockStatusString: String;
// return a string encoding the single block status and some file info
// format: <xx>#<blockstat>#<numrequired>
// <xx>:(2-digit hex) percent completed
// <blockstat>:this is just a 2-digit-hex-per-entry version of the DoneList
//             exception: 'ff' is used for blocks of segments that already have enough blocks,
//                        if the block is idle and unretrieved
// <numrequired>:hex num required blocks
var
  i,j,n,iBlock,iStat: Integer;
  SegInfo: TFECSegmentInfo;
  SegDone: Boolean;
begin
  Result := IntToHex(Round(GetCompletionRatio),2) + '#';

  if (not FIsInsertFile) and (FNumSegments > 0) then begin
    n := 0; iBlock := 0;
    for i := 0 to FNumSegments-1 do begin
      Util_GetFECSegmentInfo(FSegHeaders[i], SegInfo);
      n := n + SegInfo.BlocksRequired;
      SegDone := (BlocksCompletedInSeg[i] >= BlocksRequiredInSeg[i]);
      for j := 0 to SegInfo.BlockCount + SegInfo.CheckBlockCount - 1 do begin
        iStat := Byte(FDoneList.Items[iBlock]);
        if SegDone and (iStat and PREPFILEDONEMASK_WORKING = 0) and (iStat and PREPFILEDONEMASK_DONE = 0) then
          Result := Result + IntToHex($FF,2)
        else
          Result := Result + IntToHex(iStat,2); // don't use Hex() here
        inc(iBlock);
      end;
    end;
  end else begin
    for i := 0 to FDoneList.Count-1 do
      Result := Result + IntToHex(Byte(FDoneList.Items[i]),2); // don't use Hex() here
    n := FNumBlocks;
  end;
  Result := Result + '#' + Hex(n);
end;

function TPreparedFile.FixLastDatablockPadding(BlockNum: DWord; pData: PChar; DataLen: DWord): Boolean;
// fix zero-padding of last datablock (Fuqid 1.2-insert bug)
// Return True if bug detected and fixed, False otherwise
var
  i:                                Integer;
  iBlock,SegStart,NumData,NumCheck: Cardinal;
  PaddedLen,RealLen:                Int64;
  SegInfo:                          TFECSegmentInfo;
  p:                                PChar;
begin
  Result := False;
  if FIsInsertFile then exit;
  if FNumSegments <= 0 then exit;

  PaddedLen := 0;
  for i := 0 to NumSegments-1 do begin
    Util_GetFECSegmentInfo(SegHeader[i], SegInfo);
    PaddedLen := PaddedLen + SegInfo.BlockCount * SegInfo.BlockSize;
  end;
  Util_GetFECSegmentInfo(SegHeader[NumSegments-1], SegInfo);
  RealLen := SegInfo.BlockSize - (PaddedLen - SegInfo.FileLength);
  GetSegmentBlockInfo(NumSegments-1, SegStart, NumData, NumCheck);
  iBlock := SegStart + NumData - 1;

  if iBlock <> BlockNum then exit;

  if (RealLen > 0) and (RealLen < DataLen) then begin
    p := pData + RealLen;
    for i := RealLen to DataLen-1 do begin
      if p^ <> #0 then begin Result := True; p^ := #0; end;
      inc(p);
    end;
  end;
end;

{ TFreenetURI }

constructor TFreenetURI.Create;
begin
  inherited;
  FMetaInfo    := TStringList.Create;
  FMetaStrings := TStringList.Create;
end;

destructor TFreenetURI.Destroy;
begin
  FMetaInfo.Free;
  FMetaStrings.Free;
  inherited;
end;

procedure TFreenetURI.ClearAll;
begin
  FKeyType    := '';
  FRoutingKeyB64 := '';
  FCryptoKeyB64  := '';
  FMetaInfo.Clear;
  FDocName    := '';
  FMetaStrings.Clear;
end;

function TFreenetURI.GetURI(WithPrefix,WithMeta,WithDocName: Boolean): String;
var i: Integer;
begin
  if WithPrefix then Result := 'freenet:' else Result := '';
  Result := Result + FKeyType + '@';
  if FKeyType <> 'KSK' then begin
    Result := Result + FRoutingKeyB64;
    if FCryptoKeyB64 <> '' then Result := Result + ',' + FCryptoKeyB64;
    if WithMeta then
      for i := 0 to FMetaInfo.Count-1 do
        Result := Result + ',' + FMetaInfo.Strings[i];
    if WithDocName and (FDocName <> '') then Result := Result + '/';
  end;
  if WithDocName then Result := Result + FDocName;
  if WithMeta then
    for i := 0 to FMetaStrings.Count-1 do
      Result := Result + '//' + FMetaStrings.Strings[i];
  Result := OptionalURLEncode(Result);
end;

procedure TFreenetURI.SetURI(const Value: String);
var
  s,sMet,sMet2: String;
  i,i2: Integer;
begin
  ClearAll;

  s := URLDecode(Trim(Value));
  i := Pos(':',s);
  if i <> 0 then begin
    if CompareText(Copy(s,1,i), 'freenet:') <> 0 then
      raise EFreenetURI.Create('Invalid Freenet URI');
    Delete(s,1,i);
  end;
  i := Pos('@',s);
  if i <> 0 then begin
    FKeyType := UpperCase(Copy(s,1,i-1));
    Delete(s,1,i);
  end else
    FKeyType := 'KSK';

  FMetaStrings.Clear;
  i := Pos('//',s);
  if i <> 0 then begin
    sMet := Copy(s, i+2, Length(s)); Delete(s, i, Length(s));
    repeat
      i := Pos('//',sMet);
      if i = 0 then
        sMet2 := sMet
      else begin
        sMet2 := Copy(sMet,1,i-1);
        Delete(sMet,1,i+1);
      end;
      if sMet2 <> '' then FMetaStrings.Add(sMet2);
    until i = 0;
  end;

  if FKeyType = 'KSK' then begin
    FDocName := s; exit;
  end;

  i := Pos('/',s);
  if i <> 0 then begin
    FDocName := Copy(s,i+1,Length(s));
    Delete(s,i,Length(s));
  end;

  s := s + ',';
  i := Pos(',',s);
  FRoutingKeyB64 := Copy(s,1,i-1); Delete(s,1,i);
  i := Pos(',',s);
  if (i <> 0) and (Pos('=',Copy(s,1,i-1)) = 0) then begin
    FCryptoKeyB64 := Copy(s,1,i-1); Delete(s,1,i);
  end;
  i := Pos(',',s);
  while i <> 0 do begin
    sMet := Copy(s,1,i-1); Delete(s,1,i);
    i2 := Pos('=',sMet);
    if i2 <= 1 then raise EFreenetURI.Create('Invalid meta info');
    FMetaInfo.Values[Copy(sMet,1,i2-1)] := Copy(sMet,i2+1,Length(sMet));
    i := Pos(',',s);
  end;
end;

procedure TFreenetURI.Debug;
var
  s: String;
  i: Integer;
  fn: TFreenetFileNumber;
begin
  s := 'KeyType: ' + FKeyType
     + #13'Routing: ' + FRoutingKeyB64
     + #13' decoded:' + Base64decodeAsString(FRoutingKeyB64)
     + #13'Crypto: '  + FCryptoKeyB64
     + #13' decoded:' + Base64decodeAsString(FCryptoKeyB64)
     + #13'DocName: ' + FDocName;
  for i := 0 to FMetaStrings.Count-1 do
    s := s + #13'MetaString['+IntToStr(i)+']: ' + FMetaStrings.Strings[i];
  for i := 0 to FMetaInfo.Count-1 do
    s := s + #13'MetaInfo['+IntToStr(i)+']: ' + FMetaInfo.Strings[i];
  s := s + #13#13'URI: ' + GetURI(True);
  fn := FileNumber;
  s := s + #13#13'FileNo: Long:' + IntToHex(fn.LongHash,16) + ' Short:' + IntToHex(fn.ShortHash,8);
  s := s + #13#13'DSFile: ' + DSFile;
  MessageDlg(s, mtInformation, [mbOk], 0);
end;

function TFreenetURI.URLDecode(const S: String): String;
var i: Integer;
begin
  Result := '';
  i := 1;
  while i <= Length(S) do begin
    if S[i] = '+' then
      Result := Result + ' '
    else if (S[i] = '%') then begin
      Result := Result + Chr(StrToIntDef('$' + Copy(S,i+1,2), 0));
      inc(i,2);
    end else
      Result := Result + S[i];
    inc(i);
  end;
end;

function TFreenetURI.URLEncode(const S: String): String;
var i: Integer;
begin
  Result := '';
  for i := 1 to Length(S) do begin
    if S[i] = ' ' then
      Result := Result + '+'
    else if (S[i] > #128) or (S[i] < #44) then
      Result := Result + '%' + LowerCase(IntToHex(Ord(S[i]),2))
    else
      Result := Result + S[i];
  end;
end;

function TFreenetURI.OptionalURLEncode(const S: String): String;
begin
  if FDontURLEncode then Result := S else Result := URLEncode(S); 
end;

function TFreenetURI.GetMetaInfoAsString: String;
var i: Integer;
begin
  Result := '';
  for i := 0 to FMetaInfo.Count-1 do begin
    if i > 0 then Result := Result + ',';
    Result := Result + FMetaInfo.Strings[i];
  end;
  Result := OptionalURLEncode(Result);
end;

function TFreenetURI.GetMetaStringsAsString: String;
var i: Integer;
begin
  Result := '';
  for i := 0 to FMetaStrings.Count-1 do begin
    if i > 0 then Result := Result + '//';
    Result := Result + FMetaStrings.Strings[i];
  end;
  Result := OptionalURLEncode(Result);
end;

procedure TFreenetURI.SetDocName(const Value: String);
begin
  FDocName := URLDecode(Value);
end;

{
function TFreenetURI.GetFileNumber: TFreenetFileNumber;
  function unTC(b: ShortInt): Integer;
  begin
    Result := b;
    if Result < 0 then inc(Result,256);
  end;
var
  i: Integer;
  h,x:   Int64;
  s:   String;
begin
  s := FRoutingKeyB64;
  h := 0;
  for i := Length(s)-1 downto 0 do begin
    x := unTC(ShortInt(s[i+1]));
    h := h xor (x shl ((i and 7) shl 3));
  end;
  Result.LongHash  := h;
  Result.ShortHash := Cardinal( ((((h and $FFFFFFFF00000000) shr 32) or ((h and $00000000FFFFFFFF) shl 32)) xor h) and $00000000FFFFFFFF );
end;
}

function TFreenetURI.GetFileNumber: TFreenetFileNumber;
var
  i: Integer;
  h,x:   Int64;
  s:   String;
  pB:  PByte;
  Len: Integer;
begin
  Base64decode(FRoutingKeyB64, pB, Len);
  try
    h := 0;
    for i := Len-1 downto 0 do begin
      x := PByte(PChar(pB) + i)^;
      h := h xor (x shl ((i and 7) shl 3));
    end;
  finally
    FreeMem(pB);
  end;
  Result.LongHash  := h;
  Result.ShortHash := Cardinal( ((((h and $FFFFFFFF00000000) shr 32) or ((h and $00000000FFFFFFFF) shl 32)) xor h) and $00000000FFFFFFFF );
end;

function TFreenetURI.GetDSFile: String;
var
  dig,digx: TSHA1Digest;
  sha,shax: TSHA1Context;
  sRawRK:   String;
  pData:    PByte;
  DataLen:  Integer;
  sDir:     String;
begin
  sRawRK := Base64decodeAsString(FRoutingKeyB64);
  if FKeyType = 'CHK' then begin
    Result := '1-' + sRawRK;
  end else if FKeyType = 'SSK' then begin
    if FDocName = '' then
      Result := '1-' + sRawRK
    else begin
      pData := nil; DataLen := 0;
      try
        Base64decode(FRoutingKeyB64, pData, DataLen);
        Assert(DataLen >= 20);
        SHA1Reset(sha);
        SHA1Input(sha, PChar(pData), 20);
        SHA1Reset(shax);
        SHA1Input(shax, PChar(FDocName), Length(FDocName));
        SHA1Result(shax, digx);
        SHA1Input(sha, @digx, SizeOf(digx));
        SHA1Result(sha, dig);
        Result := '1-' + SHA1DigestToLoHexString(dig) + Copy(sRawRK,1+20*2,Length(sRawRK));
      finally
        FreeMem(pData);
      end;
    end;
  end else
    Result := '';

  if Result <> '' then begin
    sDir := Hex(FileNumber.ShortHash and $FF) + '\';
    Insert(sDir, Result, 1);
  end;
end;

initialization
  FreenetThreadCritSec := TCriticalSection.Create;

{$IFDEF DEBUG_LOGFCPOUT}
  if ExtractFilePath(DEBUG_LOGFCPOUTFILE) = '' then
    Insert(ExtractFilePath(Application.ExeName), DEBUG_LOGFCPOUTFILE, 1);
{$ENDIF}
{$IFDEF DEBUG_LOGFCPIN}
  if ExtractFilePath(DEBUG_LOGFCPINFILE) = '' then
    Insert(ExtractFilePath(Application.ExeName), DEBUG_LOGFCPINFILE, 1);
{$ENDIF}
{$IFDEF DEBUG_LOG}
  if ExtractFilePath(DEBUG_LOGFILE) = '' then
    Insert(ExtractFilePath(Application.ExeName), DEBUG_LOGFILE, 1);
{$ENDIF}

finalization
  FreenetThreadCritSec.Free;

end.

