//PROFILE-NO
unit FixedThread;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

{
// TFixedThread is a bugfixed version of TThread in D4
//
// bugs fixed:
//   Synchronize may not be called, since SendMessage fails
//   this also means that OnTerminate might not be called
//   - fixed by using SendMessageTimeout, retrying if necessary
//
//   race condition when destroying the last thread and immediately creating a new one.
//   - fixed by not freeing the threadwin when the thread count reaches zero
}

// As far as I know the TThread bug is gone with Delphi 5
// FixedThread seems to work there anyway though.
// If you experience Thread bugs with D5 or up, just remove the {$UNDEF ..}

{$DEFINE USEFIXEDTHREAD}
{$IFDEF DELPHI_5_OR_HIGHER} {$UNDEF USEFIXEDTHREAD} {$ENDIF}

{$IFNDEF USEFIXEDTHREAD}

interface

uses Classes;

type TFixedThread = TThread;

implementation

{$ELSE}

interface

uses Windows,Classes;

type
  TThreadMethod = procedure of object;
  TFixedThread = class(TThread)
  private
    FSyncExcept: TObject;
    FSyncMeth:   TThreadMethod;
    FOnTerminate: TNotifyEvent;
    procedure DoTerminateSyncedProc;
  protected
    procedure Synchronize(Method: TThreadMethod);
    procedure DoTerminate; override;
  public
    constructor Create(CreateSuspended: Boolean);
    destructor Destroy; override;
    property OnTerminate: TNotifyEvent read FOnTerminate write FOnTerminate;
  end;

var
  FixedThreadWin: HWND    = INVALID_HANDLE_VALUE;
  FixedThreadCnt: Integer = 0;

implementation

// util functions
const
  CM_EXECPROC      = $8FFF;
  CM_DESTROYWINDOW = $8FFE;

type
  PRaiseFrame = ^TRaiseFrame;
  TRaiseFrame = record
    NextRaise:       PRaiseFrame;
    ExceptAddr:      Pointer;
    ExceptObject:    TObject;
    ExceptionRecord: PExceptionRecord;
  end;

procedure FreeFixedThreadWin;
begin
  if FixedThreadWin <> INVALID_HANDLE_VALUE then begin
    DestroyWindow(FixedThreadWin);
    FixedThreadWin := INVALID_HANDLE_VALUE;
  end;
end;

function FixedThreadWndProc(Window: HWND; Msg, wParam, lParam: Longint): Longint; stdcall;
begin
  case Msg of
    CM_EXECPROC:
      with TFixedThread(lParam) do begin
        Result := 0;
        try
          FSyncExcept := nil;
          FSyncMeth;
        except
          if RaiseList <> nil then begin
            FSyncExcept := PRaiseFrame(RaiseList)^.ExceptObject;
            PRaiseFrame(RaiseList)^.ExceptObject := nil;
          end;
        end;
      end;
    CM_DESTROYWINDOW:
      begin
        FreeFixedThreadWin;
        Result := 0;
      end;
    else
      Result := DefWindowProc(Window, Msg, wParam, lParam);
  end;
end;

procedure CreateFixedThreadWin;
const
  FixedThreadWinClassName = 'TFixedThreadWindow';
var
  FixedThreadWinClass,Cls: TWndClass;
  ok:  Boolean;
begin
  if FixedThreadWin <> INVALID_HANDLE_VALUE then exit;
  
  ok := GetClassInfo(HInstance, FixedThreadWinClassName, Cls);
  if (not ok) or (Cls.lpfnWndProc <> @FixedThreadWndProc) then begin
    if ok then Windows.UnregisterClass(FixedThreadWinClassName, HInstance);
    FillChar(FixedThreadWinClass, SizeOf(FixedThreadWinClass), 0);
    FixedThreadWinClass.lpfnWndProc   := @FixedThreadWndProc;
    FixedThreadWinClass.hInstance     := HInstance;
    FixedThreadWinClass.lpszClassName := FixedThreadWinClassName;
    Windows.RegisterClass(FixedThreadWinClass);
  end;
  FixedThreadWin := CreateWindow(
                      FixedThreadWinClassName,
                      '', 0, 0, 0, 0, 0, 0, 0,
                      HInstance, nil
                    );
end;



{ TFixedThread }

constructor TFixedThread.Create(CreateSuspended: Boolean);
begin
  //outputdebugstring('fixedthread.create');
  CreateFixedThreadWin; // create global thread window if it doesn't exist yet
  inc(FixedThreadCnt);  // inc. global thread count
  inherited Create(CreateSuspended);
end;

destructor TFixedThread.Destroy;
begin
  //outputdebugstring('fixedthread.destroy');
  inherited Destroy;
  dec(FixedThreadCnt);  // dec. global thread count
  // note: don't free the win
  //if FixedThreadCnt = 0 then FreeFixedThreadWin; // free global thread window if cnt = 0
end;

procedure TFixedThread.DoTerminateSyncedProc;
begin
  FOnTerminate(Self);
end;

procedure TFixedThread.DoTerminate;
begin
  //outputdebugstring('fixedthread.doterminate');
  if Assigned(FOnTerminate) then Synchronize(DoTerminateSyncedProc);
end;

procedure TFixedThread.Synchronize(Method: TThreadMethod);
var ret: Cardinal;
begin
  //outputdebugstring('fixedthread.synchronize');
  FSyncExcept := nil;
  FSyncMeth   := Method;
  while 0 = SendMessageTimeout(FixedThreadWin, CM_EXECPROC, 0, Longint(Self), SMTO_NORMAL, 1000, ret) do
    if 0 <> GetLastError() then break // LastError=0 if timed out
    ;//else OutputDebugString('FixedThread: Failed to SendMessage, retrying');
  if Assigned(FSyncExcept) then raise FSyncExcept;
end;

initialization
finalization
  FreeFixedThreadWin;

{$ENDIF}
end.
