//PROFILE-NO
unit FEC;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

// this is basically a Delphi port of fec.c by onionnetworks
// note: only 8-bit fields are supported

// the file fec.c this unit is based on has the following disclaimer:

// fec.c -- forward error correction based on Vandermonde matrices
// 980624
// (C) 1997-98 Luigi Rizzo (luigi@iet.unipi.it)
//
// Portions derived from code by Phil Karn (karn@ka9q.ampr.org),
// Robert Morelos-Zaragoza (robert@spectra.eng.hawaii.edu) and Hari
// Thirumoorthy (harit@spectra.eng.hawaii.edu), Aug 1995
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above
//    copyright notice, this list of conditions and the following
//    disclaimer in the documentation and/or other materials
//    provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
// OF SUCH DAMAGE.


{$INCLUDE CompilerOpts.pas}

{-DEFINE FEC_DEBUG}

interface


function  FECCode_New(k,n: Integer): Pointer;
procedure FECCode_Free(code: Pointer);
procedure FECCode_Encode(code, SourceArray, DestArray: Pointer; RequestArray: Pointer; RequestCount,BlockSize: Integer);
function  FECCode_Decode(code, DataArray, IndexArray: Pointer; BlockSize: Integer): Boolean;

procedure self_test;

implementation

uses Windows,Classes,SysUtils {$IFDEF FEC_DEBUG} ,Dialogs,Controls {$ENDIF} ;

type
  gf = Byte;
  gf_pointer = ^gf;
  gf_pointer_pointer = ^gf_pointer;
  int_pointer = ^Integer;

  Tfec_parms = record
    magic: Cardinal;
    k,n:   Integer;
    enc_matrix: gf_pointer;
  end;
  Pfec_parms = ^Tfec_parms;

{$IFDEF FEC_DEBUG}
  procedure print_err(const s: String; NoNewLine: Boolean = False);
  begin
    if System.IsConsole then begin
      if NoNewLine then Write(s) else WriteLn(s);
    end else begin
      if s = '' then exit;
      if MessageDlg('FEC: ' + s, mtWarning, [mbOk,mbAbort], 0) = mrAbort then
        raise Exception.Create('Aborted');
    end;
  end;

  procedure pr_matrix(m1: Pointer; rows,cols: Integer; s: String);
  var
    r,c: Integer;
    m:   gf_pointer;
  begin
    m := m1;
    print_err(s);
    for r := 0 to rows-1 do begin
      for c := 0 to cols-1 do
        print_err(LowerCase(Format('%.2x ',[gf_pointer(PChar(m)+r*cols+c)^])), True);
      print_err('');
    end;
    print_err('');
  end;

  procedure pr_packets(pkt: gf_pointer_pointer; k, sz: Integer; s: String);
  var
    i,j: integer;
    p:   gf_pointer;
  begin
    print_err(s);
    for i := 0 to k-1 do begin
      p := gf_pointer_pointer(PChar(pkt) + i*SizeOf(gf_pointer))^;
      print_err(LowerCase(Format('%.3x: ',[i])), True);
      if p <> nil then begin
        for j := 0 to sz-1 do begin
          print_err(LowerCase(IntToHex(p^,2)), True);
          inc(p);
        end;
        print_err('');
      end else
        print_err('<NULL>');
    end;
    print_err('');
  end;

  procedure pr_index(index: int_pointer; k: Integer; s: String);
  var i: integer;
  begin
    print_err(s);
    for i := 0 to k-1 do begin
      print_err(LowerCase(Format('%.3x: %d',[i, int_pointer(PChar(index)+i*SizeOf(Integer))^])));
    end;
    print_err('');
  end;

{$ELSE}
  procedure print_err(const s: String; NoNewLine: Boolean = False); begin end;
{$ENDIF}

procedure TICK(var t: Int64);
begin t := GetTickCount; end;

procedure TOCK(var t: Int64);
var tmp: Int64;
begin
  tmp := GetTickCount;
  if tmp < t then inc(tmp, $100000000);
  t := 1000 * (tmp - t);
end;

const
  GF_BITS = 8;
  GF_SIZE = (1 shl GF_BITS) - 1;

const
  allPp: Array [0..16] of String = (
    '',                 //  0 no code
    '',                 //  1 no code
    '111',              //  2 1+x+x^2
    '1101',             //  3 1+x+x^3
    '11001',            //  4 1+x+x^4
    '101001',           //  5 1+x^2+x^5
    '1100001',          //  6 1+x+x^6
    '10010001',         //  7 1 + x^3 + x^7
    '101110001',        //  8 1+x^2+x^3+x^4+x^8
    '1000100001',       //  9 1+x^4+x^9
    '10010000001',      // 10 1+x^3+x^10
    '101000000001',     // 11 1+x^2+x^11
    '1100101000001',    // 12 1+x+x^4+x^6+x^12
    '11011000000001',   // 13 1+x+x^3+x^4+x^13
    '110000100010001',  // 14 1+x+x^6+x^10+x^14
    '1100000000000001', // 15 1+x+x^15
    '11010000000010001' // 16 1+x+x^3+x^12+x^16
  );

var
  gf_exp:  packed Array [0..2*GF_SIZE-1] of gf;
  gf_log:  packed Array [0..GF_SIZE+1-1] of Integer;
  inverse: packed Array [0..GF_SIZE+1-1] of gf;

function  modnn(x: Integer): gf;
begin
  while (x >= GF_SIZE) do begin
    dec(x, GF_SIZE);
    x := (x shr GF_BITS) + (x and GF_SIZE);
  end;
  Result := x;
end;

procedure SWAP(var a,b: Integer);    overload; var tmp: Integer;    begin tmp := a; a := b; b := tmp; end;
procedure SWAP(var a,b: gf);         overload; var tmp: gf;         begin tmp := a; a := b; b := tmp; end;
procedure SWAP(var a,b: gf_pointer); overload; var tmp: gf_pointer; begin tmp := a; a := b; b := tmp; end;

var gf_mul_table: Array [0..GF_SIZE+1-1,0..GF_SIZE+1-1] of gf;
function gf_mul(x,y: gf): gf; begin Result := gf_mul_table[x,y]; end;
type GF_MULC_VAR = ^gf;
procedure GF_MULC0(var gf_mulc: GF_MULC_VAR; c: gf); begin gf_mulc := @(gf_mul_table[c,0]); end;
procedure GF_ADDMULC(gf_mulc: GF_MULC_VAR; var dst: gf; x: gf); begin dst := dst xor GF_MULC_VAR(PChar(gf_mulc)+x)^; end;

procedure init_mul_table;
var i,j: gf;
begin
  for i := 0 to GF_SIZE+1-1 do
    for j := 0 to GF_SIZE+1-1 do
      gf_mul_table[i,j] := gf_exp[modnn(gf_log[i] + gf_log[j])];
  for j := 0 to GF_SIZE+1-1 do begin
    gf_mul_table[0,j] := 0;
    gf_mul_table[j,0] := 0;
  end;
end;

function NEW_GF_MATRIX(rows,cols: Integer): gf_pointer;
begin GetMem(Result, rows * cols * SizeOf(gf)); end;

procedure generate_gf;
var
  i:    Integer;
  mask: gf;
  Pp:   String;
begin
  Pp := allPp[GF_BITS];
  mask := 1;
  gf_exp[GF_BITS] := 0;
  for i := 0 to GF_BITS-1 do begin
    gf_exp[i] := mask;
    gf_log[gf_exp[i]] := i;
    if Pp[1+i] = '1' then gf_exp[GF_BITS] := gf_exp[GF_BITS] xor mask;
    mask := mask shl 1;
  end;
  gf_log[gf_exp[GF_BITS]] := GF_BITS;
  mask := 1 shl (GF_BITS-1);
  for i := GF_BITS+1 to GF_SIZE-1 do begin
    if gf_exp[i-1] >= mask then
      gf_exp[i] := gf_exp[GF_BITS] xor ((gf_exp[i-1] xor mask) shl 1)
    else
      gf_exp[i] := gf_exp[i-1] shl 1;
    gf_log[gf_exp[i]] := i;
  end;
  gf_log[0] := GF_SIZE;
  for i := 0 to GF_SIZE-1 do gf_exp[i+GF_SIZE] := gf_exp[i];
  inverse[0] := 0;
  inverse[1] := 1;
  for i := 2 to GF_SIZE do inverse[i] := gf_exp[GF_SIZE - gf_log[i]];
end;

// HIGHLY OPTIMIZE THIS and/or GF_ADDMULC!
procedure addmul(dst1,src1: gf_pointer; c: gf; sz: Integer);
const UNROLL = 16;
var
  gf_mulc: GF_MULC_VAR;
begin
  if c = 0 then exit;

  gf_mulc := @(gf_mul_table[c,0]);
  while sz > 15 do begin
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dec(sz,16);
  end;
  while sz > 0 do begin
    dst1^ := dst1^ xor gf_pointer(PChar(gf_mulc)+src1^)^; inc(dst1); inc(src1);
    dec(sz);
  end;

end;

procedure matmul(a,b,c: gf_pointer; n,k,m: Integer);
var
  row,col,i: Integer;
  pa,pb:     gf_pointer;
  acc:       gf;
begin
  for row := 0 to n-1 do begin
    for col := 0 to m-1 do begin
      pa := gf_pointer(PChar(a) + row * k);
      pb := gf_pointer(PChar(b) + col);
      acc := 0;
      for i := 0 to k-1 do begin
        acc := acc xor gf_mul(pa^,pb^);
        inc(pa); inc(pb,m);
      end;
      gf_pointer(PChar(c) + row * m + col)^ := acc;
    end;
  end;
end;

function elm_gf (const p: gf_pointer;  const idx: Integer): gf_pointer;
begin Result := gf_pointer( PChar(p) + idx*SizeOf(gf) ); end;
function elm_gfp(const p: gf_pointer_pointer;  const idx: Integer): gf_pointer_pointer;
begin Result := gf_pointer_pointer( PChar(p) + idx*SizeOf(gf_pointer) ); end;
function elm_int(const p: int_pointer; const idx: Integer): int_pointer;
begin Result := int_pointer( PChar(p) + idx*SizeOf(Integer) ); end;

function  invert_mat(src: gf_pointer; k: Integer): Integer;
// TODO: OPTIMIZE THAT!!
var
  c: gf;
  p,id_row,temp_row,pivot_row: gf_pointer;
  irow,icol,row,col,i,ix: Integer;
  indxc,indxr,ipiv: int_pointer;
begin
  Result := 1;
  indxc := nil; indxr := nil; ipiv := nil; id_row := nil; temp_row := nil;
  try
    try
      GetMem(indxc, k*SizeOf(Integer));
      GetMem(indxr, k*SizeOf(Integer));
      GetMem(ipiv,  k*SizeOf(Integer));
      id_row   := NEW_GF_MATRIX(1,k);
      temp_row := NEW_GF_MATRIX(1,k);
      FillChar(id_row^, k*SizeOf(gf), 0);

      for i := 0 to k-1 do
        elm_int(ipiv, i)^ := 0;

      for col := 0 to k-1 do begin
        irow := -1; icol := -1;
        if (elm_int(ipiv,col)^ <> -1) and (elm_gf(src,col*k + col)^ <> 0) then begin
          irow := col; icol := col;
        end else begin
          for row := 0 to k-1 do begin
            if elm_int(ipiv,row)^ <> -1 then begin
              for ix := 0 to k-1 do begin
                if elm_int(ipiv,ix)^ = 0 then begin
                  if elm_gf(src,row*k + ix)^ <> 0 then begin
                    irow := row; icol := col;
                    break;
                  end;
                end else if elm_int(ipiv,ix)^ > 1 then
                  raise EAbort.Create('singular matrix');
              end;
            if icol <> -1 then break;
            end;
          end;
        end;
        if icol = -1 then raise EAbort.Create('XXX pivot not found!');

        inc(elm_int(ipiv,icol)^);
        if irow <> icol then
          for ix := 0 to k-1 do
            SWAP( elm_gf(src,irow*k + ix)^, elm_gf(src,icol*k + ix)^ );
        elm_int(indxr,col)^ := irow;
        elm_int(indxc,col)^ := icol;
        pivot_row := elm_gf(src,icol*k);
        c := elm_gf(pivot_row,icol)^;
        if c = 0 then raise EAbort.Create('singular matrix 2');
        if c <> 1 then begin
          c := inverse[c];
          elm_gf(pivot_row,icol)^ := 1;
          for ix := 0 to k-1 do
            elm_gf(pivot_row,ix)^ := gf_mul(c, elm_gf(pivot_row,ix)^);
        end;
        elm_gf(id_row,icol)^ := 1;
        if not CompareMem(pivot_row, id_row, k*SizeOf(gf)) then begin
          p := src;
          for ix := 0 to k-1 do begin
            if ix <> icol then begin
              c := elm_gf(p,icol)^;
              elm_gf(p,icol)^ := 0;
              addmul(p, pivot_row, c, k);
            end;
            inc(p,k);
          end;
        end;
        elm_gf(id_row,icol)^ := 0;
      end;

      for col := k-1 downto 0 do begin
        if (elm_int(indxr,col)^ < 0) or (elm_int(indxr,col)^ >= k) then
          print_err('AARGH, indxr[col] ' + IntToStr(elm_int(indxr,col)^))
        else if (elm_int(indxc,col)^ < 0) or (elm_int(indxc,col)^ >= k) then
          print_err('AARGH, indxc[col] ' + IntToStr(elm_int(indxc,col)^))
        else if elm_int(indxr,col)^ <> elm_int(indxc,col)^ then
          for row := 0 to k-1 do
            SWAP( elm_gf(src,row*k + elm_int(indxr,col)^)^, elm_gf(src,row*k + elm_int(indxc,col)^)^ );
      end;

      Result := 0;

    except
      on E: EAbort do print_err(E.Message)
      else raise;
    end;
  finally
    if indxc    <> nil then FreeMem(indxc);
    if indxr    <> nil then FreeMem(indxr);
    if ipiv     <> nil then FreeMem(ipiv);
    if id_row   <> nil then FreeMem(id_row);
    if temp_row <> nil then FreeMem(temp_row);
  end;
end;

function  invert_vdm(src: gf_pointer; k: Integer): Integer;
var
  i,j,col,row: Integer;
  b,c,p: gf_pointer;
  t,xx,p_i: gf;
begin
  Result := 0;
  if k = 1 then exit;

  c := nil; b := nil; p := nil;
  try
    c := NEW_GF_MATRIX(1, k);
    b := NEW_GF_MATRIX(1, k);
    p := NEW_GF_MATRIX(1, k);
    j := 1;
    for i := 0 to k-1 do begin
      elm_gf(c,i)^ := 0;
      elm_gf(p,i)^ := elm_gf(src,j)^;
      inc(j,k);
    end;
    elm_gf(c,k-1)^ := elm_gf(p,0)^;
    for i := 1 to k-1 do begin
      p_i := elm_gf(p,i)^;
      for j := k-1 - (i-1) to k-2 do
        elm_gf(c,j)^ := elm_gf(c,j)^ xor gf_mul(p_i, elm_gf(c,j+1)^);
      elm_gf(c,k-1)^ := elm_gf(c,k-1)^ xor p_i;
    end;
    for row := 0 to k-1 do begin
      xx := elm_gf(p,row)^;
      t := 1;
      elm_gf(b,k-1)^ := 1;
      for i := k-2 downto 0 do begin
        elm_gf(b,i)^ := elm_gf(c,i+1)^ xor gf_mul(xx, elm_gf(b,i+1)^);
        t := gf_mul(xx,t) xor elm_gf(b,i)^;
      end;
      for col := 0 to k-1 do
        elm_gf(src,col*k + row)^ := gf_mul(inverse[t], elm_gf(b,col)^);
    end;
  finally
    if c <> nil then FreeMem(c);
    if b <> nil then FreeMem(b);
    if p <> nil then FreeMem(p);
  end;
end;

var fec_initialized: Boolean = False;

procedure init_fec;
begin
  generate_gf;
  init_mul_table;
  fec_initialized := True;
end;

const FEC_MAGIC = $FECC0DEC;

procedure fec_free(p: Pfec_parms);
begin
  if (p = nil)
  or (p^.magic <> ( ( (FEC_MAGIC xor Cardinal(p^.k)) xor Cardinal(p^.n)) xor Cardinal(p^.enc_matrix)) )
  then begin
    print_err('bad parameters to fec_free');
    exit;
  end;
  if p^.enc_matrix <> nil then FreeMem(p^.enc_matrix);
  FreeMem(p);
end;

function  fec_new(k,n: Integer): Pfec_parms;
var
  row,col: Integer;
  p,tmp_m: gf_pointer;
begin
  Result := nil;
  if not fec_initialized then init_fec;
  if (k > (GF_SIZE+1)) or (n > (GF_SIZE+1)) or (k > n) then begin
    print_err(Format('Invalid parameters k %d n %d GF_SIZE %d',[k,n,GF_SIZE]));
    exit;
  end;
  tmp_m := nil;
  try
    try
      GetMem(Result, SizeOf(Tfec_parms));
      FillChar(Result^, SizeOf(Tfec_parms), 0);
      Result^.k := k;
      Result^.n := n;
      Result^.enc_matrix := NEW_GF_MATRIX(n,k);
      Result^.magic := ( ( FEC_MAGIC xor Cardinal(k)) xor Cardinal(n)) xor Cardinal(Result^.enc_matrix);
      tmp_m := NEW_GF_MATRIX(n,k);

      elm_gf(tmp_m,0)^ := 1;
      for col := 1 to k-1 do elm_gf(tmp_m,col)^ := 0;
      p := tmp_m; inc(p,k);
      for row := 0 to n-2 do begin
        for col := 0 to k-1 do elm_gf(p,col)^ := gf_exp[modnn(row*col)];
        inc(p,k);
      end;

      invert_vdm(tmp_m, k);
      matmul(elm_gf(tmp_m,k*k), tmp_m, elm_gf(Result^.enc_matrix,k*k), n-k, k, k);
      FillChar(Result^.enc_matrix^, k*k*SizeOf(gf), 0);
      p := Result^.enc_matrix;
      for col := 0 to k-1 do begin
        p^ := 1;
        inc(p,k+1);
      end;

      {$IFDEF FEC_DEBUG}
        // pr_matrix(Result^.enc_matrix, n, k, 'encoding_matrix');
      {$ENDIF}
    except
      if Result <> nil then begin
        if Result^.enc_matrix <> nil then FreeMem(Result^.enc_matrix);
        FreeMem(Result);
      end;
      raise;
    end;
  finally
    if tmp_m <> nil then FreeMem(tmp_m);
  end;
end;

procedure fec_encode(code: Pfec_parms; src: gf_pointer_pointer; fec: gf_pointer; index,sz: Integer);
var
  i,k: Integer;
  p:   gf_pointer;
begin
  k := code^.k;
  if (GF_BITS > 8) then sz := sz div 2;
  if index < k then
    Move(elm_gfp(src,index)^^, fec^, sz*SizeOf(gf))
  else if index < code^.n then begin
    p := elm_gf(code^.enc_matrix,index*k);
    FillChar(fec^, sz*SizeOf(gf), 0);
    for i := 0 to k-1 do
      addmul(fec, elm_gfp(src,i)^, elm_gf(p,i)^, sz);
  end else
    print_err(Format('Invalid index %d (max %d)',[index,code^.n-1]));
end;

function  shuffle(pkt: gf_pointer_pointer; index: int_pointer; k: Integer): Integer;
var
  i,c: Integer;
begin
  i := 0;
  while i < k do begin
    if (elm_int(index,i)^ >= k) or (elm_int(index,i)^ = i) then
      inc(i)
    else begin
      c := elm_int(index,i)^;
      if elm_int(index,c)^ = c then begin
        Result := 1; exit;
      end;
      SWAP(elm_int(index,i)^, elm_int(index,c)^);
      SWAP(elm_gfp(pkt,  i)^, elm_gfp(pkt,  c)^);
    end;
  end;
  Result := 0;
end;

function  build_decode_matrix(code: Pfec_parms; pkt: gf_pointer_pointer; index: int_pointer): gf_pointer;
var
  i,k: Integer;
  p,matrix: gf_pointer;
begin
  k := code^.k;
  matrix := NEW_GF_MATRIX(k,k);
  p := matrix;
  for i := 0 to k-1 do begin
    if elm_int(index,i)^ < k then begin
      FillChar(p^, k*SizeOf(gf), 0);
      elm_gf(p,i)^ := 1;
    end else if elm_int(index,i)^ < code^.n then
      Move(elm_gf(code^.enc_matrix,elm_int(index,i)^*k)^, p^, k*SizeOf(gf))
    else begin
      print_err(Format('decode: invalid index %d (max %d)',[elm_int(index,i)^,code^.n-1]));
      FreeMem(matrix);
      Result := nil; exit;
    end;
    inc(p,k);
  end;
  if invert_mat(matrix,k) <> 0 then begin
    FreeMem(matrix); matrix := nil;
  end;
  Result := matrix;
end;

function  fec_decode(code: Pfec_parms; pkt: gf_pointer_pointer; index: int_pointer; sz: Integer): Integer;
var
  m_dec:   gf_pointer;
  new_pkt: gf_pointer_pointer;
  row,col,k: Integer;
begin
  k := code^.k;
  if GF_BITS > 8 then sz := sz div 2;

  Result := 1; m_dec := nil; new_pkt := nil;
  try
    if shuffle(pkt, index, k) <> 0 then exit;
    m_dec := build_decode_matrix(code, pkt, index);
    if m_dec = nil then exit;

    GetMem(new_pkt, k*SizeOf(gf_pointer));
    FillChar(new_pkt^, k*SizeOf(gf_pointer), 0);
    try
      for row := 0 to k-1 do begin
        if elm_int(index,row)^ >= k then begin
          GetMem(elm_gfp(new_pkt,row)^, sz*SizeOf(gf));
          FillChar(elm_gfp(new_pkt,row)^^, sz*SizeOf(gf), 0);
          for col := 0 to k-1 do
            addmul(elm_gfp(new_pkt,row)^, elm_gfp(pkt,col)^, elm_gf(m_dec,row*k + col)^, sz);
        end;
      end;

      for row := 0 to k-1 do begin
        if elm_int(index,row)^ >= k then begin
          Move(elm_gfp(new_pkt,row)^^, elm_gfp(pkt,row)^^, sz*SizeOf(gf));
          FreeMem(elm_gfp(new_pkt,row)^); elm_gfp(new_pkt,row)^ := nil;
          elm_int(index,row)^ := row;
        end;
      end;

      Result := 0;
    except
      for row := 0 to k-1 do
        if elm_gfp(new_pkt,row)^ <> nil then FreeMem(elm_gfp(new_pkt,row)^);
      raise;
    end;
  finally
    if m_dec <> nil then FreeMem(m_dec);
    if new_pkt <> nil then FreeMem(new_pkt);
  end;
end;

function  test_decode(code: Pointer; k: Integer; index: int_pointer; sz: Integer; s: String): Integer;
const
  prev_k:  Integer = 0;
  prev_sz: Integer = 0;
  d_original: gf_pointer_pointer = nil;
  d_src:      gf_pointer_pointer = nil;
var
  errors,reconstruct,item,i: Integer;
  ticks: Array [1..2] of Int64;
  bCleanup: Boolean;
begin
  bCleanup := (code = nil) and (k = 0) and (index = nil) and (sz = 0);
  if not bCleanup then begin
    if (sz < 1) or (sz > 8192) then begin
      print_err(Format('test_decode: size %d invalid, must be 1..8K',[sz]));
      Result := 1; exit;
    end;
    if (k < 1) or (k > (GF_SIZE+1)) then begin
      print_err(Format('test_decode: k %d invalid, must be 1..%d',[k,GF_SIZE+1]));
      Result := 2; exit;
    end;
  end;
  if (prev_k <> k) or (prev_sz <> sz) then begin
    if d_original <> nil then begin
      for i := 0 to prev_k-1 do begin
        if elm_gfp(d_original,i)^ <> nil then FreeMem(elm_gfp(d_original,i)^);
        if elm_gfp(d_src,     i)^ <> nil then FreeMem(elm_gfp(d_src,     i)^);
      end;
      FreeMem(d_original); d_original := nil;
      FreeMem(d_src);      d_src      := nil;
    end;
  end;
  prev_k := k; prev_sz := sz;

  if bCleanup then begin Result := 0; exit; end;

  if d_original = nil then begin
    GetMem(d_original, k*SizeOf(Pointer)); FillChar(d_original^, k*SizeOf(Pointer), 0);
    GetMem(d_src,      k*SizeOf(Pointer)); FillChar(d_src^,      k*SizeOf(Pointer), 0);
    for i := 0 to k-1 do begin
      GetMem(elm_gfp(d_original,i)^, sz);
      GetMem(elm_gfp(d_src,     i)^, sz);
    end;
    for i := 0 to k-1 do
      for item := 0 to sz-1 do
        elm_gf(elm_gfp(d_original,i)^,item)^ := ((item xor i) + 3) and GF_SIZE;
  end;

  errors := 0;
  reconstruct := 0;
  for i := 0 to k-1 do
    if elm_int(index,i)^ >= k then inc(reconstruct);

  TICK(ticks[2]);
  for i := 0 to k-1 do
    fec_encode(code, d_original, elm_gfp(d_src,i)^, elm_int(index,i)^, sz);
  TOCK(ticks[2]);

  TICK(ticks[1]);
  if fec_decode(code, d_src, index, sz) <> 0 then begin
    print_err('detected singular matrix for ' + s + '     ');
    Result := 1; exit;
  end;
  TOCK(ticks[1]);
//  pr_packets(d_original, k, sz, 'd_original');
//  pr_packets(d_src, k, sz, 'd_src');
//  pr_index(index, k, 'index');

  for i := 0 to k-1 do
    if not CompareMem(elm_gfp(d_original,i)^, elm_gfp(d_src,i)^, sz) then begin
      inc(errors);
      print_err('error reconstructing block ' + IntToStr(i));
    end;
  if errors <> 0 then
    print_err(Format('Errors reconstructing %d blocks out of %d',[errors,k]));

  if ticks[1] = 0 then ticks[1] := 1;
  if ticks[2] = 0 then ticks[2] := 1;
  print_err(Format('  k %3d, l %3d  c_enc %10.6f MB/s c_dec %10.6f MB/s          ',[
                   k,reconstruct,
                   (k * sz * reconstruct) / ticks[2],
                   (k * sz * reconstruct) / ticks[1]
                   ])
            + #13, True
           );
  Result := errors;
end;

procedure test_gf;
var i: Integer;
begin
  if not fec_initialized then init_fec;
  for i := 0 to GF_SIZE-1 do begin
    if gf_exp[gf_log[i]] <> i then
      print_err(Format('bad exp/log i %d log %d exp(log) %d',[i,gf_log[i],gf_exp[gf_log[i]]]));
    if (i <> 0) and (gf_mul(i, inverse[i]) <> 1) then
      print_err(Format('bad mul/inv i %d inv %d i*inv(i) %d',[i,inverse[i],gf_mul(i,inverse[i])]));
    if gf_mul(0,i) <> 0 then
      print_err(Format('bad mul table 0,%d',[i]));
    if gf_mul(i,0) <> 0 then
      print_err(Format('bad mul table %d,0',[i]));
  end;
end;

procedure self_test;
const
  KK_MAX = 64;
  SZ     = 1024;
var
  buf:  String;
  code: Pointer;
  kk,i,j: Integer;
  ixs:  int_pointer;
  lim,max_i0:  Integer;
begin
  lim := GF_SIZE + 1;
  if lim > 1024 then lim := 1024;

if FALSE then test_gf;

  kk := KK_MAX;
  while kk > 2 do begin
    code := fec_new(kk,lim);
    GetMem(ixs, kk*SizeOf(Integer));

    for i := 0 to kk-1 do elm_int(ixs,i)^ := kk-i;
    buf := Format('kk=%d, kk - i',[kk]);
    test_decode(code, kk, ixs, SZ, buf);

    for i := 0 to kk-1 do elm_int(ixs,i)^ := i;
    test_decode(code, kk, ixs, SZ, 'i');

if FALSE then begin
    for i := 0 to kk-1 do elm_int(ixs,i)^ := i;
    elm_int(ixs,0)^ := elm_int(ixs,kk div 2)^;
    test_decode(code, kk, ixs, SZ, '0 = 1 (error expected)');
end;

if FALSE then begin
    for i := lim-1 downto kk do begin
      for j := 0 to KK_MAX-1 do elm_int(ixs,j)^ := kk-j;
      elm_int(ixs,0)^ := i;
      test_decode(code, kk, ixs, SZ, '0 = big');
    end;
end;

if FALSE then begin
    i := lim - kk;
    while (i >= 0) and (i >= (lim-kk-4)) do begin
      for j := 0 to kk-1 do elm_int(ixs,j)^ := kk-1-j+i;
      test_decode(code, kk, ixs, SZ, 'shifted j');
      dec(i);
    end;
end;

if TRUE then begin
    max_i0 := KK_MAX div 2;
    if (max_i0 + KK_MAX) > lim then max_i0 := lim - KK_MAX;
    for i := 0 to max_i0 do begin
      for j := 0 to kk-1 do elm_int(ixs,j)^ := j+i;
      test_decode(code, kk, ixs, SZ, 'shifted j');
    end;
end;

    print_err('');
    test_decode(nil,0,nil,0,''); // cleanup
    FreeMem(ixs);
    fec_free(code);
    dec(kk);
  end;
end;

// utility funcs
function  FECCode_New(k,n: Integer): Pointer;
begin
  Result := fec_new(k,n);
end;

procedure FECCode_Free(code: Pointer);
begin
  if code <> nil then fec_free(code);
end;

procedure FECCode_Encode(code, SourceArray, DestArray, RequestArray: Pointer; RequestCount,BlockSize: Integer);
// SourceArray:  Pointer to an array of k sourceblock pointers
// DestArray:    Pointer to an array of RequestCount allocated destinationblock pointers
// RequestArray: Pointer to an array of RequestCount indices (0..n-1); the requested blocks
// RequestCount: Number of requested blocks
// BlockSize:    Size of each block
var
  i: Integer;
  pSrc,pDst: gf_pointer_pointer;
  pIdx:      int_pointer;
begin
  pSrc := SourceArray;
  pDst := DestArray; pIdx := RequestArray;
  for i := 0 to RequestCount-1 do begin
    fec_encode(code, pSrc, pDst^, pIdx^, BlockSize);
    inc(pDst); inc(pIdx);
  end;
end;

function  FECCode_Decode(code, DataArray, IndexArray: Pointer; BlockSize: Integer): Boolean;
// SourceArray:  Pointer to an array of k sourceblock pointers (the available data and checkblocks)
//               During decoding the checkblocks are overwritten by decoded datablocks.
//               After decoding it contains the requested datablocks in correct order
// IndexArray:   Pointer to an array of k indices of the blocks in SourceArray (modified)
// BlockSize:    Size of each block
//
// Note: The contents of IndexArray are modified!
begin
  Result := (0 = fec_decode(code, DataArray, IndexArray, BlockSize));
end;

initialization

Assert(GF_BITS = 8, 'Only GF_BITS = 8 supported');

finalization

end.
