//PROFILE-NO
unit CryptUtils;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

interface

uses Windows, SysUtils;

const
  CIPHERNAME_TWOFISH  = 'Twofish';
  CIPHERNAME_RIJNDAEL = 'Rijndael';

type
  ECryptUtilsError = class (Exception);

  TCipher = class
  private
  protected
    function    GetBlockSize:  Integer; virtual; abstract;
    function    GetKeySize:    Integer; virtual; abstract;
    function    GetCipherName: String;  virtual; abstract;
  public
    procedure   Initialize(pKey: PByte; KeyLen: Integer); virtual; abstract;
    procedure   Encipher(pIn,pOut: PByte); virtual; abstract;
    procedure   Decipher(pIn,pOut: PByte); virtual; abstract;
    property    BlockSize:  Integer read GetBlockSize; // in BITs !
    property    KeySize:    Integer read GetKeySize;   // in BITs !
    property    CipherName: String  read GetCipherName;
  end;

  TPeriodicCipherFeedBackMode = class
  private
    FCipher:         TCipher;
    FpFeedbackReg:   PByte;
    FFeedbackRegLen: Integer;
    FFeedbackRegPos: Integer;
    procedure RefillFeedbackReg;
  public
    constructor Create(Cipher: TCipher; pInitialIV: PByte = nil);
    destructor  Destroy; override;
    procedure   Reset(pIV: PByte);
    function    Encipher(b: Byte): Byte;
    function    Decipher(b: Byte): Byte;
  end;

function  Freenet_log2(n: Cardinal): Integer;
function  CreateCipherByName(CipherName: String): TCipher;

implementation

uses Twofish,Rijndael;

function  Freenet_log2(n: Cardinal): Integer; // rounded UP!
begin
  Result := 0;
  while (Result < 63) and ((1 shl Result) < n) do inc(Result);
end;

function  CreateCipherByName(CipherName: String): TCipher;
begin
  if CompareText(CipherName, CIPHERNAME_TWOFISH) = 0 then
    Result := TTwofish.Create
  else if CompareText(CipherName, CIPHERNAME_RIJNDAEL) = 0 then
    Result := TRijndael.Create
  else
    raise ECryptUtilsError.Create('Unknown cipher');
end;

{ TPeriodicCipherFeedBackMode }

constructor TPeriodicCipherFeedBackMode.Create(Cipher: TCipher; pInitialIV: PByte);
begin
  inherited Create;
  FCipher := Cipher;
  FFeedbackRegLen := (Cipher.BlockSize shr 3);
  GetMem(FpFeedbackReg, FFeedbackRegLen);
  FillChar(FpFeedbackReg^, FFeedbackRegLen, 0);
  FFeedbackRegPos := FFeedbackRegLen;
  if pInitialIV <> nil then Reset(pInitialIV);
end;

destructor TPeriodicCipherFeedBackMode.Destroy;
begin
  if FpFeedbackReg <> nil then FreeMem(FpFeedbackReg);
  inherited;
end;

procedure TPeriodicCipherFeedBackMode.RefillFeedbackReg;
begin
  FCipher.Encipher(FpFeedbackReg,FpFeedbackReg);
  FFeedbackRegPos := 0;
end;

procedure TPeriodicCipherFeedBackMode.Reset(pIV: PByte);
begin
  Move(pIV^, FpFeedbackReg^, FFeedbackRegLen);
  FFeedbackRegPos := FFeedbackRegLen;
end;

function TPeriodicCipherFeedBackMode.Encipher(b: Byte): Byte;
var p: PByte;
begin
  if FFeedbackRegPos >= FFeedbackRegLen then RefillFeedbackReg;
  p := PByte(PChar(FpFeedbackReg) + FFeedbackRegPos);
  p^ := p^ xor b;
  Result := p^;
  inc(FFeedbackRegPos);
end;

function TPeriodicCipherFeedBackMode.Decipher(b: Byte): Byte;
var p: PByte;
begin
  if FFeedbackRegPos >= FFeedbackRegLen then RefillFeedbackReg;
  p := PByte(PChar(FpFeedbackReg) + FFeedbackRegPos);
  Result := p^ xor b;
  p^ := b;
  inc(FFeedbackRegPos);
end;

end.
