//PROFILE-NO
unit Base64;

// *****************************************************************************
// * Copyright 2003-2005 mxbee                                                 *
// *****************************************************************************
// * This program is free software; you can redistribute it and/or modify      *
// * it under the terms of the GNU General Public License as published by      *
// * the Free Software Foundation; either version 2 of the License, or         *
// * (at your option) any later version.                                       *
// *                                                                           *
// * This program is distributed in the hope that it will be useful,           *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
// * GNU General Public License for more details.                              *
// *                                                                           *
// * You should have received a copy of the GNU General Public License         *
// * along with this program; if not, write to the Free Software               *
// * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
// *****************************************************************************

{$INCLUDE CompilerOpts.pas}

// this implements Base64 like Freenet does, i.e. not quite standard:
// -slightly different character set
// -padding with '_' is optional

interface

uses Windows,SysUtils;

type
  EBase64Exception = class(Exception);

function  Base64encode(pData: PChar; DataLen: Integer; DoPadding: Boolean = False): String;
procedure Base64decode(sInput: String; out pData: PByte; out DataLen: Integer);
function  Base64decodeAsString(sInput: String): String;

implementation

const
  Base64Alphabet: Array [0..63] of Char = (
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', '0', '1', '2', '3',
    '4', '5', '6', '7', '8', '9', '~', '-'
  );

var
  Base64RevAlphabet: Array [Char] of Byte;

function  Base64encode(pData: PChar; DataLen: Integer; DoPadding: Boolean): String;
var
  Len: Integer;
  val: Cardinal;
  p:   PChar;
begin
  Result := '';
  Len := DataLen; p := pData;
  while Len > 0 do begin
    val := Ord(p^) shl 16; dec(Len); inc(p);
    if Len > 0 then begin val := val or (Ord(p^) shl 8); dec(Len); inc(p); end;
    if Len > 0 then begin val := val or Ord(p^); dec(Len); inc(p); end;
    Result := Result
            + Base64Alphabet[(val shr 18) and $3F]
            + Base64Alphabet[(val shr 12) and $3F]
            + Base64Alphabet[(val shr  6) and $3F]
            + Base64Alphabet[ val         and $3F];
  end;
  if ((DataLen mod 3) <> 0) then begin
    if DoPadding then begin
      if (DataLen mod 3) = 1 then Result[Length(Result)-1] := '_';
      Result[Length(Result)] := '_';
    end else begin
      if (DataLen mod 3) = 1 then Delete(Result,Length(Result),1);
      Delete(Result,Length(Result),1);
    end;
  end;
end;

procedure Base64decode(sInput: String; out pData: PByte; out DataLen: Integer);
// caller is responsible for freeing pData later!
var
  inLen,outLen,wholeInLen,wholeOutLen,blocks,remainder: Integer;
  i,OrValue,OutVal: Integer;
  in1,in2,in3,in4: Byte;
  p: PByte;
begin
  pData := nil; DataLen := 0;
  try
    while (Length(sInput) > 0) and (sInput[Length(sInput)] = '_') do Delete(sInput,1,1);
    inLen := Length(sInput);
    blocks := inLen div 4;
    remainder := inLen and 3;
    wholeInLen := blocks * 4;
    wholeOutLen := blocks * 3;
    case remainder of
      0: outLen := wholeOutLen;
      2: outLen := wholeOutLen + 1;
      3: outLen := wholeOutLen + 2;
      else raise EBase64Exception.Create('Invalid Base64 length');
    end;
    GetMem(pData, outLen); DataLen := outLen;
    p := pData; i := 0;
    while i < wholeInLen do begin
      in1 := Base64RevAlphabet[sInput[1+i  ]];
      in2 := Base64RevAlphabet[sInput[1+i+1]];
      in3 := Base64RevAlphabet[sInput[1+i+2]];
      in4 := Base64RevAlphabet[sInput[1+i+3]];
      OrValue := in1 or in2 or in3 or in4;
      if (OrValue and $80) <> 0 then raise EBase64Exception.Create('Invalid Base64 character');
      OutVal := (in1 shl 18) or (in2 shl 12) or (in3 shl 6) or in4;
      p^ := (OutVal shr 16) and $FF; inc(p);
      p^ := (OutVal shr  8) and $FF; inc(p);
      p^ :=  OutVal         and $FF; inc(p);
      inc(i,4);
    end;
    case remainder of
      2: begin
           in1 := Base64RevAlphabet[sInput[1+i  ]];
           in2 := Base64RevAlphabet[sInput[1+i+1]];
           OrValue := in1 or in2;
           OutVal := (in1 shl 18) or (in2 shl 12);
           p^ := (OutVal shr 16) and $FF; //inc(p);
         end;
      3: begin
           in1 := Base64RevAlphabet[sInput[1+i  ]];
           in2 := Base64RevAlphabet[sInput[1+i+1]];
           in3 := Base64RevAlphabet[sInput[1+i+2]];
           OrValue := in1 or in2 or in3;
           OutVal := (in1 shl 18) or (in2 shl 12) or (in3 shl 6);
           p^ := (OutVal shr 16) and $FF; inc(p);
           p^ := (OutVal shr  8) and $FF; //inc(p);
         end;
      else OrValue := 0;
    end;
    if (OrValue and $80) <> 0 then raise EBase64Exception.Create('Invalid Base64 character');
  except
    if pData <> nil then FreeMem(pData);
    raise;
  end;
end;

function  Base64decodeAsString(sInput: String): String;
// Utility func, returns hex representation of decoded value
var
  pB,p: PByte;
  Len:  Integer;
begin
  Result := '';
  Base64decode(sInput, pB, Len);
  try
    p := pB;
    while Len > 0 do begin
      Result := Result + LowerCase(IntToHex(p^,2));
      inc(p); dec(Len);
    end;
  finally
    FreeMem(pB);
  end;
end;

procedure InitBase64RevAlphabet;
var
  i: Integer;
  c: Char;
begin
  for c := Low(Base64RevAlphabet) to High(Base64RevAlphabet) do
    Base64RevAlphabet[c] := $FF;
  for i := Low(Base64Alphabet) to High(Base64Alphabet) do
    Base64RevAlphabet[Base64Alphabet[i]] := i;
end;

initialization
  InitBase64RevAlphabet;

finalization

end.
